/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.util.time.DateTimeException;
import net.xfra.qizxopen.util.FileUtil;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dm.FONIDataModel;
import net.xfra.qizxopen.xquery.dm.LibraryLink;
import net.xfra.qizxopen.xquery.dt.*;
import net.xfra.qizxopen.xquery.op.Expression;

import java.io.PrintWriter;
import java.io.Writer;
import java.io.Reader;
import java.io.File;
import java.io.FileReader;
import java.io.StringReader;
import java.io.IOException;
import java.util.Vector;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Properties;
import java.util.Enumeration;
import java.text.Collator;
// XSLT:
import javax.xml.transform.*;
import javax.xml.transform.sax.*;
import javax.xml.transform.stream.*;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 *	Main interface to XML Query services.
 *	
 * <p>XQueryProcessor provides a static environment to compile a query from
 * some text source, and a dynamic environment (in particular a Document Manager)
 * to execute this query.
 * <p>The standard way for using it is as follows:<ul>
 * <li>Instantiate a XQueryProcessor using one of the constructors.
 * <li>compile a query using one form of <code>compileQuery</code>.
 * A message {@link Log} has to be instantiated and passed to these methods.
 * This yields a compiled <code>Query</code> object.
 * <li>Execute the query one or several times using one of the
 * <code>executeQuery</code> methods.
 *</ul>
 * <p>Miscellaneous optional informations can be set through specialized methods
 * for use in the static or dynamic query contexts:
 * global variable values, default collation, implicit timezone, document input, 
 * base URI, default serialization output.
 * <p>XQueryProcessors can share a common {@link ModuleManager} or
 * {@link DocumentManager}. It avoids redundant compilation or document loading in
 * an environment where many processors use the same queries or documents. The way
 * to achieve this is to create a "master" processor with its own ModuleManager and
 * DocumentManager and to use it in the proper constructor for each needed instance.
 */
public class XQueryProcessor
{
    // a shared predefined module with private extensions:
    public static String    EXTENSIONS_URI = "net.xfra.qizxopen.ext.Xfn";
    public static Namespace EXTENSIONS_NS = Namespace.get(EXTENSIONS_URI);
    public static final String XSLT_OUTPUT_FILE = "output-file";

    // **** to define *after* EXTENSIONS_URI!
    static PredefinedModule OwnPredefined = newOwnPredefinedModule();

    protected DocumentManager docMan;
    protected ModuleManager   moduleMan;
    protected String baseURI = ".";
    	// function input():
    protected String docInputURI;
    protected Value input;

    protected PredefinedModule predefined = OwnPredefined;
    protected HashMap globals = new HashMap();	// specified values for globals
    protected HashMap collations;
    protected HashMap properties = new HashMap();
    protected String defaultCollation = null;
    protected String implicitTimezone = null;
    protected PrintWriter defaultOutput = new PrintWriter(System.out, true);
    protected Log log;
    protected NSPrefixMapping extraNS;
    // current execution contexte (used for stop and pause)
    protected DefaultEvalContext execContext;

    /**
     *	Creation without Module Manager and Document Manager.
     *	<p>These objects must be specified before execution.
     */
    public XQueryProcessor() {
	init();
    }
    
    /**
     *	Simple creation with private Module Manager and Document Manager.
     *	@param baseURI default base URI (also used for Document Manager)
     *	@param moduleBaseURI base URI used to resolve module locations
     *	@throws IOException thrown by ModuleManager or DocumentManager
     *	constructors.
     */
    public XQueryProcessor(String moduleBaseURI, String baseURI)
	throws IOException { 
        setDocumentManager( new DocumentManager(baseURI) );
        setModuleManager( new ModuleManager(moduleBaseURI) );
	init();
    }
    
    /**
     *	Creates a new XQueryProcessor from a "master" processor, inheriting and
     *	sharing the document manager, module manager, predefined functions and
     *	global variables.
     *	@param master a XQueryProcessor used as template.
     */
    public XQueryProcessor( XQueryProcessor master ) {
        docMan = master.getDocumentManager();
        moduleMan = master.getModuleManager();
        predefined = master.predefined;
        collations = master.collations;
        defaultCollation = master.defaultCollation;
        implicitTimezone = master.implicitTimezone;
	properties = (HashMap) master.properties.clone();
	extraNS = (master.extraNS == null)? null : master.extraNS.copy();
	setSysProperty(":processor", this);
    }

    private void init() {
	// qizx: and x: are prefix aliases for extension functions
	predefineNamespace("qizx", EXTENSIONS_NS.getURI() );
	predefineNamespace("x", EXTENSIONS_NS.getURI() );
	// SQL/JDBC extension: regular Java binding, but predefined
	predefineNamespace("sqlx", "java:net.xfra.qizxopen.ext.SqlConnection" );
	predefineNamespace("sql",  "java:java.sql" );	// generic
	predefineNamespace("sqlc", "java:java.sql.Connection" ); // helper
	predefineNamespace("sqlr", "java:java.sql.ResultSet" ); // helper
	predefineNamespace("sqlp", "java:java.sql.PreparedStatement"); //helper

	// system properties:
	setSysProperty(":processor", this);
	setSysProperty("version", "1.0");	// of XQuery
	setSysProperty("vendor", "Xavier Franc");
	setSysProperty("vendor-url", "http://www.xfra.net/qizxopen/");
	setSysProperty("product-name", "Qizx/open");
	setSysProperty("product-version", Version.get());
    }

    /**
     *	Gets the current version of the XML Query engine.
     */
    public String getVersion() {
        return Version.get();
    }
    
    /**
    *	Sets up the Module Manager.
    */
    public void setModuleManager( ModuleManager moduleManager ) {
        moduleMan = moduleManager;
    }
    /**
    *	Returns the current Module Manager.
    */
    public ModuleManager getModuleManager( ) {
        return moduleMan;
    }
    
    /**
    *	Defines the Document Manager.
    */
    public void setDocumentManager( DocumentManager documentManager ) {
        docMan = documentManager;
    }
    /**
    *	Returns the current Document Manager.
    */
    public DocumentManager getDocumentManager() {
        return docMan;
    }
    
    /**
     *	Defines the input() sequence by a XML fragment.
     *	@param xmlFragment a fragment of XML to be parsed and used as
     *	implicit input. 
     */
    public void setInput( String xmlFragment )
	throws XQueryException {
	setInput( new InputSource(new StringReader(xmlFragment)) );
    }
    
    /**
     *	Defines the input() sequence by a XML fragment parsed from
     *	a SAX InputSource.
     *	@param xmlSource a SAX InputSource reffering to a fragment of XML
     *	to be parsed and used as implicit input. 
     */
    public void setInput( InputSource xmlSource )
	throws XQueryException {
        if( docMan == null )
            throw new EvalException("no Document Manager defined");
	docInputURI = null; // undefine input URI
	try {
	    FONIDocument doc = docMan.parseDocument( xmlSource );
	    FONIDataModel dm = new FONIDataModel(doc);
	    input = new SingleNode( dm.getDocumentNode() );
	}
        catch (SAXException e) {
            throw new XQueryException("XML parsing error: "+e.getMessage(), e);
        }
        catch (IOException ex) {	//
            throw new XQueryException("XML input error: "+ex.getMessage(), ex);
        }
    }
    
    /**
     *	Defines the input() sequence by a document URI.
     *	@param docURI uri of a document to be opened or parsed and used as
     *	implicit input. This URI can be relative to the document base URI (if
     *	the default Document Manager is used).
     */
    public void setDocumentInput( String docURI ) throws XQueryException {
        if( docMan == null )
            throw new EvalException("no Document Manager defined");
	docInputURI = docURI;
    }
    
    /**
    *	Defines the input() sequence by a collection URI.
    *	Implemented only in XQuest.
    */
    public void setCollectionInput( String uri ) throws XQueryException {
	throw new XQueryException("setCollectionInput not implemented");
    }
    
    /**
    *	Defines the default output channel for serialization.
    */
    public void setDefaultOutput( PrintWriter output ) {
        this.defaultOutput = output;
    }
    
    /**
    *	Defines a runtime log.
    */
    public void setLog(Log log) {
        this.log = log;
    }

    /**
     *	Defines a property, a named object that can be retrieved by the
     *	extension function x:system-property(name) or by any application.
     *	@param name of a property
     *	@param property value of the property to store.
     */
    public void setSysProperty(String name, Object property) {
	properties.put(name, property);
    }

    /**
     *	Retrieves a previously defined system property.
     *	@param name of a property
     *	@return the stored property, or null if not found.
     */
    public Object getSysProperty(String name) {
	return properties.get(name);
    }

    /**
     *	Clears all predefined variables added, and initial values for globals.
     */
    public void resetDeclarations() {
	predefined = OwnPredefined;
	globals = new HashMap();
    }

    /**
     * Defines a global variable in the predefined static context.
     * The initial value must then be set by a variant of
     * initGlobal() suitable to the type.
     * @param varName local name of the variable.
     * @param type assigned to the variable.
     */
    public void predefineGlobal( QName varName, Type type ) {
        if(predefined == OwnPredefined)
            predefined = newOwnPredefinedModule();	// ie clone
        predefined.defineGlobal(varName, type);
    }

    private static PredefinedModule newOwnPredefinedModule() {
	PredefinedModule pdm = new PredefinedModule();
	pdm.registerFunctionPlugger( new PredefinedModule.BasicFunctionPlugger(
					 EXTENSIONS_NS, EXTENSIONS_URI+ "%C"));
	return pdm;
    }
    
    /**
    * Defines a global variable in the predefined static context.
    * The initial value must then be set by a variant of initGlobal()
    * suitable for the type specified.
    *	@param varName local name of the variable.
    *	@param type assigned to the variable.
    */
    public void predefineGlobal( String varName, Type type ) {
        predefineGlobal(QName.get(varName), type);
    }
    
    /**
     *	Sets an initial value for a global variable.
     *	<p>This is not attached to a particular
     *	query, so it does not raise an error if the variable doesn't exist.
     * @param varName qualified name of the variable (can be in the namespace
     *	of a module or 'local').
     * @param value provided initial value: must be compatible with the declared
     *	type (not checked immediately). If the value is a string, an attempt 
     *	to cast to the proper type will be made.
     */
    public void initGlobal( QName varName, Value value ) {
        globals.put(varName, value);
	// compat hack: if 'local' namespace, store value also into variable
	// with same local name and no namespace.
	if(varName.getNamespace() == Module.LOCAL_NS)
	    globals.put(QName.get(varName.getLocalName()), value);
    }
    
    /**
     *	Utility for use with predefineGlobal and initGlobal: 
     *	convert a NCName to a QName in 'local' namespace.
    */
    public static QName toLocalNS( String name ) {
        return QName.get(Module.LOCAL_NS, name);
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, boolean value ) {
        initGlobal(varName, new SingleBoolean(value));
    }
    
    /**
     *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
     */
    public void initGlobal( QName varName, long value ) {
        initGlobal(varName, new SingleInteger(value));
    }
    
    /**
     *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
     */
    public void initGlobal( QName varName, double value ) {
        initGlobal(varName, new SingleDouble(value));
    }
    
    /**
     *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
     */
    public void initGlobal( QName varName, String value ) {
        initGlobal(varName, new SingleString(value));
    }
    
    /**
    *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
    */
    public void initGlobal( QName varName, String[] value ) {
        SingleString[] v = new SingleString[value.length];
        for(int i = 0; i < value.length; i++)
            v[i] = new SingleString(value[i]);
        initGlobal( varName, new ArraySequence(v, v.length));
    }
    
    /**
     *	Convenience method. See {@link #initGlobal(QName varName, Value value)}.
     */
    public void initGlobal( QName varName, Object value ) {
        initGlobal( varName, new SingleWrappedObject(value));
    }
    
    /**
    * Defines a namespace mapping, visible by queries compiled with
    * this processor.
    */
    public void predefineNamespace( String prefix, String uri) {
        if(extraNS == null)
            extraNS = new NSPrefixMapping();
        extraNS.addMapping(prefix, uri);
    }
    
    /**
    *	Registers a custom collation for use in the processed queries.
    */
    public void registerCollation(String uri, Collator collator) {
        if(collations == null)
            collations = new HashMap();
        collations.put(uri, collator);
    }
    
    /**
    *	Defines the URI of the default collation.
    */
    public void setDefaultCollation(String uri) {
        defaultCollation = uri;
    }
    
    /**
    *	Defines the implicit timezone in xs:duration format.
    *	@param duration for example "PT4H30M" or "-PT5H".
    *  If not specified, the implicit timezone is taken from the system default.
    */
    public void setImplicitTimezone(String duration) {
        implicitTimezone = duration;
    }
    
    /**
     *	Allow a Java class to be used as extension (more precisely, its public
     *	methods can be called as extension functions).
     *	<p><b>Caution:</b> using this method enforces an explicit control: all 
     *	classes to be used as extensions must then be explicitly declared.
     *	This is a security feature.
     *	@param className fully qualified name of Java class,
     *	for example <pre>java.io.File</pre>
     */
    public void authorizeClass(String className) {
        predefined.authorizeJavaClass(className);
    }
    
    /**
    *	Parses and checks a query from a text input.
    *  Errors are reported through the log.
    * @param textInput the actual text to parse
    * @param uri of the query source (for messages), or null if not applicable.
    * @param log message collector
    *  @return the parsed and type-checked query if no error is detected.
    *  @throws SyntaxException as soon as a syntax error is detected.
    *  @throws XQueryException at end of compilation if static errors have been
    *  detected (error details are reported through the Log).
    */
    public XQuery compileQuery( CharSequence textInput, String uri, Log log )
    throws XQueryException {
        if(textInput == null)
            throw new IllegalArgumentException("null textInput");
        if(uri == null)
            throw new IllegalArgumentException("null URI");
        if(log == null)
            throw new IllegalArgumentException("null log");
        if(moduleMan == null)
            throw new XQueryException("no Module Manager specified");
        if(docMan == null)
            throw new XQueryException("no Document Manager specified");
        Parser parser = new Parser(moduleMan);
        parser.setPredefinedModule(predefined);
        parser.setCollations(collations);
        if(extraNS != null)
            parser.setPredefinedNamespaces(extraNS);
	try {
	    int initialErrCnt = log.getErrorCount();
	    XQuery query = parser.parseQuery( textInput, uri, log );
	    query.setBaseURI( docMan.getBaseURI() );
	    int errCnt = log.getErrorCount() - initialErrCnt;
	    if(errCnt == 0)
		query.staticCheck( moduleMan, log );
	    log.flush();  // proper error display before raising an exception
	    errCnt = log.getErrorCount() - initialErrCnt;
	    if(errCnt > 0)
		throw new XQueryException("static analysis: "+ errCnt
					  + " error"+ (errCnt > 1? "s":""));
	    return query;
	}
	catch (SyntaxException e) {
	    throw new XQueryException("syntax: "+ e.getMessage(), e);
	}
	catch (SecurityException sec) {
	    throw new XQueryException(sec.getMessage(), sec);
	}
    }
    
    /**
     *	Helper for compiling a query from a stream. See {@link #compileQuery}
     *	@param input query source
     *	@param uri uri of the source (can be a dummy value)
     *	@param log message collector
     */
    public XQuery compileQuery( Reader input, String uri, Log log )
	throws XQueryException, IOException {
        StringBuffer buffer = new StringBuffer(1000);
        int count;
        char[] chars = new char[4096];
        while ((count = input.read(chars, 0, chars.length)) > 0)
            buffer.append(chars, 0, count);
        return compileQuery(buffer, uri, log);
    }
    
    /**
     *	Helper for compiling a query from a file. See {@link #compileQuery}
     *	@param input query source
     *	@param log message collector
     */
    public XQuery compileQuery( File input, Log log )
	throws XQueryException, IOException {
        return compileQuery(new FileReader(input),
			    input.getAbsolutePath(), log);
    }

    private DefaultEvalContext prepareQuery(XQuery query )
	throws XQueryException {
        if(query == null)
            throw new IllegalArgumentException("null query");
        if( docMan == null )
            throw new EvalException("no Document Manager defined");
        execContext = new DefaultEvalContext(query, query.getLocalSize());
	execContext.setProperties(properties);
        execContext.setDocumentManager(docMan);
        execContext.setDefaultOutput(defaultOutput);
        execContext.setLog(log);
        try {
            if(implicitTimezone != null)
                execContext.setImplicitTimezone(implicitTimezone);
        }
        catch (DateTimeException e) {
            throw new XQueryException("implicit timezone: "+e.getMessage());
        }
        if(defaultCollation != null)
            query.setDefaultCollation(defaultCollation);

        if(input != null)
            execContext.setInput(input);
	else if(docInputURI != null)
	    execContext.setInput( execContext.getDocument(docInputURI) );
		
	Object traceExec = getSysProperty("sys:traceExec");
	if(traceExec != null && traceExec != Boolean.FALSE)
	    execContext.setExecTrace(true);
	Integer timeout = (Integer) getSysProperty("sys:timeout");
	if(timeout != null) {
	    final DefaultEvalContext fctx = execContext;
	    Timer.request( timeout.intValue(), new Timer.Handler() {
		    public void timeEvent( Timer.Request r ) {
			fctx.setTimeOut(true);
		    }
		});
	}
        query.initGlobals(execContext, globals);
        return execContext;
    }
    
    /**
     *	Executes a query in the static and dynamic environment provided by
     *	this processor.
     *	The Document Manager is used for access to documents by the XQuery
     *	function doc(), the {@link #setDocumentInput} or
     *	{@link #setCollectionInput} methods define
     *  the data accessible by the XQuery function input().
     *  @param query a query compiled with compileQuery.
     *  May be used by several threads.
     *  @return the evaluated value. Enumerates items and their value.
     *  @throws EvalException run-time exception. A stack trace can be obtained
     *  from the exception.
     *  @throws XQueryException other exception. Happens only in serious cases.
     */
    public Value executeQuery( XQuery query ) throws XQueryException {
        prepareQuery(query);
	Value res = query.eval( null, execContext );	// no initial focus
	execContext = null;
	return res;
    }
    
    /**
     *	Executes a query with direct output to a serial XML event receiver
     *	 (SAX or XML stream). The query must evaluate as a single Node, 
     *	otherwise an EvalException is raised.
     *  <p>See {@link #executeQuery(XQuery query)} for more details
     *  @param query a query compiled with compileQuery.
     *  May be used by several threads.
     *  @param receiver a handler for generated events. In practice a
     *  XMLSerializer or a SAXEventReceiver.
     */
    public void executeQuery( XQuery query, XMLEventReceiver receiver )
	throws XQueryException {
        prepareQuery(query);
        
        receiver.reset();
        receiver.definePrefixHints( query.getInScopeNS() );
        try {
	    query.evalAsEvents(receiver, null, execContext);// no initial focus
	    receiver.terminate();
        }
        catch (DataModelException e) {
            execContext.error(query.body, new EvalException(e.getMessage(), e));
        }
	execContext = null;
    }
    
    /**
     *	Executes a query with direct output to a serial XML stream (with the 
     *	default serialization options).
     *	The query must evaluate as a single Node, otherwise an EvalException
     *	is raised.
     *  <p>See {@link #executeQuery(XQuery query)} for more details
     */
    public void executeQuery( XQuery query, Writer output )
	throws XQueryException {
        XMLSerializer serialOut = new XMLSerializer();
        serialOut.setOutput(output);
        executeQuery(query, serialOut);
    }

    /**
     *	Stops a running execution.
     *	Must of course be called by another thread.
     */
    public void stopExecution() {
	if(execContext != null)
	    execContext.setStopped(true);
    }

    /**
     *	An object used to pause execution. See {@link #pauseExecution}.
     */
    public interface PauseHandler {
	/**
	 *  Called when the pause request is detected. Passes the current
	 *  evaluation context, which can be used for tracing the stack or
	 *  printing the local variables.
	 */
	void pauseAt( EvalContext ctx );
    }

    /**
     *	Requests asynchronously a pause in execution.
     *	Upon reception of this request, the processor calls the 'pauseAt'
     *	method of the PauseHandler interface, then suspends execution
     *	by waiting on the PauseHandler object.
     *	Therefore a notify() has to be done on the handler to restart it.
     */
    public void pauseExecution( PauseHandler pause ) {
	if(execContext != null)
	    execContext.setPauseHandler(pause);
    }

    /**
     *	Compiles and executes a simple expression in the context specified.
     *	The expression has access to the global context but not to local
     *	variables.
     */
    public Value eval( String expression, Focus focus, EvalContext context )
	throws XQueryException {
        Parser parser = new Parser(moduleMan);
	// this should be obtained from context:
        parser.setPredefinedModule(predefined);
        parser.setCollations(collations);
        if(extraNS != null)
            parser.setPredefinedNamespaces(extraNS);
	XQuery expr = parser.parseInContext(expression, context);

	Log log = context.getLog();
	int initialErrCnt = log.getErrorCount();
	expr.staticCheck( moduleMan, log );
	int errCnt = log.getErrorCount() - initialErrCnt;
	if(errCnt > 0)
	    throw new XQueryException("static analysis: "+ errCnt
				      + " error"+ (errCnt > 1? "s":""));
	return expr.eval( focus, context );
    }

    /**
     *  Runs a XSLT transformation on an element (internal use).
     *  @param source node on which the transformation is applied.
     *  @param templates URI of a stylesheet (resolved by the Module Manager).
     *  @param parameters a set name/value pairs specifying initial values for
     *  global xsl:param in the stylesheet.
     *  @param options a set name/value pairs specifying 
     *  options for the transformation:<ul>
     *	<li> 'output-file': name of an output file for the transformation;
     *	in that case a null tree is returned and a StreamResult is used.
     *	<li> standard JAXP "output keys"
     *	<li> other options specific to the XSLT engine can be accepted.
     *	</ul>
     *  @return a Document in memory, or null if the options specify an
     *  output file.
     */
    public IDocument xslTransform (Node source, String templates,
				   Properties parameters, Properties options)
	throws TransformerException, XQueryException {
	// cached loading:
	Templates sheet = moduleMan.loadTemplates(templates);
	Transformer trans = sheet.newTransformer();
	Source tSource = null;
	Result result = null;
	IDocument tree = null;

	// use SAX input for the XSLT engine. This is not optimal (because
	// we copy the tree), but acceptable for small or medium-size transfo.
	tSource = new SAXSource(new SAXSourceWrapper(source),
			// dummy source: most engines dont like null input!
				new InputSource(new StringReader("<dummy/>")));
	// is there an output file?
	String out = options.getProperty(XSLT_OUTPUT_FILE);
	if(out != null) {
	    result = new StreamResult(out);
	}
	else {	// build a tree:
	    tree = new IDocument();
	    SAXResult tresult = new SAXResult();
	    tresult.setHandler(tree);
	    tresult.setLexicalHandler(tree);
	    tresult.setSystemId(FileUtil.fileToURLName(".")); // Saxon wants it!
	    result = tresult;
	}
	// options (String value always)
	for(Enumeration enum = options.keys() ; enum.hasMoreElements(); ) {
	    String name = (String) enum.nextElement();
	    if(name.equals(XSLT_OUTPUT_FILE))
		continue;
	    trans.setOutputProperty( name, options.getProperty(name) );
	}
	// parameters (String value always)
	for(Enumeration enum = parameters.keys() ; enum.hasMoreElements(); ) {
	    String name = (String) enum.nextElement();
	    trans.setParameter( name, parameters.get(name) );
	}
	trans.transform(tSource, result);
	return tree;
    }
}

