/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import net.xfra.qizxopen.util.Collations;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dt.NodeType;
import net.xfra.qizxopen.xquery.dt.UntypedAtomicType;
import net.xfra.qizxopen.xquery.dt.Conversion;

import java.text.Collator;
import java.io.PrintStream;
import java.math.BigDecimal;

/**
 *	Implementation of XQuery Data Model above Fully-ordered documents.
 *	Used for indexed and parsed documents.
 */
public class FONIDataModel extends FONIDM
{
    public FONIDataModel( FONIDocument dom ) {
	super(dom);
	root = newNode(dom.getRootNode());
    }

    public Node getDocumentNode() {
	return (Node) root;
    }

    public Node newNode( int id ) {
	if(id == 0)
	    return null;
	return new INode(id);	
    }

    public net.xfra.qizxopen.dm.Node newDmNode( int id ) { 
	return newNode(id);
    }

    public static Node convertBasicNode( FONIDM.BaseNode node ) {
	FONIDataModel dm = new FONIDataModel(node.getDom());
	return dm.newNode( node.getNodeId() );
    }

    // Basic Node (single id)
    class INode extends FONIDM.BaseNode implements Node {
	INode( int id ) {
	    super(id);
	}

	public ItemType  getType() {
	    return NodeType.getTypeByKind( dom.getKind(id) );
	}

	public Node  getParent() {
	    return newNode(dom.getParent(id));
	}

	public Value  getTypedValue() { //TODO
	    return null;	
	}

	public Value getChildren() {
	    // OPTIMIZED by avoiding an inner iterator
	    //return dom.newChildrenIterator(id);
	    return new ISequence(- dom.getFirstChild(id));
	}

	public Value getAttributes() {
	    return new ASequence( id, dom.attrIterator(id) );
	}

	public Value getNamespaces( boolean inScope ) {
	    return new ASequence( id, dom.namespaceIterator(id, inScope) ) {
		    ANode makeNode(int id) {
			return new NSNode(ownerId, id);
		    }
		};
	}

	public boolean deepEqual( Item item, Collator collator ) {
	    if(! (item instanceof Node) )
		return false;
	    try {
		
		return deepEq( id, (Node) item, collator);
	    } catch (Exception e) {
		return false;   //should not happen
	    }
	}

	/**
	 * Value comparison: equivalent to (untypedAtomic, string-value).
	 * TODO schema import
	 */
	public int compareTo( Item that, Collator collator, int implicitTimeZone)
	    throws TypeException {
	    // TODO: can be optimised by traversal comparison (stream on the stringvalue)
	    return UntypedAtomicType.comparison( this, that, collator );
	}

	

	public Node  getDocument() {
	    return newNode(dom.getRootNode());
	}

	public Node  getAttribute( QName name ) {
	    return newNode(dom.getAttribute(id, dom.internOtherName(name)));
	}

	

	public Value getAncestors( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(id, nodeTest);
	}

	public Value getAncestorsOrSelf( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(id, nodeTest);
	}

	public Value getParent( NodeTest nodeTest ) {
	    return new Parent(id, nodeTest);
	}

	public Value getChildren( NodeTest nodeTest ) {
	    return new Children(id, nodeTest);
	}

	public Value getDescendants( NodeTest nodeTest ) {
	    return new Descendants(id, nodeTest);
	}

	public Value getDescendantsOrSelf( NodeTest nodeTest ) {
	    return new DescendantsOrSelf(id, nodeTest);
	}

	public Value getAttributes( NodeTest nodeTest ) {
	    return new Attributes(id, nodeTest);
	}

	public Value getFollowingSiblings( NodeTest nodeTest ) {
	    return new FollowingSiblings(id, nodeTest);
	}

	public Value getPrecedingSiblings( NodeTest nodeTest ) {
	    return new PrecedingSiblings(id, nodeTest);
	}

	public Value getFollowing( NodeTest nodeTest ) {
	    return new Following(id, nodeTest);
	}

	public Value getPreceding( NodeTest nodeTest ) {
	    return new Preceding(id, nodeTest);
	}

	

	public boolean  isNode() {
	    return true;
	}

	public Node     asNode() throws TypeException {
	    return this;
	}

	public Item     asItem() throws TypeException {
	    return this;
	}

	public String   asString()  throws TypeException {
	    return getStringValue();
	}
    
	public boolean  asBoolean() throws TypeException {
	    return asString().length() != 0;	// OPTIM	
	}

	public long     asInteger() throws TypeException {
	    return Conversion.toInteger( asString() ); 
	}

	public BigDecimal asDecimal() throws TypeException {
	    return Conversion.toDecimal( asString() ); 
	}

	public float    asFloat() throws TypeException {
	    return Conversion.toFloat( asString() ); 
	}

	public double   asDouble() throws TypeException {
	    return Conversion.toDouble( asString() ); 
	}
    }
 
    // Attribute and namespace nodes: 
    // the id is the owner node's, an additional offset points the value
    class ANode extends INode
    {
	int offset;

	ANode( int id, int offset ) {
	    super(id);
	    this.offset = offset;
	}

	public boolean equals( Object that ) {
	    if(!(that instanceof ANode))
		return false;
	    ANode thatNode = (ANode) that;
	    return thatNode.id == id && thatNode.offset == offset &&
		   thatNode.getDom() == dom;
	}

	public int hashCode() {
	    return id * offset;	// CAUTION id and offset are close to each other
	}

	public int orderCompare( Node node ) {
	    int cmp = super.orderCompare(node);
	    if(cmp != 0)
		return cmp;
	    if( !(node instanceof ANode) )
		return 1;	// after owner
	    // attributes of the same node:
	    ANode anode = (ANode) node;
	    cmp = offset - anode.offset;
	    return cmp < 0 ? -1 : cmp > 0 ? 1 : 0;
	}

	public int getNature() {
	    return Node.ATTRIBUTE;
	}

	public String getNodeKind() {
	    return "attribute";
	}

	public ItemType  getType() {
	    return Type.ATTRIBUTE;
	}

	public QName  getNodeName() {
	    return dom.getOtherName(dom.pnGetNameId(offset));
	}

	public String getStringValue() {
	    return dom.pnGetStringValue(offset);
	}

	public Node getParent() {
	    return newNode(id);
	}

	public Value getChildren() {
	    return Value.empty;
	}

	public Value getAttributes() {
	    return Value.empty;
	}

	public Value getNamespaces() {
	    return Value.empty;
	}
    }

    // namespace Node:
    class NSNode extends ANode {
	NSNode( int id, int offset ) {
	    super(id, offset);
	}
	public String getNodeKind() {
	    return "namespace";
	}
	public int getNature() {
	    return Node.NAMESPACE;
	}
	public ItemType  getType() {
	    return Type.NAMESPACE;
	}
    }

    // Sequence on INodes:
    class ISequence extends NodeSequenceBase
    {
	int startId, curId;

	ISequence( int startId ) {
	    curId = this.startId = startId;
	}

	public Value  bornAgain() {
	    return new ISequence( startId );
	}

	public boolean next() {
	    if(curId < 0)
		 curId = -curId;	// trick for children iterator
	    else curId = dom.getNextSibling(curId);
	    return curId != 0;
	}

	public int     currentId() {
	    return curId;
	}

	public Node  asNode() {
	    return curId == 0 ? null : (Node) newNode(curId);
	}

	public ItemType  getType() {
	    return NodeType.getTypeByKind( dom.getKind(curId) );
	}
    }

    abstract class TypedSequence extends ISequence {

	NodeTest nodeTest;
	int nameId = -1;
	boolean started = false;

	TypedSequence( int id, NodeTest nodeTest ) {
	    super(id);
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    if( curId == 0)
		return false;
	    return ( nodeTest == null ||
		     (nodeTest.needsNode() ? nodeTest.accepts(newNode(curId))
		      : nodeTest.accepts( dom.getKind(curId), dom.getName(curId) ) ));
	}
    }

    class Parent extends TypedSequence
    {

	Parent( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public boolean next() {
	    if(started)
		return false;
	    started = true;
	    curId = dom.getParent(curId);
	    return checkNode();
	}
	
	public Value  bornAgain() {
	    return new Parent( startId, nodeTest );
	}
    }

    class AncestorsOrSelf extends TypedSequence
    {
	AncestorsOrSelf( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public boolean next() {
	    for(; curId != 0; ) {
		if(started)
		    curId = dom.getParent(curId);
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public Value  bornAgain() {
	    return new AncestorsOrSelf( startId, nodeTest );
	}
    }

    class Ancestors extends AncestorsOrSelf
    {
	Ancestors( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    started = true;
	}

	public Value  bornAgain() {
	    return new Ancestors( startId, nodeTest );
	}
    }

    class Children extends TypedSequence
    {
	Children( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    curId = dom.getFirstChild( id );
	}

	public Value  bornAgain() {
	    return new Children( startId, nodeTest );
	}

	public boolean next() {
	    for(; curId != 0; ) {
		if(started)
		     curId = dom.getNextSibling( curId );
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class DescendantsOrSelf extends TypedSequence
    {
	int lastNode;

	DescendantsOrSelf( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    lastNode = dom.getNodeAfter(Math.abs(id));
	    if(lastNode == 0)
		lastNode = Integer.MAX_VALUE;
	}

	public Value  bornAgain() {
	    return new DescendantsOrSelf( startId, nodeTest );
	}

	public boolean next() {
	    for(; curId != 0; ) {
		if(started)
		    curId = dom.getNodeNext(curId);
		started = true;
		if(curId >= lastNode)
		    curId = 0;
		else if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Descendants extends DescendantsOrSelf
    {
	Descendants( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    started = true;
	}

	public Value  bornAgain() {
	    return new Descendants( startId, nodeTest );
	}
    }

    class FollowingSiblings extends TypedSequence
    {
	FollowingSiblings( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public Value  bornAgain() {
	    return new FollowingSiblings( startId, nodeTest );
	}

	public boolean next() {
	    for( ; curId != 0; ) {
		curId = dom.getNextSibling(curId);
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Following extends TypedSequence
    {
	Following( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public Value  bornAgain() {
	    return new Following( startId, nodeTest );
	}

	public boolean next() {
	    for( ; curId != 0; ) {
		curId = dom.getNodeNext(curId);
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class PrecedingSiblings extends TypedSequence
    {
	PrecedingSiblings( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    curId = dom.getFirstChild( dom.getParent(id) );
	}

	public Value  bornAgain() {
	    return new PrecedingSiblings( startId, nodeTest );
	}

	public boolean next() {
	    for( ; curId != 0; ) {
		if(started)
		    curId = dom.getNextSibling(curId);
		started = true;
		if(curId == startId) {
		    curId = 0; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Preceding extends PrecedingSiblings
    {
	Preceding( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    curId = dom.getRootNode();
	}

	public Value  bornAgain() {
	    return new Preceding( startId, nodeTest );
	}

	public boolean next() {
	    for( ; curId != 0; ) {
		if(started)
		    curId = dom.getNodeNext(curId);
		started = true;
		if(curId == startId) {
		    curId = 0; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class ASequence extends NodeSequenceBase {
	FONIDocument.NodeIterator domIter;
	int ownerId;
	int curId;

	ASequence( int ownerId, FONIDocument.NodeIterator iter ) {
	    domIter = iter;
	    this.ownerId = ownerId;
	}

	public boolean next() {
	    if( !domIter.next())
		return false;
	    curId = domIter.currentId();
	    
	    return true;
	}

	public Node  asNode() {
	    curId = domIter.currentId();
	    return curId == 0 ? null : (Node) makeNode(curId);
	}

	public ItemType  getType() {
	     return Type.ATTRIBUTE;
	}

	ANode makeNode(int id) {
	    return new ANode(ownerId, id);
	}

	public Value  bornAgain() {
	    return new ASequence( ownerId, domIter.reborn() );
	}
    }

    class Attributes extends ASequence
    {
	NodeTest nodeTest;

	Attributes( int ownerId, NodeTest nodeTest ) {
	    super(ownerId, dom.attrIterator(ownerId));
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    if( curId == 0)
		return false;
	    return ( nodeTest == null ||
		     (nodeTest.needsNode() ? nodeTest.accepts(makeNode(curId))
		      : nodeTest.accepts( Node.ATTRIBUTE, dom.pnGetName(curId) ) ));
	}

	public Value  bornAgain() {
	    return new Attributes( ownerId, nodeTest );
	}

	public boolean next() {
	    for( ; super.next() ; )
		if(checkNode())
		    return true;
	    return false;
	}
    }
}
