/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.dt;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.Node;
import java.util.Vector;

/**
 *	A sequence that stores items in an array. Used for local variables holding a true
 *	sequence. 
 */
public class ArraySequence extends GenericValue
{
    protected Object[] items;	// not Item[] to avoid reallocating after FLWR sort
    protected int size;
    protected int index = -1;
    protected Vector overflow;	// when growing too much, use overflow blocks
    final static int MAX_BLOCK_SIZE = 65536;

    public ArraySequence( Object[] items, int size ) {
        this.items = (items == null)? null : (Object[]) items.clone();
	this.size = size;
    }

    public ArraySequence( int initialSize ) {
        items = new Object[initialSize];
	size = 0;
    }

    public Value  bornAgain() {
	return new ArraySequence( items, size );
    }

    public int getSize() {
	return size;
    }

    public void addItem( Item item ) {
	//if(item == null) new Exception().printStackTrace();
	if(size >= items.length) {
	    if(items.length < MAX_BLOCK_SIZE) {
		Object[] old = items;
		items = new Object[ old.length * 2 ];
		System.arraycopy(old, 0, items, 0, old.length);
	    }
	    else {
		// avoid big blocks: accumulate blocks into a vector,
		// then use pack() to cleanup
		if(overflow == null)
		    overflow = new Vector();
		overflow.add(items);
		items = new Object[items.length];
		size = 0;
	    }
        }
	items[size ++] = item;
    }

    public void addItems( Item[] aitems, int count ) {
	if(size + count >= items.length) {	// OPTIM todo
	    for(int i = 0; i < count; i++)
		addItem(aitems[i]);
	}
	else {	// most frequent case: no OVF
	    System.arraycopy(aitems, 0, items, size, count);
	    size += count;
	}
    }

    /**
     *	if addItem has been used, must be called at the end of the construction.
     */
    public void pack() {
	if(overflow == null)
	    return;
	int nsize = overflow.size() * items.length + size;
	Object[] nitems = new Object[nsize];
	int ptr = 0;
	for(int b = 0; b < overflow.size(); b++) {
            System.arraycopy((Object[]) overflow.get(b), 0, nitems, ptr, items.length);
	    ptr += items.length;
        }
	System.arraycopy(items, 0, nitems, ptr, size);
	items = nitems;
	size = nsize;
	overflow = null;
	reset();
    }

    public boolean next() throws XQueryException {
	// automatic packing:
	if(overflow != null)
	    pack();
	
	if(index >= size-1) 
	    return false;
	item = (Item) items[++ index];
	return true;
    }

    public long quickCount(EvalContext context) throws XQueryException {
	// no need to pack (assumes no duplication)
	int ovfsize = (overflow!= null)? overflow.size() : 0;
	return ovfsize * items.length + size;
    }

    public void reset() {
	index = -1;
    }

    public Item asItem() throws TypeException {
	
	return  item;
    }

    public boolean notWorthExpanding() {
	return true;	// because it is already expanded!
    }

    /**
     *	Converts a Value representing a sequence of Nodes into an array.
     */
    public static Node[] expandNodes(Value value) throws XQueryException {
	Node[] nodes = new Node[8];
	int ptr = 0;
	for(; value.next(); ) {
	    if(ptr >= nodes.length) {
		Node[] old = nodes;
		nodes = new Node[ old.length * 2 ];
		System.arraycopy(old, 0, nodes, 0, old.length);
	    }
	    nodes[ptr ++] = value.asNode();
        }
	// return a full array: TODO OPTIM
	Node[] rnodes = new Node[ptr];
	System.arraycopy(nodes, 0, rnodes, 0, ptr);
	return rnodes;
    }
} 
