/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.fn;
import net.xfra.qizxopen.xquery.impl.*;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.LocalVariable;
import net.xfra.qizxopen.xquery.dt.SingleInteger;
import net.xfra.qizxopen.xquery.dt.NodeType;

/**
 *  User Function Definition.
 *  As a subclass of Function, this descriptor holds several prototypes if
 *  user functions are overloaded.
 */
public class UserFunction extends Function {

    Signature[] protos = new Signature[1];
    Module module;
    int    location;
    int    localSize;

    public UserFunction( Module module, int location ) {
	this.module = module;
	this.location = location;
    }

    public static class Signature extends Prototype {
	public Expression body;
	public Expression[] argInit;	// used for XSLT2: default value
	public boolean[]    tunnelArg;	// used for XSLT2: tunnel parameter
	// argument storage (register or plain local var)
	LocalVariable[] argDecls;

	Signature( QName qname ) {
	    super(qname, Type.ANY, Call.class);
	    argInit = new Expression[argNames.length];	
	    tunnelArg = new boolean[argNames.length];
	}

	public void arg( QName name, Type argType,
			 Expression init, boolean tunnel) {
	    super.arg(name, argType);
	    if(argCnt >= argInit.length) {
		Expression[] oldi = argInit;
		argInit = new Expression[ argNames.length ];
		System.arraycopy(oldi, 0, argInit, 0, oldi.length);

		boolean[] old = tunnelArg;
		tunnelArg = new boolean[ argNames.length ];
		System.arraycopy(old, 0, tunnelArg, 0, old.length);
	    }
	    argInit[argCnt - 1] = init;
	    tunnelArg[argCnt - 1] = tunnel;
	}
    }

    public Prototype[] getProtos() { return protos; } // never called

    public Signature addPrototype( QName name ) {
	Signature proto = new Signature(name);
	if(protos[0] == null)
	    protos[0] = proto;
	else {
            Signature[] old = protos;
            protos = new Signature[ old.length + 1 ];
            System.arraycopy(old, 0, protos, 0, old.length);
	    protos[old.length] = proto;
        }
	return proto;
    }

    public void dump( ExprDump d ) {
	d.println("UserFunction" );
	for(int p = 0; p < protos.length; p++) {
	    d.display("proto", protos[p].toString());
	    d.display("body", protos[p].body);
	}
    }

    public void  setLocalSize( int size ) {
	localSize = size;
    }

    public Value eval( Focus focus, EvalContext context ) {
        throw new RuntimeException("UserFunction not evaluable");
    }

    // Static analysis of the definition
    public void staticCheck( StaticContext context ) {
	LocalVariable mark = context.markLocalVariables();
	for(int pr = 0; pr < protos.length; pr++) {
	    Signature proto = protos[pr];

	    proto.argDecls = new LocalVariable[proto.argCnt];
	    for(int p = 0; p < proto.argCnt; p++) {
		QName param = proto.argNames[p];
		proto.argDecls[p] = 
		    context.defineLocalVariable(param, proto.argTypes[p], null);
	    }

	    if(proto.body == null) {
		context.getLog().error( module, location,
				   "external functions not supported this way");
		return;
	    }
	    proto.body = context.staticCheck( proto.body , 0 );
	    
	    if(!proto.returnType.accepts(proto.body.getType()))
		context.error( proto.body,
		       "improper type %1 for returned expression, expecting %2",
			       proto.body.getType().toString(context),
			       proto.returnType.toString(context));
	}
	context.popLocalVariables(mark);
    }

    private boolean needsDynamicCheck( Type t ) {
	NodeTest nt = null;
	return t instanceof NodeType &&
	       (nt = ((NodeType) t).nodeTest) != null &&
	       !nt.staticallyCheckable();
    }

    // Static analysis of a call
    public Expression staticCheck(StaticContext context, Expression[] arguments,
				  Expression ocall ) 
    {
	Call call = (Call) context.resolve(protos, arguments, ocall);
	call.def = this;
	Signature proto = (Signature) call.prototype;
	call.argChecks = new Type[arguments.length];

	int argCnt = Math.min( arguments.length, proto.argCnt);	// protect
	for(int a = 0; a < argCnt; a++) {
	    // if formal arg of type item* or anytype: no check
	    Type formal = proto.argTypes[a], actual = arguments[a].getType();
	    boolean dynCheck = needsDynamicCheck(formal.getItemType());
	    // if actual arg has occ 1: no check, unless specific node tests
	    if( actual.getOccurrence() == Type.ONE_OCC) {
		if(dynCheck)
		    call.argChecks[a] = formal;
	    }
	    // if occ != *, need to check the occ count: Type.accepts(value)
	    if( formal.getOccurrence() != Type.OPTMULTI_OCC || dynCheck )
		call.argChecks[a] = formal;
	}
	return call;
    }

    /**
     *	Run-time User Function implementation.
     */
    public static class Call extends Function.Call
    {
	UserFunction def;
	Expression body;
	Type[]  argChecks;	  // type actually used for dynamic checking
	LocalVariable[] argDecls; // just points to the prototype's

	public void dump( ExprDump d ) {
	    d.header( this, getClass().getName() );
	    d.display("prototype", prototype.toString());
	    d.display("localSize", ""+ getLocalSize());
	    d.display("actual arguments", args);
	}

	public int getLocalSize() {
	    return def.localSize;
	}

	protected EvalContext createContext( Focus focus, EvalContext context )
	    throws XQueryException {
	    // due to deferred evaluation in many constructs (Flower, sequence,
	    // paths...), the function context (arguments and locals) has to
	    // be allocated (cant be in a classical stack)
	    // and *can even persist after exit of the function*.
	    EvalContext callee = context.subContext( this );
	    Type[] argTypes = prototype.argTypes;

	    // due to forward references, this might be uncompletely resolved:
	    if(body == null) {
		Signature proto = (Signature) prototype;
		body = proto.body;	// in case not 
		argDecls = proto.argDecls;
	    }
	    
	    // eval and bind actual arguments to formal arguments:
	    for(int a = 0; a < args.length; a++) {
		try {
		   callee.storeLocal(argDecls[a].address, args[a], argChecks[a],
				     focus, context );
		} catch(TypeException err) {
		    context.error( args[a], "type mismatch on argument '"+
				   prototype.argNames[a]+"' of function "+
				   prototype.displayName(context.getStaticContext())+
				   " : "+ err.getMessage() );
		}
	    }
	    context.at(this);
	    return callee;
	}

	public Value eval( Focus focus, EvalContext context )
	    throws XQueryException {

	    context.at(this);
	    // CAUTION: createContext can change 'body', so make sure of what happens
	    EvalContext called = createContext(focus, context);

	    Value result = body.eval( null,   // context item not inherited
				      called );
	    return result;
	}

	// ----------- optimizations: 

	public void evalAsEvents(XMLEventReceiver output,
				 Focus focus, EvalContext context)
	    throws XQueryException, DataModelException {
	    context.at(this);

	    // CAUTION: createContext can change 'body'
	    EvalContext called = createContext(focus, context);

	    // context item not inherited
	    body.evalAsEvents( output, null, called);
	}

	public long evalAsInteger( Focus focus, EvalContext context)
	    throws XQueryException {
	    // CAUTION: createContext can change 'body'
	    EvalContext called = createContext(focus, context);

	    // context item not inherited
	    return body.evalAsInteger( null, called );
	}

	public long evalAsOptInteger( Focus focus, EvalContext context)
	    throws XQueryException {
	    // CAUTION: createContext can change 'body'
	    EvalContext called = createContext(focus, context);

	    // context item not inherited
	    return body.evalAsOptInteger( null, called );
	}

	public double evalAsDouble( Focus focus, EvalContext context)
	    throws XQueryException {
	    // CAUTION: createContext can change 'body'
	    EvalContext called = createContext(focus, context);

	    // context item not inherited
	    return body.evalAsDouble( null, called );
	}

	public double evalAsOptDouble( Focus focus, EvalContext context)
	    throws XQueryException {
	    // CAUTION: createContext can change 'body'
	    EvalContext called = createContext(focus, context);

	    // context item not inherited
	    return body.evalAsOptDouble( null, called );
	}
    }
}
