/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery.impl;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.*;
import net.xfra.qizxopen.util.*;

import java.lang.reflect.*;

/**
 *	Management of predefined functions and variables.
 *	Features an automatic registration of functions.
 */
public class PredefinedModule extends Module
{
    // before next decl!
    static String funPack = Function.class.getPackage().getName();
    // unique reference instance:
    public static final PredefinedModule BASE = new PredefinedModule();


    FunctionPlugger[] pluggers = new FunctionPlugger[0];
    JavaFunction.Plugger javaPlugger;

    /**
     *	Mechanism to dynamically load/plug functions:
     * When an unknown function is encountered by the parser, it calls
     * PredefinedModule.localFunctionLookup which in turn asks 
     * its registered FunctionPluggers to look for this function.
     * This mechanism manages predefined functions fn: (with the side benefit
     * that the system start-up is faster because functions are lazily loaded),
     * type functions (xs: and xdt:), Java extensions, and application extensions.
     */
    public interface FunctionPlugger {
	Function plug( QName functionName, PredefinedModule target )
	    throws SecurityException;
    }

    public PredefinedModule() {
	// fn:functions: name is camelCased,
	//    eg get-year-from-date -> GetYearFromDate
	registerFunctionPlugger(new BasicFunctionPlugger(Namespace.FN,
							 funPack+ ".%C"));
	// Type functions: like fn:functions but the name is left untouched
	registerFunctionPlugger(new BasicFunctionPlugger(Namespace.XSD,
							 funPack+".XS_%n"));
	registerFunctionPlugger(new BasicFunctionPlugger(Namespace.XDT,
							 funPack+".XS_%n"));
	// Java binding:
	registerFunctionPlugger(javaPlugger = new JavaFunction.Plugger());

        try {
	    // explicit declaration of two functions with problematic names:
	    declareFunction(new _Boolean());
	    declareFunction(new _String());
	    // predefined global var $arguments contains command line options
	    defineGlobal( QName.get("arguments"), Type.STRING.star );
        }
        catch (Exception e) { e.printStackTrace(); }

	importedModules.setSize(0);	// final
	predefined = null;
    }

    /**
     *	Used by applications: 
     */
    public void registerFunctionPlugger( FunctionPlugger plugger ) {
	FunctionPlugger[] old = pluggers;
	pluggers = new FunctionPlugger[ old.length + 1 ];
	System.arraycopy(old, 0, pluggers, 0, old.length);
	pluggers[old.length] = plugger;
    }

    /**
     *	Explicitly authorizes a Java class to be used as extension.
     *<p>Caution: this is a security feature. By default, all classes are allowed.
     *	However, when this method is used once, the control becomes explicit and
     *	each class whose methods are used as extension functions must be declared.
     */
    public void authorizeJavaClass( String className ) {
	javaPlugger.authorizeClass(className);
    }

    public Function localFunctionLookup( QName name ) throws SecurityException {
	// caching:
	Function fun = (Function) functionMap.get(name);
	if(fun != null)
	    return fun;
	// invoke pluggers:
	for(int p = 0; p < pluggers.length; p++)
	    if( (fun = pluggers[p].plug(name, this)) != null ) {
		declareFunction(fun);
		return fun;
	    }
	// maybe failure: but try again, as a Java class might have declared 
	// the function in its plug hook
	return (Function) functionMap.get(name);
    }

    /**
     *	Basic implementation of function plugger: recognize function by NS and
     *	instantiate a pattern with the localname. The pattern can contain
     *	%n (unmodified name of the function),
     *	%c (camelcased name),
     *	%C (camelcased and capitalized name).
     */
    public static class BasicFunctionPlugger implements FunctionPlugger {
	Namespace ns;
	String pattern;

	public BasicFunctionPlugger( Namespace ns, String pattern ) {
	    this.ns = ns;
	    this.pattern = pattern;
	}

	public Function plug( QName functionName, PredefinedModule target ) {
	    
	    if(functionName.getNamespace() != ns) {
		return null;
	    }
	    String name = functionName.getLocalName();
	    StringBuffer className =
		new StringBuffer(pattern.length() + name.length());
	    for(int c = 0, C = pattern.length(); c < C; c++) {
		char ch = pattern.charAt(c);
		if(ch != '%') {
		    className.append(ch); continue;
		}
		ch = pattern.charAt(++ c);
		if(ch == 'n')
		    className.append(name);
		else if(ch == 'C' || ch == 'c')
		    className.append(Util.camelCase(name, ch == 'C'));
	    }
	    try {
		Class fclass = Class.forName(className.toString());
		return (Function) fclass.newInstance();
	    }
	    catch (ClassNotFoundException e) {
		//e.printStackTrace();
	    }
	    catch (Exception ie) {
		System.err.println("*** error instantiating function "+functionName);
		ie.printStackTrace();	// abnormal
	    }
	    return null;
	}
    }
} // end of class PredefinedModule

