/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.dt.GenericValue;
import net.xfra.qizxopen.xquery.dt.ArraySequence;
import net.xfra.qizxopen.xquery.dt.IntegerValue;
import net.xfra.qizxopen.xquery.dt.StringValue;
import java.util.Arrays;

/**
 *  The FLOWER expression.
 *  Features the detection and optimization of joins.
 */
public class FLWRExpr extends Expression {

    Expression[] clauses = new Expression[0];
    public Expression  where;
    public boolean  stableOrder;
    public boolean  checked;
    OrderSpec[]  orderSpecs;
    public Expression  expr;

    static int genId;

    public FLWRExpr( ) {
    }

    public void  addClause( VarClause clause ) {
        clauses = addExpr(clauses, clause);
    }

    // adds before position
    public void  addClause( VarClause clause, VarClause position ) {
        clauses = addExpr(clauses, clause);
	for(int c = clauses.length; --c > 0; ) {
	     clauses[c] = clauses[c - 1];
	     if(clauses[c] == position) {
		clauses[c - 1] = clause;
		break;
	    }
	}
    }

    VarClause  getClause( int rank ) {
        return rank < 0 || rank >= clauses.length ? null : (VarClause) clauses[rank];
    }

    synchronized static int generateId() {
	return ++ genId;
    }

    public void  addOrderSpec( OrderSpec orderSpec ) {
        if(orderSpecs == null)
	    orderSpecs = new OrderSpec[] { orderSpec };
	else {
            OrderSpec[] old = orderSpecs;
            orderSpecs = new OrderSpec[ old.length + 1 ];
            System.arraycopy(old, 0, orderSpecs, 0, old.length);
	    orderSpecs[old.length] = orderSpec;
        }
    }

    OrderSpec  getOrderSpec( int rank ) {
        return rank < 0 || orderSpecs == null || rank >= orderSpecs.length ?
	    null : orderSpecs[rank];
    }

    public Expression child(int rank) {
	if(rank < clauses.length)
	    return clauses[rank];
	rank -= clauses.length;
	if(where != null)
	    if(rank == 0) 
		return where;
	    else -- rank;
	if(rank < orderSpecs.length)
	    return orderSpecs[rank];
	rank -= orderSpecs.length;
	return rank == 0 ? expr : null;
    }

    public void dump( ExprDump d ) {
	d.header( this, "FLOWER" );
        d.display("clauses", clauses);
        if(where != null) d.displayp("where", where);
        //d.display("stableOrder", ""+stableOrder);
        if(orderSpecs != null) d.display("orderby", orderSpecs);
        d.displayp("return", expr);
    }

    public Expression staticCheck( StaticContext context ) {
	LocalVariable mark = context.markLocalVariables();
	// for and let clauses, declare local variables
	for(int c = 0; c < clauses.length; c ++) {
	    ((VarClause) clauses[c]).owner = this;
	    context.staticCheck(clauses[c], 0);
	}

	// where clause:
	if(where != null)
	    where = context.staticCheck(where, 0);

	// returned expression:
	expr = context.staticCheck(expr, 0);
	type = expr.getType().getItemType().star;

	// order by:
	if(orderSpecs != null)
	    for(int os = 0; os < orderSpecs.length; os++)
		context.staticCheck(orderSpecs[os], 0);

	// join detection:
	if(where != null) {
	    JoinInfo jinfo = joinablePredicate(where);
	    if(jinfo != null) {
		// we have a join: 
		// 1- suppress the where or replace it by predicate remainder
		where = jinfo.remainder;
		// 2- insert the join creation as a 'let' before the 'for' clause of
		// the outerloop
		ForClause inner = (ForClause) jinfo.innerVar.owner;

		// actually it is much better to try to put the join creation 
		// outside of as many loops as possible
		LocalVariable outerVar = jinfo.outerVar, prev;
		//inner.expr.dump(new ExprDump(false));
		for(; (prev = outerVar.before) != null; outerVar = prev)
		    if(prev.name == null || Expr.depends(inner.expr, prev))
			break;

		// a special local to store the join table: 
		QName name = QName.get("#join"+ generateId());
		LocalVariable joinVar =
		    new LocalVariable(name, Type.WRAPPED_OBJECT, null);
		joinVar.declareBefore(outerVar);
		LetClause newLet = new LetClause(name);
		joinVar.owner = newLet;
		newLet.varDecl = joinVar;
		// a temporary for creating the join:
		//    replaces $innerVar in innerExpr
		QName tmpName = QName.get("#jtmp");
		LocalVariable tmpVar =
		    new LocalVariable(tmpName, Type.NODE, null);
		joinVar.addAfter(tmpVar);	
		new Expr.VarReplacer(jinfo.innerVar, tmpVar)
		    .visit(jinfo.innerExpr);
		newLet.expr = new Join.Maker(inner.expr, jinfo.innerExpr,
					     tmpVar, jinfo.type);

		VarClause outer = (VarClause) outerVar.owner;
		FLWRExpr outerFlwr = (FLWRExpr) outer.owner;
		newLet.owner = outerFlwr;
		outerFlwr.addClause( newLet, outer );

		// 2- transform the source expression of the inner loop into
		// an access to the join table: join-get(outer-key)
		inner.expr = new Join.GetExpr(jinfo.outerExpr, joinVar,
					      jinfo.comparison);
	    }
	}
	context.popLocalVariables(mark);

	// try to transform a where clause into a predicate 
	// (implies that this is not a join)
	FLWRExpr reduced = tryToRemoveWhere(this);
	// simplify dummy loop?
	return tryToRemoveLoop(reduced);
    }

    // Returns non-null info if the predicate represents a join.
    // The predicate is normalized.
    JoinInfo joinablePredicate(Expression pred) {
	// accept AND, comparison:
	if(pred instanceof AndExpr) {
	    AndExpr and = (AndExpr) pred;
	    int argc = and.args.length;
	    // look for an AND member that is "joinable":
	    //TODO: select best (eg equality is better than inequality)
	    for(int a = 0; a < argc; a++) {
		JoinInfo info = joinablePredicate(and.args[a]);
		if(info != null) {
		    // remove first operand:
		    if(argc == 2)
			info.remainder = and.args[1 - a];
		    else {
			AndExpr nand = new AndExpr();
			nand.args = new Expression[argc - 1];
			System.arraycopy(and.args, 0, nand.args, 0, a);
			System.arraycopy(and.args, a + 1, nand.args, a, argc - a - 1);
			info.remainder = nand;
		    }
		    return info;
		}
	    }
	    return null;
	}
	if( !(pred instanceof Comparison.Exec) )
	    return null;

	Comparison.Exec comp = (Comparison.Exec) pred;
	Expression left = comp.args[0], right = comp.args[1];
	
	if(left instanceof NodeSortExpr)
	    left = ((NodeSortExpr) left).expr;
	if(right instanceof NodeSortExpr)
	    right = ((NodeSortExpr) right).expr;
	
	Expr.ForVarCollector vcol = new Expr.ForVarCollector();
	vcol.visit(left);
	if(vcol.found == null || vcol.found2 != null)
	    return null;
	LocalVariable var1 = vcol.found;
	vcol.found = null;
	vcol.visit(right);
	if(vcol.found == null || vcol.found2 != null)
	    return null;
	LocalVariable var2 = vcol.found;
	// to be a join, the 2 variables must be different and the comparison
	// must be equality or order comparison, but not inequality.
	if(var1 == var2 || comp.test == ValueNeOp.TEST)
	    return null;
	// variable types must be Node:
	if(!Type.NODE.accepts(var1.getType().getItemType()) ||
	   !Type.NODE.accepts(var2.getType().getItemType()))
	    return null;
	// we have a comparison between two expressions which each have one 'for'
	// variable reference.
	// Find which var corresponds to the innermost loop,
	// normalize so that the innermost is rights
	JoinInfo info = new JoinInfo();
	info.outerVar = var1; info.innerVar = var2;
	info.outerExpr = left; info.innerExpr = right;
	LocalVariable var2in = var2;
	for(; var2in != null && var2in != var1; ) 
	    var2in = var2in.before;
	if(var2in == null) {	// ie var1 after var2
	    comp.args[0] = right;
	    comp.args[1] = left;
	    comp.test = comp.test.reverse();

	    info.outerVar = var2; info.innerVar = var1;
	    info.outerExpr = right; info.innerExpr = left;
	}
	info.comparison = comp.test;
	// determine the type of keys in the join table:
	ItemType type1 = left.getType().getItemType(),
	         type2 = right.getType().getItemType();
	
	if(Type.NUMERIC.isSuperType(type1) || Type.NUMERIC.isSuperType(type2))
	    info.type = Type.NUMERIC;
	else if(Type.STRING.accepts(type1) && Type.STRING.accepts(type2))
	    info.type = Type.STRING;
	return info;
    }

    // collects info about the join
    public static class JoinInfo {
	LocalVariable outerVar;	// variable of outer loop
	LocalVariable innerVar;	// variable of inner loop
	Expression outerExpr;	// comparison member containing outer var
	Expression innerExpr;	// comparison member containing inner var
	Expression remainder;	// non optimised remainder of a predicate
	Comparison.Test comparison;
	Type  type;	// type of keys in join table (NUMERIC, STRING or default)
    }

    // in a simple FOR, try to transform a WHERE into a predicate of the source expr: 
    // 	for $x in EXPR where pred($x) return $x
    // 	==>  for $x in EXPR [ pred(.) ] return $x
    //
    static FLWRExpr tryToRemoveWhere( FLWRExpr flower ) {
	if(flower.clauses.length != 1 || flower.where == null || flower.orderSpecs != null)
	    return flower;	// 
	VarClause clause = flower.getClause(0);
	if( !(clause instanceof ForClause) )
	    return flower;
	// is the source a path ?
	Expression src = clause.expr;
	if(src instanceof NodeSortExpr)
	    src = ((NodeSortExpr) src).expr;
	if(!(src instanceof PathExpr))
	    return flower;	// fail
	LocalVariable loopVar = clause.varDecl;
	// does the where depend on loop variable ? but no ref inside a predicate!
	Expr.DottingVarFinder finder = new Expr.DottingVarFinder(loopVar);
	if(!finder.visit(flower.where))
	    return flower;
	// replace loopVar by null (i.e '.') in WHERE:
	new Expr.VarReplacer(loopVar, null).visit(flower.where);
	clause.expr = Expr.addPredicate( clause.expr, flower.where );
	flower.where = null;
	return flower;
    }

    // Flower cutter. 
    // 	for $x in EXPR return $x  ==>  EXPR
    // 	This can likely happen after other transformations (or with dumb XQ programmers!).
    static Expression tryToRemoveLoop( FLWRExpr flower ) {
	if(flower.clauses.length != 1 || flower.orderSpecs != null || flower.where != null)
	    return flower;	// fails: no change
	VarClause clause = flower.getClause(0);
	if( !(clause instanceof ForClause) ||
	    !(flower.expr instanceof VarReference.Local) )
	    return flower;
	LocalVariable loopVar = clause.varDecl;
	VarReference.Local ret = (VarReference.Local) flower.expr;
	if(ret.decl != loopVar)
	    return flower;
	return clause.expr;
    }

    // -----------------------------------------------------------------------------

    public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	// pipe the iterators of the different clauses:
	VarClause.Sequence src = new VarClause.Sequence(focus, context); // dummy
	for(int c = 0; c < clauses.length; c ++) { 
	    VarClause.Sequence newSrc =
		(VarClause.Sequence) clauses[c].eval( focus, context );
	    newSrc.setSource(src);
	    src = newSrc;
	}
	// sequence of the Flower itself (performs 'where')
	if(orderSpecs == null) {
	    ItemType itemType = expr.getType().getItemType();
	    int occ = expr.getType().getOccurrence();
	    // optimizable by specialized sequences:
	    if(itemType == Type.INTEGER && !Type.isRepeatable(occ))
		return new IntSequence( src, focus, context );
	    else if(itemType == Type.STRING && !Type.isRepeatable(occ))
		return new StringSequence( src, focus, context );
	    else if( occ == Type.ONE_OCC)
		return new ItemSequence( src, focus, context );
	}
	// general case:
        Value v = new Sequence( src, focus, context );
	if(orderSpecs != null)
	    return sorted(v, focus, context);
	return v;
    }

    public void evalAsEvents( XMLEventReceiver output, Focus focus, EvalContext context)
	throws XQueryException, DataModelException {
	context.at(this);
	// if it is sorted, there is no gain to recursively evalAsEvents: use std method
	if( orderSpecs != null) {
	    super.evalAsEvents( output, focus, context );
	    return;
	}

	VarClause.Sequence src = new VarClause.Sequence(focus, context); // dummy
	for(int c = 0; c < clauses.length; c ++) { 
	    VarClause.Sequence newSrc =
		(VarClause.Sequence) clauses[c].eval( focus, context );
	    newSrc.setSource(src);
	    src = newSrc;
	}

	for( ; src.next(); ) {
	    if( where != null && !where.evalEffectiveBooleanValue( focus, context ) )
		continue;
	    context.at(expr);	// for timeout
	    expr.evalAsEvents( output, focus, context );
	}
    }

    public class Sequence extends GenericValue
    {
	Focus focus;
	EvalContext context;
	Value source;	// innermost clause
	Value current;	// evaluated 'return' expression

	Sequence( Value source, Focus focus, EvalContext context ) {
	    this.source = source;
	    this.focus = focus;
	    this.context = context;
	    current = Sequence.empty;
	}

	public Value  bornAgain() {
	    return new Sequence( source.bornAgain(), focus, context );
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		//context.at(expr);  // for timeout
		if(current.next()) {	// inline 'return' expression {
		    if(!lazy)
			item = current.asItem();
		    return true;
		}
		for (;;) {
		    if(!source.next())
			return false;
		    if( where == null || where.evalEffectiveBooleanValue(focus, context) )
			break;
		}
		current = expr.eval( focus, context );
	    }
	}
    }

    public class ItemSequence extends GenericValue
    {
	Focus focus;
	EvalContext context;
	Value source;	// innermost clause

	ItemSequence( Value source, Focus focus, EvalContext context ) {
	    this.source = source;
	    this.focus = focus;
	    this.context = context;
	}

	public Value  bornAgain() {
	    return new ItemSequence( source.bornAgain(), focus, context );
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		if(!source.next())
		    return false;
		if( where == null || where.evalEffectiveBooleanValue(focus, context) )
		    break;
	    }
	    context.at(expr);  // for timeout
	    // expr evaluates as a single item:
	    item = expr.evalAsItem( focus, context );
	    return true;
	}

	// Qizx:
	public boolean nextCollection() throws XQueryException {
	    for (;;) {
		if(!source.nextCollection())
		    return false;
		if( where == null || where.evalEffectiveBooleanValue(focus, context) )
		    break;
	    }
	    // expr evaluates as a single itemegr item:
	    item = expr.evalAsItem( focus, context );
	    return true;
	}
    }

    public class IntSequence extends IntegerValue
    {
	Focus focus;
	EvalContext context;
	Value source;	// innermost clause
	long  curItem;

	IntSequence( Value source, Focus focus, EvalContext context ) {
	    this.source = source;
	    this.focus = focus;
	    this.context = context;
	}

	public Value  bornAgain() {
	    return new IntSequence( source.bornAgain(), focus, context );
	}

	public long asInteger() {
	    return curItem;
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		for (;;) {
		    if(!source.next())
			return false;
		    if( where == null || where.evalEffectiveBooleanValue(focus, context) )
			break;
		}
		context.at(expr);  // for timeout
		try {
		    // expr evaluates as a single integer item: (empty allowed)
		    curItem = expr.evalAsOptInteger( focus, context );
		    
		    return true;
		} catch(EmptyException e) { }
	    }
	}
    }

    public class StringSequence extends StringValue
    {
	Focus focus;
	EvalContext context;
	Value source;	// innermost clause
	String  curItem;

	StringSequence( Value source, Focus focus, EvalContext context ) {
	    this.source = source;
	    this.focus = focus;
	    this.context = context;
	}

	public Value  bornAgain() {
	    return new StringSequence( source.bornAgain(), focus, context );
	}

	public String asString() {
	    return curItem;
	}

	public boolean next() throws XQueryException {
	    for (;;) {
		for (;;) {
		    if(!source.next())
			return false;
		    if( where == null || where.evalEffectiveBooleanValue(focus, context) )
			break;
		}
		context.at(expr);  // for timeout
		try {
		    // expr evaluates as a single String item: (empty allowed)
		    curItem = expr.evalAsOptString( focus, context );
		    return true;
		} catch(EmptyException e) { }
	    }
	}
    }

    // acquire all items from the source and sort them in an array.
    // make each tuple - formed with the return value and the K sort keys - as an array
    // of size 1+K, gather the tuples into the big array, then sort the array and return it.
    private ArraySequence sorted( Value source, Focus focus, EvalContext context )
	throws XQueryException {
	Object[] items = new Object[8];
	int size = 0;
	int K = orderSpecs.length;
	
	for (; source.next(); ) {
	    Item[] tuple = new Item[K + 1];
	    tuple[0] = source.asItem();
	    
	    for(int k = 0; k < K; k++) {
		Value key = orderSpecs[k].key.eval(focus, context );
		if(!key.next())
		    tuple[k+1] = null;  // empty
		else {
		    tuple[k+1] = key.asAtom();
		    if(key.next())
			context.error(orderSpecs[k], "order key must be atomic");
		}
	    }
	    // add to items:
	    if(size >= items.length) {
		Object[] old = items;
		items = new Object[ old.length * 2 ];
		System.arraycopy(old, 0, items, 0, old.length);
	    }
	    items[ size ++ ] = tuple;	
	}
	try {
	    if(size > 1)
		Arrays.sort( items, 0, size,
			     new KeyComparison( orderSpecs, context.getImplicitTimezone()));
	}
	catch( RuntimeException e ) {
	    if(e.getCause() instanceof XQueryException)
		throw (XQueryException) e.getCause();
	    throw e;
	}
	
	// replace tuples by the main item:
	for(int v = size; --v >= 0; ) {
	    Item[] tuple = (Item[]) items[v];
	    items[v] = tuple[0];
	}
	return new ArraySequence( items, size );
    }

    // CAUTION : necessary for thread-safe operation
    static class KeyComparison implements java.util.Comparator {

	OrderSpec[] orderSpecs;
	int keyCnt;
	int timezone;

	KeyComparison(OrderSpec[] orderSpecs, int timezone) {
	    this.orderSpecs = orderSpecs;
	    this.keyCnt = orderSpecs.length;
	    this.timezone = timezone;
	}

	public int compare(Object o1, Object o2) {
	    try {
		Item[] tuple1 = (Item[]) o1, tuple2 = (Item[]) o2;
		int cmp = 0;	
		for(int k = 1; k <= keyCnt; k++) {
		    cmp = orderSpecs[k-1].compare(tuple1[k], tuple2[k], timezone);
		    if(cmp != 0)
			break;
		}
		return cmp;
	    }
	    catch( TypeException e ) {	
		throw new RuntimeException(e);	// cannot change this interface
	    }
	}
    }
}
