/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.dt.GenericValue;
import net.xfra.qizxopen.xquery.dt.SingleWrappedObject;
import net.xfra.qizxopen.xquery.dt.IntegerValue;
import net.xfra.qizxopen.xquery.dt.StringValue;

import java.util.Arrays;
import java.util.Comparator;

/**
 *	Wrapper for join computation and retrieval.
 */
public class Join
{
    /**
     *	Builds a join hash table from a node sequence and an expression
     *	relative to each node (i.e. computed with the node as focus).
     *	Value of a LET expression inserted before the outer loop of the join.
     */
    static class Maker extends Expression
    {
	Expression source;
	Expression key;
	LocalVariable tmpVar;	
	Type keyType;

	Maker(Expression source, Expression key, LocalVariable tmpVar, Type keyType) {
	    this.source = source;
	    this.key = key;
	    this.tmpVar = tmpVar;
	    this.keyType = keyType;
	}

	public Expression child(int rank) {
	    return null;	//TODO?
	}

	public Value eval( Focus focus, EvalContext context )
	  throws XQueryException {
	    Value src = source.eval(focus, context);
	    long t0 = System.currentTimeMillis();
	    
	    Table table = null;
	    for(; src.next(); ) {
		Node curNode = src.asNode();
		// store to tmp variable:
		context.storeLocal(tmpVar.address, src, true);
		Value keys = key.eval(focus, context);
		
		if(keyType == Type.NUMERIC) {
		    if(table == null)
			table = new NTable();
		    NTable tbl = (NTable) table;
		    for(; keys.next(); )
			tbl.put( keys.asDouble(), curNode );
		}
		else if(keyType == Type.STRING) {
		    if(table == null)
			table = new STable();
		    STable tbl = (STable) table;
		    for(; keys.next(); )
			tbl.put( keys.asString(), curNode);
		}
		else {
		    if(table == null)
			table = new ITable();
		    ITable tbl = (ITable) table;
		    for(; keys.next(); )
			tbl.put(keys.asItem(), curNode);
		}
	    }
	    
	    return new SingleWrappedObject(table);
	}

	public void dump( ExprDump d ) {
	    d.header( this, "MakeJoin" );
	    d.display("source", source);
	    d.display("key", key);
	}
    }

    /**
     *	Gets a node sequence from a key.
     *	References the key expression to evaluate, and
     *	the local variable containing the join table built outside the loop.
     */
    static public class GetExpr extends Expression
    {
	LocalVariable joinVar;
	Expression key;
	Comparison.Test mode;
	
	GetExpr(Expression key, LocalVariable joinVar, Comparison.Test mode) {
	    this.key = key;
	    this.joinVar = joinVar;
	    this.mode = mode;
	    this.type = Type.NODE.star;
	}

	public Expression child(int rank) {
	    return rank == 0 ? key : null;
	}

	public void dump( ExprDump d ) {
	    d.header( this, "getJoin" );
	    d.display("join", joinVar.address); 
	    d.display("comp", mode.toString());
	    d.display("key", key);
	}

	public Value eval( Focus focus, EvalContext context )
	    throws XQueryException {

	    // retrieve table from joinVar:
	    Item tbl = context.loadLocalItem( joinVar.address );
	    Table table = (Table) ((SingleWrappedObject) tbl).getObject();
	    NodeSortExpr.Sequence seq = null;
	    // evaluate key:
	    int keyCnt = 0;
	    Value kval = key.eval(focus, context);
	    for( ; kval.next(); ) {
		Table.Entry entry = null;
		
		if(table instanceof STable)
		    entry = ((STable) table).findEntries(kval.asString(), mode);
		else if(table instanceof NTable)
		    entry = ((NTable) table).findEntries(kval.asDouble(), mode);
		else
		    entry = ((ITable) table).findEntries(kval.asItem(), mode);

		for(; entry != null; entry = table.nextEntry() ) {
		    // really something: create an array sequence (maybe to sort) if needed
		    
		    ++ keyCnt;
		    if(seq == null)
			seq = new NodeSortExpr.Sequence(entry.nodes, entry.nodeCount);
		    else {
			seq.addItems(entry.nodes, entry.nodeCount);
		    }
		}
	    }
	    if(keyCnt == 1 && seq != null)
		seq.needsSort = false;
	    return seq == null ? Value.empty : seq;
	}
    } 

    // base class for String, Numeric, Generic implementations
    public static class Table extends HTable
    {
	Entry[] keys;	// sorted keys
	Comparator comparator;
	// iteration:
	int index, lastIndex;

	static class Entry extends HTable.Key {
	    Node[] nodes;
 	    int nodeCount;

	    Entry(Node node) {
		nodes = new Node[] { node, null };
		nodeCount = 1;
	    }

	    void add( Node node ) {
		if(nodeCount >= nodes.length) {
		    Node[] old = nodes;
		    nodes = new Node[ old.length * 2 ];
		    System.arraycopy(old, 0, nodes, 0, old.length);
		}
		nodes[nodeCount++] = node;
	    }

	    public Node[] getNodes( ) {
		if(nodeCount == nodes.length)
		    return nodes;
		// need to copy:
		Node[] nodes = new Node[ nodeCount ];
		System.arraycopy(this.nodes, 0, nodes, 0, nodeCount);
		return nodes;
	    }

	    public HTable.Key duplicate() {	//not needed
		System.err.println("Key.duplicate not needed"); return null;
	    }
	}

	Entry nextEntry() {
	    
	    return (index >= lastIndex)? null : keys[index++];
	}

	void sortedKeys() {
	    keys = new Entry[count];
	    for(int s = 0, h = hash.length; --h >= 0; )
		for( HTable.Key c = hash[h]; c != null; c = c.next)		
		    keys[s++] = (Entry) c;
	    Arrays.sort(keys, comparator);	//TODO collation
	    
	}

	Entry findEntries( Entry probe, Comparison.Test mode) {
	    index = lastIndex = 0;
	    if(mode == ValueEqOp.TEST)
		return (Entry) get(probe);
	    int ix = locate(probe);	// binary search: 
	    
	    if(mode == ValueGtOp.TEST) {  // ie keys < probe
		if(ix < keys.length && comparator.compare(keys[ix], probe) == 0)
		    -- ix;
		lastIndex = ix;
	    }
	    else if(mode == ValueGeOp.TEST) {  // ie keys <= probe
		if(ix < keys.length && comparator.compare(keys[ix], probe) == 0)
		    ++ ix;
		lastIndex = ix;
	    }
	    else if(mode == ValueLtOp.TEST) {  // ie keys > probe
		if(ix < keys.length && comparator.compare(keys[ix], probe) == 0)
		    ++ ix;
		index = ix;
		lastIndex = keys.length;
	    }
	    else {	// Le:  // ie keys >= probe
		index = ix;
		lastIndex = keys.length;
	    }
	    
	    Entry res = 
		index < Math.min(lastIndex, keys.length) ? keys[index++] : null;
	    
	    return res;
	}

	// returns the position of the smallest entry >= probe
	int locate(Entry probe) {
	    if(keys == null) 
		sortedKeys();
	    int lo = 0, hi = keys.length - 1;
	    while(lo <= hi) {
		int mid = (lo + hi) / 2;
		int cmp = comparator.compare(probe, keys[mid]);
		if (cmp < 0)
		    hi = mid - 1;
		else if (cmp > 0)
		    lo = mid + 1;
		else
		    return mid;	// found
	    }
	    return lo;
	}
    }   

    //
    //	String implementation
    //
    public static class STable extends Table
    {
	Entry probe = new Entry("", null);

	STable() {
	    comparator = new Comparator() {
		    public int compare(Object o1, Object o2) {
			Entry e1 = (Entry) o1, e2 = (Entry) o2;
			return e1.key.compareTo(e2.key);
		    }
		};
	}

	public static class Entry extends Table.Entry {
	    String key;

	    Entry(String key, Node node) {
		super(node);
		this.key = key;
	    }

	    public int hashCode() {
		return key.hashCode();
	    }

	    public boolean equals( Object that ) {
		Entry other = (Entry) that;
		return other.key.equals(key);
	    }

	    public String toString() {
		return "STRING KEY "+key+" nodes: "+nodeCount;
	    }
	}

	public void put( String key, Node node ) {
	    probe.key = key;
	    Entry entry = (Entry) get(probe);
	    if(entry != null) 
		entry.add(node);
	    else
		directPut(new Entry(key, node));
	    keys = null;
	}

	Table.Entry findEntries( String key, Comparison.Test mode) {
	    probe.key = key;
	    return findEntries(probe, mode);
	}
    } // end of class STable

    //
    //	Numeric implementation
    //
    public static class NTable extends Table
    {
	Entry probe = new Entry(0, null);

	NTable() {
	    comparator = new Comparator() {
		    public int compare(Object o1, Object o2) {
			Entry e1 = (Entry) o1, e2 = (Entry) o2;
			return Util.comparison(e1.key - e2.key);
		    }
		};
	}

	public static class Entry extends Table.Entry {
	    double key;

	    Entry(double key, Node node) {
		super(node);
		this.key = key;
	    }

	    public int hashCode() {
		long h = Double.doubleToRawLongBits(key);
		return (int) (h ^ (h >>> 32));
	    }

	    public boolean equals( Object that ) {
		Entry other = (Entry) that;
		return other.key == key;
	    }

	    public String toString() {
		return "NUMERIC KEY "+key+" nodes: "+nodeCount;
	    }
	}

	public void put( double key, Node node ) {
	    probe.key = key;
	    Entry entry = (Entry) get(probe);
	    if(entry != null) 
		entry.add(node);
	    else
		directPut(new Entry(key, node));
	}

	Table.Entry findEntries( double key, Comparison.Test mode) {
	    probe.key = key;
	    return findEntries(probe, mode);
	}
    }

    //
    //	General implementation
    //
    public static class ITable extends Table
    {
	Entry probe = new Entry(null, null);

	ITable() {
	    comparator = new Comparator() {
		    public int compare(Object o1, Object o2) {
			try {
			    Entry e1 = (Entry) o1, e2 = (Entry) o2;
			    return e1.key.compareTo(e2.key, null, 0);
			}
			catch (Exception e) {
			    return 0;	// whatever
			}
		    }
		};
	}

	static class Entry extends Table.Entry {
	    Item key;

	    Entry(Item key, Node node) {
		super(node);
		this.key = key;
	    }

	    public int hashCode() {
		
		return key.hashCode();
	    }

	    public boolean equals( Object that ) {
		Entry other = (Entry) that;
		return other.key.equals(key);
	    }

	    public String toString() {
		return "ITEM KEY "+key+" nodes: "+nodeCount;
	    }
	}

	public void put( Item key, Node node ) {
	    probe.key = key;
	    Entry entry = (Entry) get(probe);
	    if(entry != null) 
		entry.add(node);
	    else
		directPut(new Entry(key, node));
	}

	Table.Entry findEntries( Item key, Comparison.Test mode) {
	    probe.key = key;
	    return findEntries(probe, mode);
	}
     } 
}
