/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.dt.SingleDecimal;
import net.xfra.qizxopen.xquery.dt.Conversion;
import net.xfra.qizxopen.xquery.fn.Prototype;

/**
 *   Implementation of operator '*'. 
 * 
 */
public class MulOp extends NumericOp {

    public MulOp( Expression expr1, Expression expr2 ) {
        super( expr1, expr2 );
    }

    static Prototype[] protos = { 
        Prototype.op("*", Type.DOUBLE.opt, ExecD.class).hidden()
	  .arg("op1", Type.DOUBLE.opt).arg("op2", Type.UNTYPED_ATOMIC.opt),
        Prototype.op("*", Type.DOUBLE.opt, ExecD.class).hidden()
	  .arg("op1", Type.UNTYPED_ATOMIC.opt).arg("op2", Type.DOUBLE.opt),
        Prototype.op("*", Type.INTEGER.opt, ExecI.class)
	  .arg("op1", Type.INTEGER.opt).arg("op2", Type.INTEGER.opt),
        Prototype.op("*", Type.DECIMAL.opt, ExecDec.class)
	  .arg("op1", Type.DECIMAL.opt).arg("op2", Type.DECIMAL.opt),
        Prototype.op("*", Type.FLOAT.opt, ExecF.class)
	  .arg("op1", Type.FLOAT.opt).arg("op2", Type.FLOAT.opt),
        Prototype.op("*", Type.DOUBLE.opt, ExecD.class)
	  .arg("op1", Type.DOUBLE.opt).arg("op2", Type.DOUBLE.opt),
    };
    public Prototype[] getProtos() { return protos; }

    public void dump( ExprDump d ) {
	d.header( this, "Op *" );
        d.display("expr1", operands[0]);
        d.display("expr2", operands[1]);
    }

    public static class ExecDec extends Function.Call {
	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    Value e1 = args[0].eval(focus, context);
	    Value e2 = args[1].eval(focus, context);
	    context.at(this);
	    return new SingleDecimal( e1.asDecimal().multiply( e2.asDecimal() ));
	}
    }

    public static class ExecI extends Function.OptIntegerCall {

	public long evalAsOptInteger( Focus focus, EvalContext context )
	    throws XQueryException {
	    long e1 = args[0].evalAsOptInteger(focus, context);
	    long e2 = args[1].evalAsOptInteger(focus, context);
	    context.at(this);
	    if(!Conversion.isIntegerRange( e1 * (double) e2 ))
		context.error(this, "integer overflow");
	    return e1 * e2;
	}
    }

    public static class ExecF extends Function.OptFloatCall {
	public float evalAsOptFloat( Focus focus, EvalContext context )
	    throws XQueryException {
	    float e1 = args[0].evalAsOptFloat(focus, context);
	    float e2 = args[1].evalAsOptFloat(focus, context);
	    context.at(this);
	    
	    return e1 * e2;
	}
    }

    public static class ExecD extends Function.OptDoubleCall {
	public double evalAsOptDouble( Focus focus, EvalContext context )
	    throws XQueryException {
	    double e1 = args[0].evalAsOptDouble(focus, context);
	    double e2 = args[1].evalAsOptDouble(focus, context);
	    context.at(this);
	    
	    return e1 * e2;
	}
    }
}
