/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dt.ArraySequence;
import java.util.Arrays;

/**
 *	Sorts a Node Sequence in document order without duplicates.
 */
public class NodeSortExpr extends Expression
{
    public Expression expr;

    public NodeSortExpr( Expression expr ) {
	this.expr = expr;
	type = expr.getType();
    }

    public Expression child(int rank) {
	return (rank == 0)? expr : null;
    }

    public void dump( ExprDump d ) {
	d.header( this, "Node-Sort" );
        d.display("expr", expr);
    }

    public int getFlags() {
	return DOCUMENT_ORDER;	// that's the purpose
    }

    public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	Value v = expr.eval(focus, context);
	return new Sequence( v );
    }

    public static class Sequence extends ArraySequence
    {
	public boolean needsSort = true;

	public Sequence( Value source ) throws XQueryException {
	    super(new Item[16], 0);
	    for (; source.next(); ) {
		addItem(source.asNode());
	    }
	    if(size == 0)
		return;
	}

	public Sequence(Node[] nodes, int count) {	// for reusing
	    super(nodes, count);
	}

	public void sort() {
	    pack(); // if necessary
	    Arrays.sort( items, 0, size, new java.util.Comparator() {
		    public int compare(Object o1, Object o2) {
			Node n1 = (Node) o1, n2 = (Node) o2;
			return n1.orderCompare(n2);
		    }
		});
	    // remove duplicates:
	    Object previous = items[0];
	    int nsize = Math.min(size, 1);
	    for(int i = 1; i < size; i++) {
		Object node = items[i];
		if(!node.equals(previous)) {
		    previous = node;
		    items[nsize ++] = node;
		}
	    }
	    size = nsize;
	    needsSort = false;
	}

	public boolean next() throws XQueryException {
	    // automatic packing:
	    if(needsSort || overflow != null)
		sort();
	    return super.next();
	}

	public long quickCount(EvalContext context) throws XQueryException {
	    // no need to pack (assumes no duplication)
	    if(needsSort)
		sort();
	    return super.quickCount(context);
	}

	public Value  bornAgain() {
	    // what is done is done and not to do again: use a simple array sequence
	    return new ArraySequence(items, size);
	}
    }
}
