package jp.sourceforge.armadillo;

import java.io.*;
import java.nio.channels.*;

import jp.sourceforge.armadillo.io.*;
import jp.sourceforge.armadillo.zip.*;

/**
 * ZipA[JCoB
 */
public final class ZipArchiver implements Archiver {

    private ZipOutputStream zos;
    private boolean includesDirectory;

    /**
     * ZipArchiver̐B
     * @param os OutputStream
     */
    public ZipArchiver(OutputStream os) {
        this.zos = new ZipOutputStream(os);
    }

    /**
     * includesDirectory̐ݒB
     * @param includesDirectory includesDirectory
     */
    public void setIncludesDirectory(boolean includesDirectory) {
        this.includesDirectory = includesDirectory;
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.File)
     */
    public void addEntry(ArchiveEntry ae, File file) throws IOException {
        final long fileSize = file.length();
        if (file.isDirectory()) {
            if (!includesDirectory) {
                return;
            }
            addDirectoryEntry(ae);
        } else {
            ae.setSize(fileSize);
            if (fileSize > 0) {
                RandomAccessFile r = new RandomAccessFile(file, "r");
                try {
                    FileChannel fch = r.getChannel();
                    InputStream is = Channels.newInputStream(fch);
                    ArchiveEntry trialEntry = new ArchiveEntry(ae.getName());
                    tryOut(trialEntry, is);
                    ae.setCrc(trialEntry.getCrc());
                    long deflatedSize = trialEntry.getCompressedSize();
                    fch.position(0L);
                    if (deflatedSize < fileSize) {
                        ae.setCompressedSize(deflatedSize);
                    } else {
                        ae.setCompressedSize(fileSize);
                    }
                    addFileEntry(ae, is);
                } finally {
                    r.close();
                }
            } else {
                addFileEntry(ae, null);
            }
        }
        ae.setAdded(true);
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.InputStream, long)
     */
    public void addEntry(ArchiveEntry ae, InputStream is, long length) throws IOException {
        if (ae.isDirectory()) {
            if (!includesDirectory) {
                return;
            }
            addDirectoryEntry(ae);
        } else {
            ae.setSize(length);
            addFileEntry(ae, is);
        }
        ae.setAdded(true);
    }

    /**
     * fBNgGg̒ǉB
     * @param ae ArchiveEntry
     * @throws IOException o̓G[ꍇ
     */
    private void addDirectoryEntry(ArchiveEntry ae) throws IOException {
        ZipEntry entry = new ZipEntry(ae.getName());
        entry.setMethod(ZipEntry.STORED);
        entry.setCrc(0L);
        entry.setSize(0L);
        entry.setCompressedSize(0L);
        entry.setLastModified(ae.getLastModified());
        zos.putNextEntry(entry);
        zos.closeEntry();
        ae.setCrc(0L);
        ae.setSize(0L);
        ae.setCompressedSize(0L);
    }

    /**
     * t@CGg̒ǉB
     * @param ae ArchiveEntry
     * @param is InputStream (o̓TCY0̏ꍇ <code>null</code> e)
     * @throws IOException o̓G[ꍇ
     */
    private void addFileEntry(ArchiveEntry ae, InputStream is) throws IOException {
        final long fileSize = ae.getSize();
        final long compressedSize = ae.getCompressedSize();
        ZipEntry entry = new ZipEntry(ae.getName());
        entry.setLastModified(ae.getLastModified());
        if (compressedSize <= 0) {
            entry.setOption((short)8); // EXTwb_
        }
        if (compressedSize > 0 && compressedSize < fileSize) {
            entry.setMethod(ZipEntry.DEFLATED);
        } else {
            entry.setMethod(ZipEntry.STORED);
        }
        entry.setCrc(ae.getCrc());
        entry.setSize(fileSize);
        entry.setCompressedSize(ae.getCompressedSize());
        zos.putNextEntry(entry);
        long size = 0;
        if (fileSize > 0) {
            size = IOUtilities.transferAll(is, zos);
        }
        zos.closeEntry();
        assert size == fileSize : "file size";
        assert entry.getSize() == fileSize : "file size";
        assert entry.getCompressedSize() == compressedSize : "comp size";
        ae.setCrc(entry.getCrc());
        ae.setSize(size);
        ae.setCompressedSize(compressedSize);
    }

    /**
     * ksB
     * @param ae ArchiveEntry
     * @param is InputStream
     * @throws IOException o̓G[ꍇ
     */
    private static void tryOut(ArchiveEntry ae, InputStream is) throws IOException {
        ZipEntry entry = new ZipEntry();
        SizeDetectionOutputStream detector = new SizeDetectionOutputStream();
        ZipOutputStream zos = new ZipOutputStream(detector);
        try {
            entry.setOption((short)8);
            entry.setMethod(ZipEntry.DEFLATED);
            zos.putNextEntry(entry);
            IOUtilities.transferAll(is, zos);
            zos.closeEntry();
        } finally {
            zos.close();
        }
        assert entry.getCompressedSize() == detector.getSize();
        ae.setCrc(entry.getCrc());
        ae.setSize(entry.getSize());
        ae.setCompressedSize(entry.getCompressedSize());
    }

    /* (overridden)
     * @see java.io.Closeable#close()
     */
    public void close() throws IOException {
        zos.close();
    }

}
