<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: Anno.php 97 2006-04-23 10:05:16Z sakamoto $
 */

/**
 * アノテーションの情報を保持するクラス
 * このクラスを直接、コンストラクトすることはありません。
 * 大抵は、AutoSOAP_Anno::parseReflector(Reflector);ですむはずです。
 * 
 * @package jp.servlet.AutoSOAP.Anno
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Anno {
    private $commnet;
    private $parameters = array();

    /**
     * コメントをセットします。
     * 
     * @param string $comment 
     */
    public function setComment($commnet)
    {
        $this->commnet = $commnet;
    } 

    /**
     * コメントを取得します。
     */
    public function getComment()
    {
        return $this->commnet;
    } 

    /**
     * アノテーションラインを追加します。
     * 
     * @param AutoSOAP_Anno_Line $param
     */
    public function appendLine(AutoSOAP_Anno_Line $param)
    {
        $this->parameters[] = $param;
    } 

    /**
     * コマンドを実行します。
     * 
     * @param AutoSOAP_Anno_Command $exec コマンドオブジェクト
     */
    public function execute(AutoSOAP_Anno_Command $exec)
    {
        foreach($this->parameters as $parameter) {
            $parameter->accept($exec);
        } 
        return $exec->getResult();
    } 

    /**
     * リフレクションオブジェクトのgetDocComment()を利用して、アノテーション情報を取得します。
     * 
     * @static 
     * @param Reflector $ref リフレクションオブジェクト
     */
    static public function parseReflector(Reflector $ref)
    {
        $engine = AutoSOAP_Anno_ParseEngine :: getInstance();
        return $engine->parse($ref->getDocComment());
    } 
} 

/**
 * アノテーション実行コマンドインターフェース
 * 
 * @package jp.servlet.AutoSOAP.Anno
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
interface AutoSOAP_Anno_Command {
    /**
     * アノテーションコマンド処理するコマンド名を返す
     * 
     * @access public 
     */
    public function getCode();
    /**
     * アノテーションコマンド処理した結果を返します
     * 
     * @access public 
     */
    public function getResult();
    /**
     * アノテーションコマンド処理を実装します
     * 
     * @access public 
     * @param string $value アノテーションコマンドの対する文字列
     */
    public function execute($value);
} 

/**
 * アノテーションライン
 * 
 * @package jp.servlet.AutoSOAP.Anno
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Anno_Line
{
    private $key;
    private $value;

    /**
     * コンストラクタ
     * 
     * @access public 
     * @param string $key アノテーションコマンド名
     * @param string $value アノテーションコマンドの対する文字列
     */
    public function __construct($key, $value)
    {
        $this->key = $key;
        $this->value = trim($value);
    } 

    /**
     * アノテーションコマンドの処理を実行する為のコマンドのを受け口(Visitorパターン)
     * 
     * @access public 
     * @param AutoSOAP_Anno_Command $command 処理を実行するアノテーションコマンド
     */
    public function accept(AutoSOAP_Anno_Command $command)
    {
        $codes = $command->getCode();
				if(is_array($codes)) {
					if(in_array($this->key, $codes)) {
						$command->execute($this->value);
					}
				} else if ($this->key === $codes) {
            $command->execute($this->value);
        } 
    } 
} 

/**
 * アノテーションが書かれた文字列をパースします。
 * Singletonパターンが適応されています。
 * 
 * @package jp.servlet.AutoSOAP.Anno
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Anno_ParseEngine {
    private static $instance;

    /**
     * コンストラクタ
     * 
     * @access private 
     */
    private function __construct()
    {
    } 

    /**
     * インスタンの取得
     * 
     * @access public 
     */
    public static function getInstance()
    {
        if (is_null(self :: $instance)) {
            $class = __CLASS__;
            self :: $instance = new $class();
        } 
        return self :: $instance;
    } 

    /**
     * 文字列をパース
     * 
     * @access public 
     * @param string $doc アノテーションが書かれた文字列
     */
    public function parse($doc)
    {
        $annotation = new AutoSOAP_Anno();
        $args = array();
        preg_match_all("/^[ \t]*\*+[ \t^\/]+(\S.*)/m", $doc, $args, PREG_SET_ORDER);
        $p_key = '';
        $p_val = '';
        foreach($args as $lines) {
            $matches = array();
            if (!isset($lines[1])) {
            	continue;
            } else if (preg_match("/^(#+)?@([^\s]+)[\s]*(.*)/", $lines[1], $matches)) {
                if (''!=$matches[1]) {
                	continue;
                }
                $this->appendData($annotation, $p_key, $p_val);
                $p_key = $matches[2];
                $p_val = $matches[3];
            } else {
                $p_val .= $lines[1];
            } 
        } 
        $this->appendData($annotation, $p_key, $p_val);

        return $annotation;
    } 

    /**
     * アノテーション情報をClass $annotationへ追加していきます。
     * 
     * @access private 
     * @param AutoSOAP_Anno $annotation アノテーション情報を保持するクラス
     * @param string $p_key アノテーションコマンド名
     * @param string $p_val アノテーションコマンドの対する文字列
     */
    private function appendData(&$annotation, $p_key, $p_val)
    {
        if ('' != $p_val) {
            if ('' != $p_key) {
                $annotation->appendLine(new AutoSOAP_Anno_Line($p_key, $p_val));
            } else {
                $annotation->setComment($p_val);
            } 
        } 
    } 
} 

?>