<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: Server.class.php 104 2006-04-25 15:06:20Z sakamoto $
 */

/**
 * AutoSOAPサーバ本体
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Server {
    /**
     * 提供されるサービスの情報
     * 
     * @var AutoSOAP_Service
     */
    private $service;

    /**
     * ロギング制御クラス
     * 
     * @var AutoSOAP_Logger
     */
    private $logger;

    /**
     * 変換後のエンコード
     * 
     * @var string 
     */
    private $to_encoding = null;

    /**
     * 変換前のエンコード
     * 
     * @var string 
     */
    private $from_encoding = null;

    /**
     * SoapServerオブジェクト
     * 
     * @var SoapServer 
     */
    private $server;

    /**
     * メソッド"schemaValidate"のエラーメッセージ
     * 
     * @var string 
     */
    private static $error_schemaValidate;

    /**
     * フィルター
     * 
     * @var array 
     */
    private static $filters;

    /**
     * インスタンスオブジェクト
     * 
     * @var AutoSOAP_Server
     */
    private static $instance;

    /**
     * コンストラクタ
     * 
     * @access private 
     * @return void 
     */
    private function __construct()
    {
        $this -> filters = array();
    } 

    /**
     * インスタンの取得
     * 
     * @access public 
     * @return AutoSOAP_Server
     */
    static public function getInstance()
    {
        if (is_null(self :: $instance)) {
            $class = __CLASS__;
            self :: $instance = new $class();
            set_error_handler(array(self :: $instance, "soap_error_handler"));
        } 
        return self :: $instance;
    } 

    /**
     * SOAPメッセージを検証する。
     * 
     * @static 
     * @access private 
     * @param string $soap_message 
     * @param string $xsd_file 
     * @return boolean 
     */
    static private function schemaValidate(&$soap_message, $xsd_file, $error_msg)
    {
        $aDOM = new DOMDocument('1.0', 'UTF-8');
        $aDOM -> loadXML(preg_replace("/[^\s\:\"\']+\:encodingStyle=[\"\'][^\"\'\s]*[\"\']/i", "", substr($soap_message,0,1024), 1).substr($soap_message,1024));

        $old = set_error_handler(array(self :: $instance, "schemaValidate_error_handler"));
        $result = $aDOM -> schemaValidate($xsd_file);
        set_error_handler($old);

        if (false === $result) {
            if (is_null($error_msg)) {
                $error_msg = self :: $error_schemaValidate;
            } 
            trigger_error($error_msg, E_USER_ERROR);
            return false;
        } 
        return true;
    } 

    /**
     * リクエストされたメソッド名を返す
     * 失敗した場合は、NULLを返す
     * 
     * @static 
     * @access private 
     * @param string $soap_message 
     * @param string $namespace 
     * @return boolean 
     */
    static private function getSoapMethodName(&$soap_message, $namespace)
    {
        $xml = simplexml_load_string($soap_message);
        if (false === $xml) {
            trigger_error("Not found the Soap Message.", E_USER_ERROR);
            return null;
        } 

        $path = "/*[local-name() = 'Envelope']/*[local-name() = 'Body']";
        $path .= "/*[namespace-uri() = '" . $namespace . "'][position()=1]";

        $result = $xml -> xpath($path);

        if (0 < count($result)) {
            $args = array();
            if (preg_match("/^<([^: \/\"'>]*:)?([^: \/\"'>]+)/", $result[0] -> asXML(), $args)) {
                if (isset($args[2]))
                    return $args[2];
            } 
        } 
        trigger_error("Not found the Soap Method or The namespace is wrong.", E_USER_ERROR);
        return null;
    } 

    /**
     * ロギングクラスをセットする
     * 
     * @access public 
     * @param AutoSOAP_Logger $logger
     * @return boolean 
     */
    public function setLogger(AutoSOAP_Logger $logger)
    {
        $this -> logger = $logger;
    } 

    /**
     * サービスをセットする
     * 
     * @access public 
     * @param AutoSOAP_Service $service
     * @param integer $load_mode 
     * @return boolean 
     */
    public function setService(AutoSOAP_Service $service)
    {
        $this -> service = $service; 
        // サービスクラスのインクルード
        $this -> service -> include_class();

        /**
         * WSDLとXSDファイルを生成する
         */
        $timestamp = $this -> service -> getTimestamp();
        if ($this -> service -> need_update_wsdl_file()) {
            $this -> make_wsdl_file($timestamp);
        } 
        if ($this -> service -> need_update_xsd_file()) {
            $this -> make_xsd_file($timestamp);
        } 
    } 

    /**
     * サービスの実行
     * 
     * @access public 
     * @return boolean 
     */
    public function execute($request = null)
    {
        if (is_null($request)) {
            global $HTTP_RAW_POST_DATA;
            $request = &$HTTP_RAW_POST_DATA;
        } 
        // サーバ初期化
        $this -> server = null; 
        // メソッドの存在のチェック
        $web_method = self :: getSoapMethodName($request, $this -> service -> getNamespace());
        if (is_null($web_method)) {
            return false;
        } 
        // フィルター実行
        foreach($this -> filters as $filter) {
            // グローバル変数を編み込む
            AutoSOAP_Global::knitForClass(get_class($filter));

            $filter -> execute($this -> service, $web_method);
        } 
  
        // 妥当性のチェック
        if (!is_null($this -> service -> getXsdPath())) {
            $error_msg = $this -> getValidateError($web_method);
            if (false === self :: schemaValidate($request, $this -> service -> getXsdPath(), $error_msg)) {
                return false;
            } 
        } 
        // サービスクラスのインクルード
        $this -> service -> include_class(); 
        // SOAPサーバ生成
        $this -> server = new SoapServer($this -> service -> getWsdlPath(),
            array('soap_version' => SOAP_1_1,
                'classmap' => $this -> getClassmap()
                )
            ); 
        // クラス名取得
        $class_name = $this -> service -> getClassName(); 
        // グローバル変数を編み込む
        AutoSOAP_Global::knitForClass($class_name);

        // サービスの実行
        $this -> server -> setClass($class_name);
        $this -> server -> handle($request); 
        // サーバリセット
        $this -> server = null;

        return true;
    } 

    /**
     * WSDLファイルを表示する
     * 
     * @access public 
     * @return void 
     */
    public function wsdl()
    {
        // header("Content-Type: text/xml; charset=UTF-8");
        header("Content-Type: text/xml;");
        echo file_get_contents($this -> service -> getWsdlPath());
    } 

    /**
     * WSDLファイルを作成する
     * 
     * @access private
     * @return boolean 
     */
    private function make_wsdl_file($timestamp)
    {
        include_once(dirname(__FILE__) . "/class2soap.php");
        $engine = new AutoSOAP_class2wsdl($this -> service -> getClassName(),
            $this -> service -> getNamespace(),
            $this -> service -> getLocation());

        $result = file_put_contents($this -> service -> getWsdlPath(), $engine -> getResult($this -> to_encoding, $this -> from_encoding));
        if ($result && !is_null($timestamp)) {
            touch($this -> service -> getWsdlPath(), $timestamp);
        }
        
        // その他の情報も生成しておく
        $content = $engine -> exportOptions();
        $result_c = file_put_contents($this -> service -> getOptionPath(), $content);
        if ($result_c && !is_null($timestamp)) {
            touch($this -> service -> getOptionPath(), $timestamp);
        } 

        return $result;
    } 

    /**
     * XSDファイルを作成する
     * 
     * @access private 
     * @return boolean 
     */
    private function make_xsd_file($timestamp)
    {
        if (is_null($this -> service -> getXsdPath())) {
            return false;
        } 

        include_once(dirname(__FILE__) . "/class2soap.php");
        $base_dir = str_replace("\\", "/", dirname(__FILE__));
        $soapenv_uri = $base_dir . '/xsd/soapenv.xsd';
        $soapenc_uri = $base_dir . '/xsd/soapenc.xsd';

        $engine = new AutoSOAP_class2SoapXsd($this -> service -> getClassName(),
            $this -> service -> getNamespace(),
            $soapenv_uri,
            $soapenc_uri);

        $result = file_put_contents($this -> service -> getXsdPath(), $engine -> getResult($this -> to_encoding, $this -> from_encoding));
        if ($result && !is_null($timestamp)) {
            touch($this -> service -> getXsdPath(), $timestamp);
        } 

        return $result;
    } 

    /**
     * クラスマップを取得する
     * 
     * @access public 
     * @return array 
     */
    private function getClassmap()
    {
        include($this -> service -> getOptionPath());
        return (isset($_classmap))?$_classmap : array();
    } 

    /**
     * 検証時のエラーメッセージを取得する
     * 
     * @access public 
     * @return array 
     */
    private function getValidateError($method)
    {
        include($this -> service -> getOptionPath());
        return (isset($_validate_errors[$method]))?$_validate_errors[$method] : null;
    } 

    /**
     * エンコードを設定する
     * 
     * @access public 
     * @return void 
     */
    public function setEncoding($to_encoding = null, $from_encoding = null)
    {
        $this -> to_encoding = $to_encoding;
        $this -> from_encoding = $from_encoding;
    } 

    /**
     * フィルターをセットする
     * 
     * @access public 
     * @param AutoSOAP_Filter $filter
     */
    public function appendFilter(AutoSOAP_Filter $filter)
    {
        $this -> filters[] = $filter;
    } 

    /**
     * 現在の情報をかえす
     * 
     * @access public 
     * @param AutoSOAP_Filter $filter
     */
    public function getInfo()
    {
        return get_object_vars($this);
    } 

    /**
     * SOAP用エラーハンドリング
     * 
     * @access public 
     * @return void 
     */
    public function soap_error_handler($errno, $errstr, $errfile, $errline)
    { 
        // E_STRICTは無視する
        if (E_STRICT === $errno) {
            return;
        } 

        $result = false; 
        // ログ出力
        if ($this -> logger instanceof AutoSOAP_Logger) {
            $result = $this -> logger -> output($this -> service, $errno, $errstr, $errfile, $errline);
        } 
        // TRUEだった場合、SOAPエラーメッセージは出力しない。
        if (true === $result) {
            return ;
        } 

        // E_NOTICE or E_USER_NOTICEだった場合、SOAPエラーメッセージは出力しない。
        if (E_NOTICE === $errno || E_USER_NOTICE === $errno) {
            return;
        } 

        // SOAPエラーメッセージを出力
        $errortype = array (
            E_ERROR => "Error",
            E_WARNING => "Warning",
            E_PARSE => "Parsing Error",
            E_NOTICE => "Notice",
            E_CORE_ERROR => "Core Error",
            E_CORE_WARNING => "Core Warning",
            E_COMPILE_ERROR => "Compile Error",
            E_COMPILE_WARNING => "Compile Warning",
            E_USER_ERROR => "User Error",
            E_USER_WARNING => "User Warning",
            E_USER_NOTICE => "User Notice",
            E_STRICT => "Runtime Notice"
            );
        $mssg = $errortype[$errno] . ': ' . $errstr;
        $debug = debug_backtrace();
        $faultactor = "\n";
        foreach($debug as $key => $val) {
            $faultactor .= '#' . $key . ' ' . $val['class'] . $val['type'] . $val['function'] . '()' . ' called at [' . $val['file'] . ':' . $val['line'] . "]\n";
        } 
        if (is_null($this -> server)) {
            $this -> server = new SoapServer(null, array('soap_version' => SOAP_1_1, 'uri' => "http://localhost/"));
            $this -> server -> fault("Server", $mssg, null, $faultactor);
            $this -> server = null;
        } else {
            throw new SoapFault("Server", $mssg, null, $faultactor);
        } 
    } 

    /**
     * schemaValidate用SOAP用エラーハンドリング
     * 
     * @access public 
     * @return void 
     */
    public function schemaValidate_error_handler($errno, $errstr, $errfile, $errline)
    {
        self :: $error_schemaValidate = $errstr;
    } 
} 

?>