<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: Framework.class.php 119 2006-04-27 09:43:28Z sakamoto $
 */

include_once(dirname(__FILE__)."/AutoSOAP.php");
include_once(dirname(__FILE__)."/ServiceFactory.class.php");

/**
 * AutoSOAPフレームワーク
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Framework {
	const FACTORY_ORGINAL  = 'original';
	const FACTORY_AUTOSOAP = 'autosoap';
	
    /**
     * サーバインスタンス
     * @var AutoSOAP_Server
     */
    private $_server;
    
    /**
     * サービス生成ファクトリー
     * @var array
     */
    private $_factories;
    
    /**
     * ベースディレクトリー
     * @var string
     */
    private $_base_dir;
    
    /**
     * エイリアス
     * @var array
     */
    private $_alias;
    
    /**
     * タイムスタンプ(Y/m/d H:i:s)
     * @var string
     */
    private $_timestamp;
    
    /**
     * バリテーションモード
     * @var boolean
     */
    private $_validate_mode;
    
    /**
     * キャッシュファイルのネームヘッド
     * @var string
     */
    private $_cash_namehead;
    
    /**
     * インスタンスオブジェクト
     * @var AutoSOAP_Framework
     */
    private static $instance;

    /**
     * インスタンの取得
     * 
     * @access public 
     * @return AutoSOAP_Framework
     */
    static public function getInstance($base_dir=null)
    {
        if (is_null(self :: $instance) && !is_null($base_dir)) {
            $class = __CLASS__;
            self :: $instance = new $class($base_dir);
        } 
        return self :: $instance;
    } 

    /**
     * コンストラクタ
     * @param string $base_dir
     */
    private function __construct($base_dir)
    {
        /*
         * 設定
         */
        $this->_base_dir = realpath($base_dir);
        if (false === $this->_base_dir) {
            trigger_error("Please specify the existing directory for '\$base_dir' of ".__CLASS__.'::'.__FUNCTION__."().",E_USER_ERROR);
            return;
        }
        $this->_base_dir .= '/';
        $this->_alias     = null;
        $this->_timestamp = null;
        $this->_factories = null;
        $this->_validate_mode = true;
        $this->_cash_namehead = '';
        
        /*
         * SOAPサーバ生成
         */
        $this->_server = AutoSOAP_Server::getInstance();
        $this->_server->setEncoding("UTF-8", null);
    }

    /**
     * 情報を出力する
     */
    static public function info()
    {
        include(dirname(__FILE__)."/Framework/TestTool.class.php");
        AutoSOAP_Framework_TestTool::outputHTML();
    }

    /**
     * SOAPサービスの実行
     * @param string $location
     * @param string $service_name
     */
    public function execute($location=null, $service_name=null)
    {
        if (!is_null($location) && !is_string($location)) {
            trigger_error("Argument '\$location' of ".__CLASS__.'::'.__FUNCTION__."() should be a character string.",E_USER_WARNING);
            return;
        }
        
        if (!is_null($service_name) && !is_string($service_name)) {
            trigger_error("Argument '\$service_name' of ".__CLASS__.'::'.__FUNCTION__."() should be a character string.",E_USER_WARNING);
            return;
        }

        $factory = self::FACTORY_ORGINAL;
        if (is_null($service_name)) {
            $service_name = $_SERVER["QUERY_STRING"];
        }
        if (is_null($location)) {
            $flg_https = false;
            if(isset($_SERVER["HTTPS"])) {
                $flg_https = ('on'==$_SERVER["HTTPS"]);
            }
            $location = (($flg_https)?"https://":"http://") . $_SERVER["SERVER_NAME"] . $_SERVER["SCRIPT_NAME"] . '?';
        }
        
        $wsdl_mode    = false;
        $matches      = array();
        if (preg_match("/^(.+)@wsdl$/", $service_name, $matches)) {
            $service_name = $matches[1];
            $wsdl_mode = true;
        }
        
        $location_query = $service_name;
        
        // エイリアスを適用する
        if (is_array($this->_alias)) {
            foreach ($this->_alias as $pattern => $replace) {
                if ('/'===substr($pattern,0,1)) {
                    if (preg_match($pattern, $service_name)) {
                        $service_name = preg_replace($pattern, $replace, $service_name);
                        break;
                    }
                } else if ($pattern===$service_name) {
                    $service_name = $replace;
                    break;
                }
            }
        }

        if (0==strlen($service_name)) {
            trigger_error("Service was not found.",E_USER_ERROR);
            return;
        }
        
        //AutoSOAPのライブラリーからサービスを提供する
        $matches = array();
        if (preg_match("/^autosoap\.services\.(.*)/i",$service_name,$matches)) {
            $service_name = 'AutoSOAP.Services.'.$matches[1];
            $factory = self::FACTORY_AUTOSOAP;
        }
        
        /*
         * サービス生成
         */
        $service = $this->getFctory($factory)->getService($service_name, ($location . $location_query), $this->_timestamp, $this->_cash_namehead);
        
        /*
         * SOAPサービスをサーバへ設定する
         */
        $this->_server->setService($service);
        
        if ($wsdl_mode) {
            /*
             * WSDLを返す
             */
            $this->_server->wsdl();
        } else {
            /*
             * SOAPサービス実行
             */
            $this->_server->execute();
        }
    }    

    /**
     * ロギングクラスをセットする
     * 
     * @access public 
     * @param AutoSOAP_Logger $logger
     * @return void 
     */
    public function setLogger(AutoSOAP_Logger $logger)
    {
        $this->_server->setLogger($logger);
    } 

    /**
     * エイリアスをセットする
     * 
     * @access public 
     * @param array
     * @return void 
     */
    public function setAlias($alias)
    {
        $this->_alias = $alias;
    } 

    /**
     * タイムスタンプをセットする
     * 
     * @access public 
     * @param string (Y/m/d H:i:s)
     * @return void 
     */
    public function setTimestamp($timestamp)
    {
        $this->_timestamp = $timestamp;
    } 

    /**
     * キャッシュファイルのネームヘッドをセットする
     * 
     * @access public 
     * @return void 
     */
    public function setCashNamehead($namehead)
    {
        $this->_cash_namehead = $namehead;
    } 

    /**
     * フィルターを追加する
     * @param string $filter_name
     */
    public function appendFilter($filter_name)
    {
        include_once($this->_base_dir.'filters/'.$filter_name.".filter.php");
        $this->_server->appendFilter(new $filter_name());
    }

    /**
     * AutoSOAPフィルター郡からフィルターを追加する
     * @param string $filter_name
     */
    public function appendAutoSOAPFilter($filter_name)
    {
        $class_name = 'AutoSOAP_Filters_'.$filter_name;
        include_once(dirname(__FILE__).'/Filters/'.$filter_name.".filter.php");
        $this->_server->appendFilter(new $class_name());
    }

    /**
     * データ保存ディレクトリーを返す
     */
    public function getVarDir()
    {
        return $this->_base_dir."vars/";
    }
    
    /**
     * フレームワークの情報を返す
     */
    public function getInfo()
    {
    	return get_object_vars($this);
    }
    
    /**
     * バリテーションモードを切り替える
     * @param boolean $mode
     */
    public function modeValidate($mode)
    {
    	$this->_validate_mode = (boolean)$mode;
    	$this->getFctory(self::FACTORY_ORGINAL, true);
    }
    
    /**
     * サービスファクトリーの取得
     * @param string $namespace
     * @param boolean $recreate
     */
    private function getFctory($namespace,$recreate=false)
    {
    	if (is_null($this->_factories) || true==$recreate) {
    		$this->_factories = array();
    		
	        $services_dir1  = $this->_base_dir.'services';
	        $services_dir2  = dirname(dirname(__FILE__));
	        $wsdls_dir      = $this->_base_dir.'wsdls';
	        $xsds_dir       = ($this->_validate_mode) ? $this->_base_dir.'validate' : null ;
	        $options_dir    = $this->_base_dir.'options';
	        
	        $this->_factories[self::FACTORY_ORGINAL] = new AutoSOAP_ServiceFactory(
	                            $services_dir1,
	                            $wsdls_dir,
	                            $xsds_dir,
	                            $options_dir
	                       );
	
	        $this->_factories[self::FACTORY_AUTOSOAP] = new AutoSOAP_ServiceFactory(
	                            $services_dir2,
	                            $wsdls_dir,
	                            $xsds_dir,
	                            $options_dir
	                       );
	    		
    	}
    	return $this->_factories[$namespace];
    }
    
    /**
     * セットアップ
     */
    public function setup()
    {
        restore_error_handler();
        $flg_succeed = true;
        $extensions = array(
                    'mbstring',
                    'soap',
        );
        $dirs = array(
                    '',
                    'services',
                    'wsdls',
                    'validate',
                    'options',
                    'filters',
                    'vars',
        );
        //エクステンションチェック
        echo "<font color=blue><b>Check loaded extensions...</b></font>"."<br>\n";
        foreach ($extensions as $extension) {
            echo "&nbsp;&nbsp;check {$extension} extension...";
            if (extension_loaded($extension)) {
                echo "<font color=blue>OK</font>";
            } else {
                $flg_succeed = false;
                echo "<font color=red>not loaded!! Please load <b>{$extension}</b> extension.</font>";
            }
            echo "<br>\n";
        }
        if ($flg_succeed) {
            echo "<font color=blue><b>succeeded.</b></font>"."<br><br>\n";
        } else {
            echo "<font color=red><b>Please load Extensions!!</b></font>"."<br><br>\n";
            exit;
        }

        //ディレクトリー生成とチェック
        echo "<font color=blue><b>Start AutoSOAP Framework Setup...</b></font>"."<br>\n";
        foreach ($dirs as $name) {
            $dir = $this->_base_dir.$name.'/';
            $mssg_dir = '"'.$dir.'"';
            $name = (''===$name) ? 'Base Directory' : "dir:{$name}";

            echo "&nbsp;&nbsp;check {$name}...";
            if (true === file_exists($dir)) {
                if (is_file($dir)) {
                    $flg_succeed = false;
                    echo "<br>\n&nbsp;&nbsp;&nbsp;&nbsp;";
                    echo "<font color=red>This is file. Please make directory.</font>";
                } else {
                    echo "ok";
                }
            } else {
                echo "<font color=blue>create...</font>";
                if (false === @mkdir($dir)) {
                    $flg_succeed = false;
                    echo "<br>\n&nbsp;&nbsp;&nbsp;";
                    echo "<font color=red>The directory was not able to be made. Please change the permission of \"{$this->_base_dir}\" to 777.</font>";
                } else {
                    echo "ok";
                }
            }
            echo "<br>\n";
        }
        if ($flg_succeed) {
            echo "<font color=blue><b>The setup of AutoSOAP Framework succeeded. </b></font>"."<br>\n";
        } else {
            echo "<font color=red><b>The setup of AutoSOAP Framework failed. </b></font>"."<br>\n";
        }
    }
}
?>
