/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import org.jetbrains.annotations.NotNull;
import plus.runtime.BuiltInVar;
import plus.runtime.RunHelper;
import plus.util.NumHelper;

import java.util.concurrent.atomic.AtomicStampedReference;

/**
 * Atomic レコード ($).
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author Kunio Himei.
 */
@SuppressWarnings("unused")
public final class Record implements ArrayAccessor<Object, Object> {

    private static final String[] EMPTY_STRING_ARRAY = {}; // 空の文字列配列.
    /**
     * 原子的に更新可能な オブジェクト参照と整数スタンプ.
     */
    private final AtomicStampedReference<Object[]> atom =
            new AtomicStampedReference<>(EMPTY_STRING_ARRAY, 0);

    /**
     * $0 -> フィールド.
     */
    private static Object[] rebuild(Object r0) {
        String x = RunHelper.toString(null, r0); // 数値を文字列に変換
        String[] a = RunHelper.split2Array(x, null);
        int len = a.length;
        Object[] arr = new Object[len + 1];
        for (int i = 0; len > i; i++)
            arr[i + 1] = NumHelper.toStringNumber(a[i]);
        arr[0] = NumHelper.toStringNumber(r0); // $0
        return arr;
    }

    /**
     * フィールド -> $0.
     */
    private static Object rebuild(Object[] arr) {
        int len = arr.length;
        if (1 < len) { // $(1..NF)
            String ofs = BuiltInVar.OFS.toString(); // 出力フィールドセパレータ (既定値は空白)
            String convfmt = BuiltInVar.CONVFMT.toString(); // 数値から文字列への変換書式 (既定値は %.6g)
            StringBuilder sb = new StringBuilder();
            for (int i = 1; len > i; i++) {
                String x = RunHelper.toString(convfmt, arr[i]);
                sb.append(ofs).append(x);
            }
            return NumHelper.toStringNumber(sb.toString()); // $0
        }
        return "";
    }

    //* この配列にアクセスするキーを返す.
    private static int arrayKey(Object index) {
        if (index instanceof Number)
            return ((Number) index).intValue();
        return NumHelper.intValue(index);
    }

    /**
     * 指定された欄の値を返す.
     */
    private synchronized Object get(int index) { // SYNC.
        int[] holder = new int[1];
        while (true) {
            Object[] record = this.atom.get(holder); // Record Array
            int reclen = record.length;
            int len = 1 + Math.max(BuiltInVar.NF.intValue(), 0); // フィールド数
            if ((0 == index) && // $0 で欄が変更されていれば
                    ((0 != holder[0]) || (reclen != len))) {
                Object[] arr = new Object[len];
                System.arraycopy(record, 0, arr, 0, Math.min(reclen, len));
                arr[0] = rebuild(arr); // $0 を再構築
                if (this.atom.compareAndSet(record, arr, holder[0], 0)) {
                    return arr[0]; // $0
                }
            } else {
                return (reclen > index) ? record[index] : "";
            }
        }
    }

    /**
     * 指定された要素を置換する.
     */
    private void put(final int index, @NotNull final Object value) {
        int[] holder = new int[1];
        while (true) {
            Object[] record = this.atom.get(holder); // Record Array
            int isModRecord = 0; // フィールド変数(処理済み)
            Object[] arr;
            if (0 == index) {
                arr = rebuild(value); // Field を再構築
            } else {
                int len = 1 + Math.max(BuiltInVar.NF.intValue(), 0); // フィールド数
                arr = new Object[Math.max(len, index + 1)];
                System.arraycopy(record, 0, arr, 0,
                        Math.min(record.length, arr.length));
                isModRecord = 1; // フィールド変数(変更した)
                arr[index] = value;
            }
            if (this.atom.compareAndSet(record, arr, holder[0], isModRecord)) {
                int nf = -1;
                while (nf != BuiltInVar.NF.intValue()) { // NFが変更された.
                    nf = this.atom.getReference().length - 1;
                    BuiltInVar.NF.put(nf);
                }
                return;
            }
        }
    }

    @Override
    public String toString() {
        return get(0).toString();
    }

    //* ------------------------------------------------------------------------
    //* Getter / Setter - for Groovy.
    //* ------------------------------------------------------------------------
    @Override
    public Object getAt(final Object index) {
        return get(arrayKey(index));
    }

    @SuppressWarnings("UnusedReturnValue")
    @Override
    public void putAt(final Object index, final Object value) {
        Object x = NumHelper.toStringNumber(value);
        put(arrayKey(index), x);
    }
}