/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Writer;
import java.nio.charset.Charset;

/**
 * Writer -> OutputStream. - 2022.4
 * <p>
 * 標準 out/err のエンコードが切り替わらないため作成.
 * Created because the standard out / err encoding does not switch.
 *
 * @author kunio himei.
 */
public class StreamWriter extends Writer {

    private final String charsetName;
    private final OutputStream out;
    private final PrintStream print;

    StreamWriter(OutputStream out, String charsetName) {
        this.charsetName = charsetName;
        this.out = out;
        this.print = null;
    }

    StreamWriter(OutputStream out) {
        this.charsetName = Charset.defaultCharset().name(); // NOTE Java18 UTF-8
//        this.charsetName = StandardCharsets.UTF_8.name();
        this.out = out;
        this.print = null;
    }

    StreamWriter(PrintStream out) {
        this.charsetName = null;
        this.out = null;
        this.print = out;
    }

    // Closes the stream, flushing it first.
    @Override
    public void close() throws IOException {
        if (null != out) this.out.close();
        else print.close();
    }

    // Flushes the stream.
    @Override
    public void flush() throws IOException {
        if (null != out) this.out.flush();
        else print.flush();
    }

    // Writes byte[].
    private void put(byte[] b) throws IOException {
        this.out.write(b);
    }

    // Writes a string.
    @Override
    public void write(@NotNull String s) throws IOException {
        if (null != out) this.put(s.getBytes(charsetName));
        else print.print(s);
    }

    // Writes a portion of an array of characters.
    @Override
    public void write(@NotNull char[] buf, int off, int len) throws IOException {
        this.write(String.valueOf(buf, off, len));
    }

    // Writes the specified byte to this output stream.
    @Override
    public void write(int b) throws IOException {
        this.put(new byte[(byte) b]);
    }
}