/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import plus.util.NumHelper;

import java.util.ArrayList;
import java.util.Collection;

/**
 * Parser - Action Expression.
 *
 * @author kunio himei.
 */
abstract class Expression extends Function {

    //* expression
    //* `= += -= *= /= %= **='   Assignment. These operators group right-to-left
    //* orop
    //* `?:'        Conditional. This operator groups right-to-left
    //* `||'        Logical "or"
    //* `&&'        Logical "and"
    //* relativeop
    //* in`         Array membershi
    //* `~ !~'      RegExp Matching, non-matching
    //* `< <= == != > >= >> | |& !== ==='   Relational, and redirection
    //* concatop
    //* `Concatenation` No special token is used to indicate concatenation
    //* addop
    //* `+ -'       Addition, subtraction
    //* multiop
    //* `* / %'     Multiplication, division, modulus
    //* unaryop
    //* `+ - !'     Unary plus, minus, logical "not"
    //* powerop
    //* `^ **'      Exponentiation. These operators group right-to-left
    //* incdecop
    //* `++ --'     Increment, decrement
    //* term
    //* `$'         Field
    //* `(...)'     Grouping

    private static final LexRegx
            rxADDOP = new LexRegx("^([-+])$"); // NOTE 加減算

    private static final LexRegx
            rxMULTIOP = new LexRegx("^([/%]|[*]{1,2})$"); // NOTE 乗算

    private static final LexRegx
            rxPOWEROP = new LexRegx("^([*]{2})$"); // NOTE べき乗

    private static final LexRegx
            rxRELATIVEOP = new LexRegx("^([!=]={1,2}|[<>]=?|[!]?~)$"); // NOTE 比較演算子

    private static final LexRegx
            rxANDOP = new LexRegx("^(&{1,2})$"); // NOTE 論理積

    private static final LexRegx
            rxOROP = new LexRegx("^([|]{1,2})$"); // NOTE 論理和

    private static final LexRegx
            rxREDIRECT = new LexRegx("^(<|>{1,2}|[|]&)"); // NOTE リダイレクト

    //* NOTE ビット否定('~'NOT)は、と正規表現適合で競合するため'`~'とする.
    private static final LexRegx
            rxEXPR = new LexRegx("^(" + // NOTE expression
            "([-+/%]|[*]{1,2})|" + // 算術演算子
            "(<<|>{2,3})|" + // シフト演算子
            "[&|^]|`~|" + // ビット演算子 REMIND '`~'ビット否定
            ")=$"); // <- 最後は、'='

    private static final LexRegx rxCONCATOP = new LexRegx("^([($]|[+][+]|--)$");
    //* ^[ \t]*getLine.
    private static final LexRegx rxHasGETLINE = new LexRegx("^\\s*getline");
    //* ^[<|>;)}\n].
    private static final LexRegx rxREADLINE = new LexRegx("^[<|>;)}\n]");
    //* ^[-+!]$.
    private static final LexRegx rxUNARYOP = new LexRegx("^[-+!]$");
    //* 再代入を許可する.
    int enableReAssign;
    //* global定義か?.
    boolean isGlobalDef;
    //* 現在の '(' のネストを記憶.
    private int cPARENNEST;
    //* リダイレクト識別子.
    private boolean isGETLINESTAT;
    //* リダイレクト, 強制文字列化抑制.
    private boolean isPRINTSTAT;

    /**
     * 配列の添え字オブジェクトを返す.
     */
    private static Object index(Object[] e) {
        if (0 == e.length) {
            return e;
        } else if (1 < e.length) {
            return new Node.Call(Node.INDEX_OP, Type.castWrapArray(e),
                    Flags.T07STR);
        } else {
            int typ = Type.getNodeType(e[0]);
            if (Flags.isNumber(typ) && !Flags.isStrNum(typ)) {
                if (e[0] instanceof Term.NUMBER x) {
                    if (0 == (x.doubleValue() - (int) x.doubleValue())) {
                        return e;
                    }
                    return new Node.Call(Node.INDEX_OP, Type.castWrapArray(e),
                            Flags.T07STR);
                }
                return new Node.Call(Node.INDEX_OP, Type.castWrapArray(e),
                        Flags.T07STR);
            }
            return e;
        }
    }

    /**
     * 空の NUMBER かどうかを返す.
     */
    private static boolean isEmptyNumber(Object[] a) {
        return (1 <= a.length) && (a[0] instanceof Term.NUMBER)
                && ((Term.NUMBER) a[0]).id.isEmpty();
    }

    /**
     * 加減算.
     */
    private Object addOp() {
        Object e = multiop();
        while (rxADDOP.find(super.tok)) { // `+ -' Addition, subtraction
            String op = rxADDOP.group(1);
            eat(super.tok);
            Object x = multiop();
            if ((x instanceof Term.NUMBER) && "0".equals(((Term.NUMBER) x).id)) {
                if (Flags.isNumber(Type.getNodeType(e))) {
                    yyOPTIMIZE("addOp: " + e + " (" + op + " 0)"); // オプティマイズ(e [-+] 0)
                } else {
                    e = new Node.IncDec(Node.NUMBER_OP, e); // 数値正規化
                }
            } else {
                if ((e instanceof Term.NUMBER)
                        && "0".equals(((Term.NUMBER) e).id) && "+".equals(op)) {
                    if (Flags.isNumber(Type.getNodeType(x))) {
                        yyOPTIMIZE("addOp: (0 +) " + x); // オプティマイズ(0+ x)
                        e = x;
                    } else {
                        e = new Node.IncDec(Node.NUMBER_OP, x); // オプティマイズ(数値化)
                    }
                } else {
                    e = new Node.Calc(op, e, x);
                }
            }
        }
        return e;
    }

    /**
     *
     */
    private Object andOp() {
        Object e = relativeop();
        while (rxANDOP.find(super.tok)) { // '& &&'
            String op = rxANDOP.group(1);
            eat(super.tok);
            e = new Node.Comp(op, genCompare(e), genCompare(relativeop()));
        }
        return e;
    }

    /**
     * assignOp を返す.
     * <p>
     * {@literal IllegalStateException} - val 変数への再代入
     */
    private Node.Ass assignOp(Keyword id, Node.YyVariable left,
                              String sType) {
        String name = left.name;
        String op = "=".equals(super.tok) ? "=" : super.tok.toString()
                .replaceFirst("=$", ""); // NOTE '='をここで削除 ☆
        eat(super.tok);
        nl();

        if ((super.tok instanceof Node.YyVariable) && // 右辺
                ((Keyword.SymVAL == id) || (Keyword.SymVAR == id))) {
            if (!isGlobalDef) {
                Symbols.setLocalDefMode(true); // local定義開始 var left = NAME, Arr
            }
            Symbols.createType(((Node.YyVariable) super.tok).name);
            if (!isGlobalDef) {
                Symbols.setLocalDefMode(false); // local定義終了
            }
        }
        Object ex = orop(); // `(' or value
        if (Keyword.SymVAL != id) { // val への再代入チェック
            if ((Keyword.SymVAR == id) && (0 != enableReAssign)) {
                Symbols.resetLocalType(name); // ローカル変数をリセット(for var)
            }
            int typ = Symbols.getType(name);
            if ((Flags.T26NIL != typ) && (0 != (typ & Flags.T19VALTYPE))
                    && (0 == (typ & Flags.T21FORVAR))) {
                throw new IllegalStateException("ERROR: reassignment to val `"
                        + name + '`'); // val 変数への再代入
            }
        }
        int vType;
        if ((0 != enableReAssign) && (Keyword.SymVAL == id)) {
            vType = Flags.T19VALTYPE | Flags.T21FORVAR;
        } else if ((0 != enableReAssign) && (Keyword.SymVAR == id)) {
            vType = Flags.T21FORVAR;
        } else {
            vType = 0;
        }
        Object right = assignType("VAR " + op, name, ex, Flags.T20ASSIGN | vType);
        if (rxEXPR.find(super.tok)) {
            return new Node.Ass(id, op, left, // 連続代入の途中
                    assignOp(id, (Node.YyVariable) ex, sType),
                    Symbols.getType(name), sType);
        }
        return new Node.Ass(id, op, left, right, Symbols.getType(name), sType); // 単独代入または連続代入の最後
    }

    //* concatOp | getLine
    //* concatOp

    /**
     *
     */
    private Object concatOp() {
        Object e = addOp();
        assert (null != super.tok);
        if (('|' == super.tok.toString().charAt(0))
                && rxHasGETLINE.find(super.yyText)) {
            String op = super.tok.toString();
            eat(super.tok);
            eat(Keyword.SymGETLINE);
            e = getlineStmt(e, op); // 行入力 "command" | getLine [var]
        }
        if (!(e instanceof String) && isConcat()) { // `Concatenation` No special token is used to indicate concatenation
            Collection<Object> buf = new ArrayList<>();
            if (!((e instanceof Term.STRING) && (0 == ((Term.STRING) e).id
                    .length()))) {
                buf.add(e);
            }
            while (isConcat()) {
                Object x = addOp();
                if (!((x instanceof Term.STRING) && (0 == ((Term.STRING) x).id
                        .length()))) {
                    // 行入力 "command" | getLine [var]
                    if (('|' == super.tok.toString().charAt(0))
                            && rxHasGETLINE.find(super.yyText)) {
                        String op = super.tok.toString();
                        eat(super.tok);
                        eat(Keyword.SymGETLINE);
                        buf.add(getlineStmt(x, op)); // 行入力
                    } else {
                        buf.add(x);
                    } //  ;TRACE(e, e.getClass, x, x.getClass)
                }
            }
            e = new Node.Call(Node.CONCAT_OP, buf.toArray(), Flags.T07STR);
        }
        return e;
    }

    /**
     * expression を返す.
     */
    private Object expr(boolean wantColon) {
        if ((Keyword.SymVAR == super.tok) || (Keyword.SymVAL == super.tok)) { // 変数定義
            if (!isGlobalDef) {
                Symbols.setLocalDefMode(true); // local定義開始
            }
            Keyword id = (Keyword) super.tok;
            eat(super.tok);
            Object e = term();
            if (!(e instanceof Node.YyVariable x)) {
                throw new IllegalStateException("unMatch: " + e);
            }
            if (!isGlobalDef) {
                Symbols.setLocalDefMode(false); // local定義終了
            }
            T4Types rr = optType(x.name, false);
            int typ = rr.nType();
            String sType = rr.sType();
            if (!sType.isEmpty()) {
                Symbols.setType(x.name, typ, Flags.T20ASSIGN
                        | ((Keyword.SymVAL == id) ? Flags.T19VALTYPE : 0));
            }
            return assignOp(id, x, sType);
        }
        Object e = orop();
        if (e instanceof Node.YyVariable x) { // 変数名が現れた
            T4Types rr = optType(x.name, wantColon);
            int typ = rr.nType();
            String sType = rr.sType();
            if (rxEXPR.find(super.tok)) {
                if (!sType.isEmpty()) {
                    Symbols.setType(x.name, typ, ("=".equals(super.tok))
                            ? Flags.T20ASSIGN : 0);
                }
                return assignOp(Keyword.SyyASSIGN, x, sType);
            }
            return x;
        }
        return e;
    }

    /**
     * 式 expr [= += -= *= /= %= **= ? || &&] expression を返す.
     */
    @Override
    Object expression() {
        if (Keyword.SymPRINT == super.tok) {
            return printStmt(Keyword.SymPRINT);
        } else if (Keyword.SymPRINTF == super.tok) {
            return printStmt(Keyword.SymPRINTF);
        } else {
            // NOTE 正規表現が単独で出現した場合の例外処理をおこなうこと
            return expr(false);
        }
    }

    /**
     * 因子 factor を返す.
     */
    private Object factor(Node.NAME e) {
        String name = e.name; //  ;TRACE(">factor", e)
        if ("$".equals(name)) { // `$' Field
            boolean gurdeINCDEC; // 後置きinc/decを制限
            if ("(".equals(super.tok)) {
                gurdeINCDEC = false;
            } else {
                super.yyPARENg++;
                gurdeINCDEC = true;
            }
            Node.YyVariable ex = new Node.Arr(name,
                    Type.castWrapArray(nonPostSimpExp()));
            if (gurdeINCDEC) {
                super.yyPARENg--; // 後置きinc/decを許可
            }
            return ("++".equals(super.tok) || "--".equals(super.tok))
                    ? optPostIncdec(ex) : ex; // NOTE 後置きinc/dec
        } else if ("[".equals(super.tok)) { // `NAME'[...]
            Node.YyVariable ex = arrayType(new Node.Arr(name,
                    Type.castWrapArray(term())));
            return ("++".equals(super.tok) || "--".equals(super.tok))
                    ? optPostIncdec(ex) : ex; // NOTE 後置きinc/dec
        } else if ("++".equals(super.tok) || "--".equals(super.tok)) {
            if (Symbols.isClosure(e.name)) {
                super.cCLOSURE++; // Symbols.CLOSURE に登録されている場合 ☆
            }
            return optPostIncdec(e); // 後置きinc/dec
        } else if (("(".equals(super.tok) && !super.yyHasLEFTSPACE) || // 関数呼出し `NAME'(...)
                hasFunction(name)) { // () を省略した関数呼出し (これにより関数名と同一の変数は使用不可となる)
            return callStmt(name);
        } else { // 変数参照
            if (Symbols.isClosure(e.name)) {
                super.cCLOSURE++; // Symbols.CLOSURE に登録されている場合 ☆
            }
            return e;
        }
    }

    /**
     * Function Value.
     */
    private Node.YyCall functionValue(Keyword id) {
        if ("(".equals(super.tok) || "{".equals(super.tok)) {
            Node.Root e = functionDecl();
            if (e instanceof Node.Call x) {
                System.err.println(x.name + ": 無名関数値　-- ここには来ないはず");
                callStmtImpl(getFunctionId(x.name), x.args); // ユーザ関数の呼び出しを記憶する
                return new Node.Invoke(id, "", x.name,
                        x.args, x.nType, "", true);
            }
            throw new IllegalStateException("unmatch: " + e);
        }
        String name = super.tok.toString();
        advance();
        return invokeStmt(id, name);
    }

    /**
     *
     */
    private Object invokeop() {
        Object e = term();
        while (".".equals(super.tok)) { // 後置き`.' インスタンスに対する invoke
            advance();
            String methd = super.tok.toString();
            advance();
            e = invokeStmtImpl(Keyword.SymINVOKE, e, methd); // instance.method
        }
        return e;
    }

    /**
     *
     */
    private boolean isConcat() {
        return ((null != super.tok) && !(super.tok instanceof Keyword)
                && !"in".equals(super.tok) && !(super.tok instanceof String))
                || rxCONCATOP.find(super.tok);
    }

    /**
     *
     */
    private Object multiop() {
        Object e = unaryop();
        super.yyEnableANAREG++; // RegExp 解析許可
        while (rxMULTIOP.find(super.tok)) { // `* / %' Multiplication, division, modulus
            String op = rxMULTIOP.group(1);
            eat(super.tok);
            e = new Node.Calc(op, e, unaryop());
        }
        super.yyEnableANAREG--; // RegExp 解析許可
        return e;
    }

    /**
     *
     */
    private Node.YyCall newStmt() {
        Object e = super.tok;
        eat(super.tok);
        Term.BOXING obj = className(e);
        T4Types rr = optType(e.toString(), false);
        int typ = rr.nType();
        String sType = rr.sType();
        super.yyHasLEFTSPACE = false; // token の空白区切りチェックを迂回
        return new Node.Invoke(Keyword.SyyNEW, obj, "",
                Type.castWrapArray(callArgments()), typ, sType, true);
    }

    /**
     *
     */
    private Object nonPostSimpExp() {
        int sign = 1; // $の添字が+-で始まる場合
        while ("-".equals(super.tok) || "+".equals(super.tok)) {
            if ("-".equals(super.tok)) {
                sign *= -1; // '-'の出現を記憶
            }
            eat(super.tok);
        } // $i, $++i, $i++, $i * 2 もある
        Object e = term(); // 後置きinc/dec var++, -- を制限
        if (0 > sign) {
            e = new Node.Calc("-", Term.NUMBER_ZERO, e); // $-e
        }
        return e;
    }

    /**
     * 後置きinc/dec を返す.
     */
    private Object optPostIncdec(Object e) {
        if ((super.yyPARENg <= super.yyPARENca) && "++".equals(super.tok)) {
            advance();
            return new Node.IncDec("+-", incdecNegateType((Node.YyVariable) e)); // var++
        } else if ((super.yyPARENg <= super.yyPARENca)
                && "--".equals(super.tok)) {
            advance();
            return new Node.IncDec("-+", incdecNegateType((Node.YyVariable) e)); // var--
        } else {
            return e;
        }
    }

    /**
     *
     */
    private Object orop() {
        Object e = andOp();
        if ("?".equals(super.tok)) { // `?:' Conditional. This operator groups right-to-left
            eat(super.tok);
            nl();
            Object e1 = expr(true); // wantColon
            eat(":");
            nl();
            Object e2 = expr(true); // wantColon
            int t1 = getMaskType(e1); // Type check
            int t2 = getMaskType(e2);
            if ((Flags.T11ANY != t1) && (Flags.T11ANY != t2) && (t1 != t2)) {
                yyWARNING("TYPE UNMATCHED: ? " + e1 + ":<"
                        + Integer.toHexString(t1) + "> " + e2 + ":<"
                        + Integer.toHexString(t2) + '>');
            }
            e = new Node.If(Keyword.SyyQIF, genCompare(e),
                    Type.castWrapArray(e1), Type.castWrapArray(e2));
        } else {
            while (rxOROP.find(super.tok)) { // '| ||'
                String op = rxOROP.group(1);
                eat(super.tok);
                e = new Node.Comp(op, genCompare(e), genCompare(andOp()));
            }
        }
        return e;
    }

    /**
     * べき乗.
     */
    private Object powerop() {
        Object e = invokeop();
        while (rxPOWEROP.find(super.tok)) { // '**'
            String op = rxPOWEROP.group(1);
            eat(super.tok);
            e = new Node.Calc(op, e, invokeop());
        }
        return e;
    }

    /**
     * 文関数 print[f] を返す.
     */
    private Node.YyStatement printStmt(Keyword id) {
        cPARENNEST = super.yyPARENc; // 現在の '(' のネストを記憶
        advance(); // print(..) or print ..
        boolean hasParen = "(".equals(super.tok);
        isPRINTSTAT = true; // リダイレクト, 強制文字列化抑制
        Object x;
        if (rxREADLINE.find(super.tok)) {
            x = Function.EMPTY_ARRAY;
        } else if (hasParen) {
            // 'print(' は関数として解釈しない
            x = commalist(); // 文: print
        } else {
            x = optparenlist(); // 文: print ..
        }
        isPRINTSTAT = false; // リダイレクト, 強制文字列化解除
        String rop = super.tok.toString();
        Object[] a = Type.castWrapArray(x);
        if (('>' == rop.charAt(0)) || ('|' == rop.charAt(0))) {
            eat(super.tok);
            return new Node.Print(Keyword.toName(id), a, rop, expr(false)); // リダイレクト(new,append,pipe)
        }
        return new Node.Print(Keyword.toName(id), a, "", ""); // symbol,name
    }

    //*  "date" | get-line date
    //*  for (i = 1; ("ls -t" | get-line v) > 0; i++)
    //*  get-line v
    //*  if (get-line v == 0)
    //*  while (0 < get-line v <f)
    //*  while (get-line v <f > 0)

    /**
     * 文関数 "command" | get-line [term] [&lt;concatop].
     */
    private Node.Getline getlineStmt(Object command, String op) {
        isGETLINESTAT = true; // リダイレクト識別子
        Object v = (rxREADLINE.find(super.tok)) ? Function.EMPTY_ARRAY
                : term();
        isGETLINESTAT = false;
        String rid;
        Object f;

        if ("<".equals(super.tok) || "|&".equals(super.tok)) {
            rid = super.tok.toString();
            eat(super.tok);
            f = term(); // 文字列 (#?条件式,連接などは不可?#)
        } else if (!"".equals(command)) {
            rid = op;
            f = command; // リダイレクト "command" | getLine
        } else {
            rid = "";
            f = "";
        }
        if ((v instanceof Object[]) && (0 == ((Object[]) v).length)) {
            return new Node.Getline(Function.EMPTY_ARRAY, rid, f); // $0
        } else if ((v instanceof Node.Arr) && "$".equals(((Node.Arr) v).name)
                && isEmptyNumber(((Node.Arr) v).index)) {
            return new Node.Getline(Function.EMPTY_ARRAY, rid, f); // $0

        } else if (v instanceof Node.YyVariable x) { // NAME, Arr
            updateType(x, Flags.T07STR, Flags.T20ASSIGN);
            if ("".equals(f)) {
                return new Node.Getline(Type.castWrapArray(x), "", "");
            }
            return new Node.Getline(Type.castWrapArray(x), rid, f);
        }
        throw new IllegalStateException("unmatch: " + v);
    }

    /*
     * ダック・タイピング: アヒルのように歩いたり鳴いたりするものはアヒルであろう.
     * (In Computer Programming, duck)
     * typing is a style of dynamic typing in which an object`s current set of
     * methods and properties determines the valid semantics, rather than its
     * inheritance from a particular class. The name of the concept refers to
     * the duck test, attributed to James Whitcomb Riley, which may be phrased
     * as follows: If it walks like a duck and quacks like a duck, I would call
     * it a duck.
     */

    /**
     *
     */
    private Object relativeop() {
        Object e = concatOp();
        if (Operator.YY_IS.equals(super.tok) || // 'is' instanceOf
                Operator.YY_AS.equals(super.tok)) { // 'as' 強制型変換
            String id = super.tok.toString();
            eat(super.tok);
            Object ex = super.tok;
            eat(super.tok);
            Object obj = className(ex); // クラス名(NAME, BOXING)を取得
            return new Node.Comp(id, Type.castWrapArray(e), obj);

        } else if (Operator.YY_OPIN.equals(super.tok)) { // `in` 配列要素の存在検査
            eat(super.tok);
            Object ex = concatOp();
            if (e instanceof Node.Invoke call) { // '.' callExpression `in` ex (duck typing)
                // className (SimpleClassName は下記の配列検査(in NAME)と被るため指定不可)
                return new Node.Invoke(Keyword.SymHAS, ex, call.name,
                        call.args, Flags.T27DISABLE, "", true);
            }
            if (Flags.isVarArgs(Type.getNodeType(ex))) { // INDEX 'in' variableArgument(可変長引数の存在検査)
                return new Node.Comp(Operator.YY_OPIN, e, ex);
            }
            return new Node.Comp(Operator.YY_OPIN, new Node.Call(Node.INDEX_OP,
                    Type.castWrapArray(e), Flags.T07STR), ex);

        } else {
            Object ex = e; // `~ !~' RegExp Matching, non-matching
            while (!isGETLINESTAT && // print 解析開始時の '(' のネストとチェック
                    rxRELATIVEOP.find(super.tok) && // `< > <= >= != == ' Relational
                    !(isPRINTSTAT && (cPARENNEST >= super.yyPARENc) &&
                            rxREDIRECT.find(super.tok))) { // リダイレクトでない
                String op = rxRELATIVEOP.group(1);
                eat(super.tok);
                ex = new Node.Comp(op, ex, concatOp());
            }
            return ex;
        }
    }

    /**
     * 項を返す.
     */
    @Override
    Object term() {
        Object x = super.tok;
        if ("(".equals(x)) { // `(...)' Grouping
            return parenlist();

        } else if ("[".equals(x)) { // `[...]' Index
            eat(super.tok);
            Object[] ex = optparenlist();
            eat("]");
            return index(ex);

        } else if ("`~".equals(x)) { // NOTE REMIND '`~'ビット否定
            advance();
            return new Node.IncDec("~", incdecNegateType((Node.YyVariable) term()));

        } else if ("++".equals(x)) { // NOTE `++', 前置き increment
            advance();
            return new Node.IncDec("++", incdecNegateType((Node.YyVariable) term()));
        } else if ("--".equals(x)) { // NOTE `--', 前置き decrement
            advance();
            return new Node.IncDec("--", incdecNegateType((Node.YyVariable) term()));
        } else if (Keyword.SymGETLINE == x) {
            advance();
            return getlineStmt("", ""); // XXX: getLine x <$2 > 0
        } else if (Keyword.SymPRINT == x) {
            return printStmt(Keyword.SymPRINT); // XXX: print > $1 ".log"
        } else if (Keyword.SymPRINTF == x) {
            return printStmt(Keyword.SymPRINTF);

        } else if (x.toString().matches("^[.]{1,2}")) { // NOTE function value.
            Keyword id = (1 == x.toString().length()) ?
                    Keyword.SyyFVal : Keyword.SyyFValFuture;
            advance();
            return functionValue(id);

        } else if (x instanceof Term.BOXING) { // 名前に'.'を含む
            advance();
            return invokeStmt(Keyword.SymINVOKE, ((Term.BOXING) x).id);
        } else if ((x instanceof Node.NAME)
                && "new".equals(((Node.NAME) x).name)) {
            advance();
            return newStmt();
        } else if (x instanceof Node.NAME) { // NAME
            advance();
            return factor((Node.NAME) x);
        } else {
            advance();
            return x;
        }
    }

    /**
     *
     */
    private Object unaryop() {
        if (rxUNARYOP.find(super.tok)) { // `+ - !' Unary plus, minus, logical "not"
            String op = super.tok.toString();
            eat(super.tok);
            Object e = powerop();
            Object[] ea = Type.castWrapArray(e);
            if (Node.NOT_OP.equals(op)) { // `!' logical "not"
                return new Node.B00l(Node.NOT_OP, ea);
            } else if (e instanceof Term.NUMBER) { // `+ -' Unary plus, minus
                double dd = 0 - ((Term.NUMBER) e).doubleValue();
                if ((0 == dd) || "+".equals(op)) {
                    return ea; // -0 +0 +a
                }
                Number num = NumHelper.normalise(dd);
                return new Term.NUMBER(num.toString(), num); // -a
            } else {
                if ("-".equals(op)) {
                    return new Node.Calc(op, Term.NUMBER_ZERO, ea);
                }
                yyOPTIMIZE("unaryop: (" + op + ") " + Type.unWrapList(e));
                return ea;
            }
        }
        return powerop();
    }
}