/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.variable;

import org.jetbrains.annotations.NotNull;
import plus.util.NumHelper;

import java.util.Objects;

/**
 * 身代わりオブジェクトの実装.
 * <p>
 * from: <code>sub(/^[ ]+/, "", v);</code><br>
 * to: <code>sub(/^[ ]+/, "",
 * {<i>v = new DelegateVar(v)</i>, v});</code><br>
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public class DelegateVar extends Number implements CharSequence {

    /**
     * このオブジェクトに設定された値.
     */
    private volatile Object obj;

    /**
     * 指定されたオブジェクトで代理変数オブジェクトを構築.
     */
    public DelegateVar(Object x) {
        this.obj = unwrap(x);
    }

    /**
     *
     */
    private static Object unwrap(Object x) {
        return (x instanceof DelegateVar) ? unwrap(((DelegateVar) x).apply())
                : x; // recursive call
    }

    /**
     * このオブジェクトの値を返す.
     */
    public final Object apply() {
        return this.obj;
    }

    /**
     * 指定されたインデックス位置にある char 値を返す.
     */
    @Override
    public final char charAt(int index) {
        return toString().charAt(index);
    }

    /**
     * このオブジェクトが表す数値を double 型に変換した値を返す.
     */
    @Override
    public final double doubleValue() {
        return NumHelper.doubleValue(this.obj);
    }

    /**
     * このオブジェクトと等価であるかどうかを返す (for Kotlin).
     */
    @Override
    public final boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if ((null == other) || (getClass() != other.getClass())) {
            return false;
        }
        DelegateVar that = (DelegateVar) other;
        return Objects.equals(this.obj, that.obj);
    }

    /**
     * このオブジェクトが表す数値を float 型に変換した値を返す.
     */
    @Override
    public final float floatValue() {
        return (float) doubleValue();
    }

    /**
     *
     */
    @Override
    public final int hashCode() {
        return (null == this.obj) ? 0 : this.obj.hashCode();
    }

    /**
     * このオブジェクトが表す数値を int 型に変換した値を返す.
     */
    @Override
    public final int intValue() {
        return (int) doubleValue();
    }

    /**
     * この文字シーケンスの長さを返す.
     */
    @Override
    public final int length() {
        return toString().length();
    }

    /**
     * このオブジェクトが表す数値を long 型に変換した値を返す.
     */
    @Override
    public final long longValue() {
        return (long) doubleValue();
    }

    /**
     * このオブジェクトのサブシーケンスを返す.
     */
    @NotNull
    @Override
    public final CharSequence subSequence(int start, int end) {
        return toString().subSequence(start, end);
    }

    /**
     * このオブジェクトの文字列表現を返す.
     */
    @NotNull
    @Override
    public final String toString() {
        return this.obj.toString();
    }

    /**
     * このオブジェクトの値を再設定する (can overrides).
     */
    public Object update(Object x) {
        this.obj = unwrap(x);
        return this.obj;
    }
}