/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import org.jetbrains.annotations.NotNull;
import plus.util.NumHelper;

import java.util.concurrent.atomic.AtomicReference;

/**
 * Atomic Double.
 * The class to which this annotation is applied is thread-safe.
 *
 * @author Kunio Himei.
 */
@SuppressWarnings("unused")
public class AtomicNumber extends Number
        implements AtomicInterface, ArrayAccessor<Object, Object> {

//    private static final long UN_SIGNED_INT_32_MASK = 0x0FFFFFFFFL;

    private static final Double DoubleOne = 1d;
    /**
     * 原子的な更新が可能なオブジェクト参照.
     */
    private final AtomicReference<Double> atom = new AtomicReference<>();

    /**
     * 指定された初期パラメータを使ってオブジェクトを構築.
     */
    public AtomicNumber() {
        atom.set(0.);
    }

    public AtomicNumber(double value) {
        atom.set(value);
    }

    public AtomicNumber(Number value) {
        this(value.doubleValue());
    }

    /**
     * Calculater.
     * <nl>
     * <li> '=' assign right Object
     * <li> '+' '-' '*' '/' '%' '**' calculate
     * <li> '<<' left shift
     * <li> '>>;' right shift (>>> unsigned)
     * <li> '&' bitwise AND
     * <li> '|' bitwise OR
     * <li> '^' bitwise XOR
     * <li> '~' bit clear (and NOT)
     */
    private static double calculate(String op,
                                    double left, double right) {
        //noinspection EnhancedSwitchMigration
        switch (op.charAt(0)) {
            case '=':
                return right;
            case '+':
                return left + right;
            case '-':
                return left - right;
            case '*':
                return (op.startsWith("**")) ?
                        Math.pow(left, right) : left * right;
            case '/':
                return left / right;
            case '%':
                return left % right;

            case '<': // << left shift
                return ((long) left) << (int) right;

            case '>': // >>(>) unsigned right shift
                if (op.startsWith(">>>"))
//                return (UN_SIGNED_INT_32_MASK & (long) left) >>> (int) right;
                    return (long) left >>> (int) right;
                return (long) left >> (int) right;

            case '&': // & and
                return ((long) left) & (long) right;

            case '|': // | or
                return ((long) left) | (long) right;

            case '^': // ^ xor
                return ((long) left) ^ (long) right;

            case '~': // ~ bitwiseNegate
                return ~((long) left);

            default: // break;
        }
        throw new IllegalArgumentException(left + " " + op + ' ' + right);
    }

    /**
     * Atomic Calculater - 新規 AtomicNumberを返す.
     */
    public static Number calculate(String op, Object left, Object right) {
        return new AtomicNumber(calculate(op,
                NumHelper.doubleValue(left), NumHelper.doubleValue(right)));
    }

    //* 後ろ置きの increment++, decrement-- (副作用の補正)
    public static double postIncDec(String op, double value) {
        if ("+-".equals(op)) return value - 1; // 副作用の補正
        if ("-+".equals(op)) return value + 1;
        return value;
    }

    /**
     * 「現在の値==予想される値」である場合、値を指定された更新値に原子的に設定する.
     */
    @Override
    public double calculate(String op, Object right) {
        double value = NumHelper.doubleValue(right);
        if ('=' == op.charAt(0)) {
            setAtom(value);
            return value;
        }
        double expect, update;
        do {
            expect = atom.get();
            update = calculate(op, expect, value);
            // NOTE 'compareAndSet'が、動作しなくなったため代替手段をとる.
            // - } while (!atom.compareAndSet(expect, update));
        } while (expect != atom.getAndSet(update));
        return postIncDec(op, update);
    }

    //* ------------------------------------------------------------------------
    //* Atomic Interface - for Override.
    //* ------------------------------------------------------------------------
    //* 現在の値を返す.
    @Override
    public Object getAtom() {
        return NumHelper.normalise(atom.get());
    }

    //* 指定された値に設定する.
    @Override
    public void setAtom(Object value) {
        atom.set(NumHelper.doubleValue(value));
    }

    //* ------------------------------------------------------------------------
    //*  このオブジェクトが表す数値を返す.
    //* ------------------------------------------------------------------------
    @Override
    public final double doubleValue() {
        return NumHelper.doubleValue(getAtom());
    }

    @Override
    public final float floatValue() {
        return (float) doubleValue();
    }

    @Override
    public final int intValue() {
        return (int) doubleValue();
    }

    @Override
    public final long longValue() {
        return (long) doubleValue();
    }

    /**
     * このオブジェクトの文字列表現を返す.
     */
    @Override
    public String toString() {
        return getAtom().toString();
    }

    //* ------------------------------------------------------------------------
    //* 比較 (Comparable).
    //* ------------------------------------------------------------------------
    //* 大小比較. a < b, a > b, a <= b, a >= b, a <=> b
    public int compareTo(@NotNull Object right) {
        return Double.compare(NumHelper.doubleValue(getAtom()), //
                NumHelper.doubleValue(right));
    }

    //* 等値判定. a == b, a != b
    @SuppressWarnings({"EqualsWhichDoesntCheckParameterClass"})
    public boolean equals(Object right) {
        return 0 == compareTo(right);
    }

    //* ------------------------------------------------------------------------
    //* Array Getter / Setter - for Groovy.
    //* ------------------------------------------------------------------------
    @Override
    public Character getAt(Object key) { //
        return toString().charAt(NumHelper.intValue(key));
    }

    @Override
    public synchronized void putAt(Object key, Object value) { // SYNC.
        int p = NumHelper.intValue(key);
        String x = value.toString();
        StringBuilder sb = new StringBuilder(toString());
        if (!x.isEmpty())
            sb.insert(p, x);
        else sb.deleteCharAt(p);
        setAtom(Double.parseDouble(sb.toString()));
    }

    //* ------------------------------------------------------------------------
    //* Operator Overloading - for Groovy.
    //* ------------------------------------------------------------------------

    //* インクリメント next - a++, ++a (副作用は、非サポート)
    public Number next() {
        calculate("++", DoubleOne);
        return this;
    }

    //* デクリメント previous - a--, --a (副作用は、非サポート)
    public Number previous() {
        calculate("--", DoubleOne);
        return this;
    }

    //* 加算 plus - a + b
    public Number plus(Object value) {
        calculate("+", value);
        return this;
    }

    //* 減算 minus - a - b
    public Number minus(Object value) {
        calculate("-", value);
        return this;
    }

    //* 乗算 multiply - a * b
    public Number multiply(Object value) {
        calculate("*", value);
        return this;
    }

    //* 除算 div - a / b
    public Number div(Object value) {
        calculate("/", value);
        return this;
    }

    //* 剰余 mod - a % b
    public Number mod(Number value) {
        calculate("%", value);
        return this;
    }

    //* べき乗 power - a ** b
    public Number power(Number value) {
        calculate("**", value);
        return this;
    }

    //* 左シフト leftShift - a << b
    public Number leftShift(Number value) {
        calculate("<<", value);
        return this;
    }

    //* 右シフト rightShift - a >> b
    public Number rightShift(Number value) {
        calculate(">>", value);
        return this;
    }

    //* 論理右シフト rightShiftUnsigned - a >>> b
    public Number rightShiftUnsigned(Number value) {
        calculate(">>>", value);
        return this;
    }

    //* ビット積 and - a & b
    public Number and(Number value) {
        calculate("&", value);
        return this;
    }

    //* ビット和 or - a | b
    public Number or(Number value) {
        calculate("|", value);
        return this;
    }

    //* 排他的ビット和 xor - a ^ b
    public Number xor(Number value) {
        calculate("^", value);
        return this;
    }

    //* ビット否定 bigwiseNegate - ~a
    public Number bitwiseNegate() {
        calculate("~", DoubleOne);
        return this;
    }

/*
    * インクリメント next - a++, ++a (副作用は、不可)
    * デクリメント previous - a--, --a (副作用は、不可)
    * 加算 plus - a + b
    * 減算 minus - a - b
    * 乗算 multiply - a * b
    * 除算 div - a / b
    * 剰余 mod - a % b
    * べき乗 power - a ** b
    * 左シフト leftShift - a << b
    * 右シフト rightShift - a >> b
    * 論理右シフト rightShiftUnsigned - a >>> b
    * ビット積 and - a & b
    * ビット和 or - a | b
    * 排他的ビット和 xor - a ^ b

    * 正規表現適合 - a !~, ~ /regex/
    * ビット否定 bigwiseNegate - ~a
    *
    * NOTE ~ は、正規表現適合.
    * NOTE ビット否定(NOT)は、AWK~Plus(`~) -> Groovy(~)と変換する.

    * 強制型変換 a.asType(b) - a as b (標準実装)

    a.positive()		+a		単項プラス
    a.negative()		-a		単項マイナス
    a.isCase(b)			a in b	包含判定
*/
}