/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import plus.BiIO;

import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

/**
 * nanoShell - Built-in command
 *
 * @author kunio himei.
 */
public class NanoTools {
    private Analysis ana;
    private boolean isRemove;

    /**
     * @param x command line arguments
     */
    private void Initialize(String input, Object... x) {
        ana = new Analysis(input);
        isRemove = false; // never delete
        optAll.clear(); // そのモジュールで利用可能なオプション
        options.clear(); // ユーザ定義されたオプション
        optInput.clear(); // 指定された(生)オプション
        for (Object o : x) {
            String[] opts = o.toString().trim().toLowerCase()
                    .split("\\s+");
            optInput.addAll(List.of(opts));
        }
        __STDOUT = defaultOUT;
        __STDERR = defaultERR;
        redirect(x);
    }

    private void close() {
        closeImpl(__STDOUT);
        closeImpl(__STDERR);
    }

    private void closeImpl(Redirect re) {
        try {
            BiIO.fflush(re.file);
            if (re.defined)
                BiIO.close(re.file);
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("close", re.name));
        }
    }

    /**
     * Parsing redirect definitions
     *
     * @param x command line arguments
     */
    private void redirect(Object... x) {
        for (Object o : x) {
            if (o instanceof String e) {
                int ix1 = e.indexOf('>');
                if (ix1 < 0) continue;
                int ix2 = e.indexOf('>', ix1 + 2); // skip >>
                if (ix2 >= 0) {
                    ix2 = e.lastIndexOf(' ', ix2);
                    if (ix2 >= 0) {
                        redirectImpl(e.substring(0, ix2));
                        redirectImpl(e.substring(ix2 + 1));
                    } else {
                        throw new IllegalArgumentException(throwMessage(
                                "redirect", e));
                    }
                } else {
                    redirectImpl(e);
                }
            }
        }
    }

    // <- SPACE
    private static final char RE_QUOTE = '\'';
    private static final char BOM_QUOTE_BE = '\uFEFF'; // UTF16-BE <- \'
    private static final Pattern IS_REDIRECT = Pattern // 貪欲な正規表現
            .compile("([12]?)(>{1,2})\\s*(" + RE_QUOTE + "?)(.+)");

    private void redirectImpl(String input) {
        String src = input.trim()
                .replace("\\" + RE_QUOTE, Character.toString(BOM_QUOTE_BE)); // \'
        Matcher m = IS_REDIRECT.matcher(src);
        if (m.find()) {
            String rno = getValue(m.group(1));
            String rid = getValue(m.group(2));
            boolean brackets = !getValue(m.group(3)).isEmpty();
            String file = getValue(m.group(4));
            if (brackets) {
                int ix = file.lastIndexOf(RE_QUOTE); // '...'
                if (ix >= 0) { // 終端の ' を剥がす
                    file = file.substring(0, ix);
                } else {
                    throw new IllegalArgumentException(throwMessage(
                            "Paired <'> mistake", input));
                }
            }
            file = file.trim()
                    .replace(BOM_QUOTE_BE, RE_QUOTE);
            String name = rno + rid + file;
            Redirect re = new Redirect(true, name,
                    rid, file, rno.equals("2"));
            if (re.isErr) this.__STDERR = re;
            else this.__STDOUT = re;
            options.put(name, true); // Used in option display

//            System.err.println("re " + input + " | " + file);
        }
    }

    private Redirect __STDOUT; // リダイレクト
    private Redirect __STDERR; //
    private static final Redirect defaultOUT =
            new Redirect(false, "", "", Io.STDOUT, false);
    private static final Redirect defaultERR =
            new Redirect(false, "", "", Io.STDERR, true);

    private static final int STATUS_ERROR = 3;
    private static final int STATUS_WARNING = 2;
    private static final int STATUS_FILE = 1; // request file
    private static final int STATUS_INFORMATION = STATUS_FILE;
    private static final int STATUS_NORMAL = 0; // folder or file

    private static final int FLAG_ROOT = 1;
    private static final int FLAG_DIRECTORY = 2;
    private static final int FLAG_FILE = 4;

    /**
     * @param input Check paths for deletion
     * @param flags Allow root specification ('/') (e.g. tree)
     * @return STATUS
     */
    private static int checkPath(Path input, int flags) {
        Path path = input.normalize().toAbsolutePath();
        if (Files.isDirectory(path)) {
            boolean isRoot = path.equals(path.getRoot());
            if (isRoot && (flags & FLAG_ROOT) == 0) {
                rootCannotBeSpecified(input);
                return STATUS_ERROR; // ルートフォルダは指定できない
            }
//            System.err.println("  ck Dir  " + input+"\t"+Files.exists(path));
            return STATUS_NORMAL;
        } else if ((flags & FLAG_FILE) != 0) {
//            System.err.println("  ck File " + input+"\t"+Files.exists(path));

            return Files.exists(path) ?
                    STATUS_NORMAL : // ファイルが検出された
                    STATUS_FILE; // まだ作成されていない
        }
        noSubfolderExist(input);
        return STATUS_WARNING; // ファイル・フォルダが存在しない
    }

    private static final Pattern IS_OUTPUT_FOLDER = Pattern
            .compile("[/\\\\]([^/\\\\]*)$");

    /**
     * パスが ./foo/ - '/'で終わるとき、フォルダを作成する
     */
    private static Path checkOutPath(String output) {
        String src = output.trim();
        Path out = Path.of(src).normalize();
        if (Files.exists(out)) return out; // shortcut
        try {
            Matcher m = IS_OUTPUT_FOLDER.matcher(src);
            if (m.find()) {
                boolean hasFile = !getValue(m.group(1)).isEmpty();
                Path path = hasFile ? out.getParent() : out;
                if (null != path && !Files.exists(path)) {
                    Files.createDirectories(path);
                }

//                messageCYAN(src, path);
            }
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("md " + src, e));
        }
        return out;
    }

    private final Map<String, Boolean> optAll = new TreeMap<>(); // そのモジュールで利用可能なオプション
    private final Map<String, Boolean> options = new TreeMap<>(); // ユーザ定義されたオプション
    private final Set<String> optInput = new HashSet<>(); // (生)オプション

    private static final String COMMA = ","; // ls length
    private static final String FILE = "file"; // tree
    private static final String L260 = "260"; // ls
    private static final String Lno260 = "no260"; // ls
    private static final String ROOT = "root"; // remove
    private static final String SIMPLE = "0"; // ls length
    private static final String SYNC = "sync"; // copy
    private static final String UNIT = "k"; // ls length
    private static final String noRECURSIVE = "noRecursive";

    /*
     * 大小文字を区別しないで前方一致で比較
     */
    private boolean checkOption(String name, boolean val) {
        String key = name.toLowerCase();
        String key2 = key.startsWith("no") ?
                '-' + key.substring(2) : key;
        for (String x : optInput) {
            if (!x.isEmpty() && (key.startsWith(x) || key2.startsWith(x))) {
                optAll.put(name, val);
                options.put(name, val);
                return val;
            }
        }
        optAll.put(name, !val);
        return !val;
    }

    private boolean hasOption(String name) {
        if (optAll.containsKey(name))
            return optAll.get(name);
        throw new RuntimeException(throwMessage("No option", name));
    }

    private String listOptions() {
        StringBuilder sb = new StringBuilder(48);
        sb.append(BLUE);
        for (String x : options.keySet()) {
            sb.append(' ').append(x);
        }
        return sb.append(RESET).toString();
    }

    // System フォルダをスキップする
    private static boolean isSystem(Path path) {
        String str = path.toString();
        return str.contains("System Volume Information") ||
                str.contains("$RECYCLE.BIN");
    }

    private static boolean isSystem(File path) {
        return isSystem(path.toPath());
    }

    // ワイルドカードにマッチング
    private boolean isMatch(Path path) {
        return isMatch(path.toFile());
    }

    private boolean isMatch(File path) {
        if (path.exists() && !path.isDirectory()) {
            if (ana.alwaysTrue) return true;
            String name = path.getName();
            return ana.regex.matcher(name).matches();
        }
        return false;
    }

    /**
     * long time = path.lastModified();
     */
    private static void setLastModified(Path path, long time) {
        File file = path.toFile();
        if (0 < time && time != file.lastModified()) {
            if (!file.setLastModified(time))
                throw new RuntimeException(throwMessage("setLastModified", path));
        }
    }

    /**
     * throw を発生させない (フォルダが空でない場合の対策)
     *
     * @param path フォルダ or ファイル
     */
    private boolean safetyRemove(Path path) {
        return isRemove && Files.exists(path) &&
                !isSystem(path) && path.toFile().delete();
    }

    private static final boolean _STDOUT = true;
    private static final boolean _STDERR = false;

    /**
     * @param stdout [Default value: hasSTDOUT] else, hasSTDERR
     */
    private void printX(boolean stdout, String x) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        BiIO.print(re.rid, re.file, x);
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, path name
     */
    @SuppressWarnings("SameParameterValue")
    private void printX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.defined()) {
            printX(stdout, applySlashSeparator(file, type));
        } else {
            printX(stdout, truncatePath(file, type));
        }
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, path name
     */
    @SuppressWarnings("SameParameterValue")
    private String sprintX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.defined()) {
            return applySlashSeparator(file, type);
        } else {
            return truncatePath(file, type);
        }
    }

    /**
     * copy the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int copy(String input, String output, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        Path out = checkOutPath(output);
        isRemove = checkOption(SYNC, true);
        boolean isRecursive = checkOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' + output + listOptions();
        messageTitle("copy", args);
        if (in.compareTo(out) == 0) {
            inputAndOutputAreSameFile(in);
            return STATUS_ERROR; // in, out are the same
        }
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        int ro = checkPath(out, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        rc = Math.max(rc, ro);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            if (isRemove) {
                copySync(set, in, out); // synchronous copy
                if (set.size() > 0) {
                    for (File sync : set) {
                        printX(_STDERR, sync);
                    }
                    messageCYAN("Synchronized:", set.size());
                    set.clear();
                }
            }
            copyImpl(set, in, out, isRecursive);
            for (File file : set) {
                printX(_STDOUT, file);
            }
            isRemove = true; // always delete
            setDateForChildElement(out); // clean up folder
            messageCYAN("Number of processed:", set.size());
        }
        close();
        return rc;
    }

    private void copyImpl(Set<File> set, Path input, Path output,
                          boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(newIn.getFileName());
                    if (file.isDirectory()) {
                        if (recursive) {
                            createDirectory(newIn, newOut); // 日時を設定
                            //noinspection ConstantValue
                            copyImpl(set, newIn, newOut, recursive);
                        }
                    } else {
                        atomicSingleCopy(set, newIn, newOut);
                    }
                }
            }
        } else {
            atomicSingleCopy(set, input, output);
        }
    }

    /**
     * Synchronous copy
     */
    private void copySync(Set<File> syn, Path input, Path output) {
        Map<String, File> map = new HashMap<>(256);
        if (isSystem(input) || isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) { // output
            File[] files = output.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    map.put(file.getName(), file);
                }
            }
        }
        if (Files.exists(output)) {
            File file = output.toFile();
            map.put(file.getName(), file);
        }
        if (Files.isDirectory(input)) { // input
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    String name = file.getName();
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(name);
                    map.remove(name);
                    if (file.isDirectory()) {
                        copySync(syn, newIn, newOut);
                    }
                }
            }
        }
        if (Files.exists(input)) {
            String name = input.toFile().getName();
            map.remove(name);
        }
        for (File file : map.values()) {
            copySyncRemove(syn, file.toPath()); // Synchronous
        }
    }

    /**
     * Synchronous copy - 指定されたパスを無条件に削除
     *
     * @param input 　file or folder
     */
    private void copySyncRemove(Set<File> syn, Path input) {
        if (isSystem(input)) return; // skip System folder
        File[] files = input.toFile().listFiles();
        if (null != files) {
            for (File file : files) {
                if (isSystem(file)) continue; // skip System folder
                Path in = file.toPath();
                if (file.isDirectory()) {
                    copySyncRemove(syn, in);
                    safetyRemove(in); // フォルダが空でない場合の対策
                } else {
                    if (safetyRemove(in)) { // フォルダが空でない場合の対策
                        syn.add(file);
                    }
                }
            }
        }
        boolean isFile = !Files.isDirectory(input);
        if (safetyRemove(input) && isFile) { // フォルダが空でない場合の対策
            syn.add(input.toFile());
        }
    }

    /**
     * move the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int move(String input, String output, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        Path out = checkOutPath(output);
        isRemove = true; // always delete
        boolean isRecursive = checkOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' + output + listOptions();
        messageTitle("move", args);
        if (in.compareTo(out) == 0) {
            inputAndOutputAreSameFile(in);
            return STATUS_ERROR; // in, out are the same
        }
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        int ro = checkPath(out, FLAG_ROOT | FLAG_DIRECTORY | FLAG_FILE);
        rc = Math.max(rc, ro);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            moveImpl(set, in, out, isRecursive);
            for (File file : set) {
                printX(_STDOUT, file);
            }
            setDateForChildElement(out); // clean up folder
            messageCYAN("Number of processed:", set.size());
        }
        close();
        return rc;
    }

    private void moveImpl(Set<File> set, Path input, Path output,
                          boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(newIn.getFileName());
                    if (file.isDirectory()) {
                        if (recursive) {
                            createDirectory(newIn, newOut); // 日時が設定されない
                            //noinspection ConstantValue
                            moveImpl(set, newIn, newOut, recursive);
                            safetyRemove(newIn); // 子要素がなければ削除
                        }
                    } else {
                        atomicSingleMove(set, newIn, newOut);
                    }
                }
            } else if (Files.exists(input)) {
                long lastMod = input.toFile().lastModified();
                setLastModified(output, lastMod);
            }
        } else {
            atomicSingleMove(set, input, output);
            safetyRemove(input); // 子要素がなければ削除
        }
        safetyRemove(input); // 子要素がなければ削除
    }

    /**
     * 子要素の日付を親フォルダに設定 - Set date for child element
     * <p>
     * move の原因不明の不具合で入力フォルダの日付が取得できないため作成
     */
    private long setDateForChildElement(Path input) {
        if (isSystem(input)) return 0; // skip System folder
        long modTime = 0;
        if (Files.isDirectory(input)) {
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    Path path = file.toPath();
                    long time;
                    if (file.isDirectory()) {
                        time = setDateForChildElement(path);
                    } else {
                        time = file.lastModified();
                    }
                    if (modTime < time) modTime = time;
                }
            }
        }
        if (Files.isDirectory(input) && !safetyRemove(input))
            setLastModified(input, modTime); // 空のフォルダを削除
        return modTime;
    }

    /**
     * 単一のファイルを差分コピー (出力側に存在しない、日付時刻・サイズが異なる場合)
     */
    private synchronized void atomicSingleCopy(Set<File> set,
                                               Path input, Path output) {
        try {
            output = isSameFile(input, output);
            if (output == null) return;

            Files.createDirectories(output); // ディレクトリを作成

            Files.deleteIfExists(output); // 最後の要素を削除
            Files.copy(input, output, COPY_ATTRIBUTES, REPLACE_EXISTING);

            if (output.toFile().isFile()) {
                set.add(input.toFile());
            }
        } catch (IOException e) {
            throw new RuntimeException(throwMessage("copy", e));
        }
    }

    /**
     * ファイルをターゲット・ファイルに移動するか、そのファイル名を変更
     */
    private synchronized void atomicSingleMove(Set<File> set,
                                               Path input, Path output) {
        try {
            output = isSameFile(input, output);
            if (output == null) return;

            Files.createDirectories(output); // ディレクトリを作成

            Files.deleteIfExists(output); // 最後の要素を削除
//          Files.move(input, output, ATOMIC_MOVE); // 別ボリュームは NG
            Files.move(input, output, REPLACE_EXISTING);
            if (output.toFile().isFile()) {
                set.add(input.toFile());
                safetyRemove(input); // ダメ押しで、子要素がなければ削除
            }

        } catch (IOException e) { // AtomicMoveNotSupportedException
            throw new RuntimeException(throwMessage("move", e));
        }
    }

    /**
     * 属性が同一ファイルかどうかの判定
     *
     * @return true: if it no matches
     */
    private Path isSameFile(Path input, Path output) {
        if (!Files.exists(input) || Files.isDirectory(input))
            throw new RuntimeException(throwMessage("Not a file", input));
        if (!isMatch(input)) return null; // wildcard mismatch
        if (isSystem(input)) return null; // skip System folder
        if (Files.exists(output) &&  // out がディレクトリならファイル名を設定
                Files.isDirectory(output)) {
            output = output.resolve(input.getFileName());
        }
        if (input.compareTo(output) == 0) return null; // same file

        Path inParent = input.getParent();
        if (inParent != null) { // in の親ディレクトリを作成
            Path outParent = output.getParent();
            if (outParent != null) { // out の親ディレクトリを作成
                createDirectory(inParent, outParent);
            }
        }
        if (Files.exists(output) && !Files.isDirectory(output)) {
            File iFile = input.toFile();
            File oFile = output.toFile(); // Check file attributes
            boolean length = iFile.length() == oFile.length();
            long modified = iFile.lastModified() - oFile.lastModified();
            if (length && modified < 2000) // FAT(誤差2sec) <> NTFS
                return null;
        }
        return output;
    }

    /**
     * 出力側のディレクトリを作成し、日時を設定
     */
    private static void createDirectory(Path input, Path output) {
        try {
            if (Files.isDirectory(input)) {
                FileTime fileTime = Files.getLastModifiedTime(input);

                if (!Files.exists(output))
                    Files.createDirectories(output); // ディレクトリを作成

                Files.setLastModifiedTime(output, fileTime); // 日時を設定
            }

        } catch (IOException e) {
            throw new RuntimeException(throwMessage("Create directory", output));
        }
    }

    /**
     * remove - 指定されたパス以下にあるワイルドカードに一致するファイルを削除する
     * - Delete files matching wildcards under the specified path
     * - Path, Files version
     *
     * @param input Requires at least one sub-path
     * @param x     is remove [ false ]
     */
    public int remove(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isRecursive = checkOption(noRECURSIVE, false);
        boolean isRoot = checkOption(ROOT, true);
        String args = ana.virtualPath + listOptions();
        messageTitle("remove", args);
        int flags = FLAG_DIRECTORY | FLAG_FILE;
        if (isRoot) flags |= FLAG_ROOT;
        int rc = checkPath(in, flags);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            removeImpl(set, in, isRecursive);
            for (File remove : set) {
                if (!remove.isDirectory())
                    printX(_STDOUT, remove);
            }
            setDateForChildElement(in); // clean up folder
            messageCYAN("Number of processed:", set.size());
        }
        close();
        return rc;
    }

    private void removeImpl(Set<File> set, Path input,
                            boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            removeImpl(set, path, recursive);
                            if (safetyRemove(path))
                                set.add(path.toFile()); // 子供のいないフォルダ対策
                        }
                    } else {
                        if (isMatch(path) && safetyRemove(path)) {
                            set.add(path.toFile()); // フォルダが空でない場合の対策
                        }
                    }
                }
                safetyRemove(input); // フォルダが空でない場合の対策

            } catch (IOException e) { // DirectoryNotEmptyException
                throw new RuntimeException(throwMessage("remove", e));
            }
        } else if (isMatch(input) && Files.exists(input)) {
            safetyRemove(input);
            set.add(input.toFile());
        }
    }

    /**
     * tree - File version
     *
     * @param input Path to start listing
     */
    public int tree(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isRecursive = checkOption(noRECURSIVE, false);
        boolean isFile = checkOption(FILE, true);
        String args = ana.virtualPath + listOptions();
        messageTitle("tree", args);
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY);
        if (rc <= STATUS_INFORMATION) {
            int max = treeImpl(in.toFile(), "", isFile, isRecursive);
            messageCYAN("MAX_PATH:", max + " char");
        }
        close();
        return rc;
    }

    @SuppressWarnings("ConstantValue")
    private int treeImpl(File input, String indent, boolean isFile, boolean recursive) {
        if (isSystem(input)) return 0; // skip System folder
        int max = 0;
        if (input.isDirectory()) {
            File[] files = input.listFiles();
            if (null != files) {
                TreeSet<File> set = new TreeSet<>(List.of(files));
                for (File file : set) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isFile() && isMatch(file)) {
                        if (isFile) {
                            String x = indent + "| " + truncateTree(file);
                            printX(_STDOUT, x);
                        }
                        max = Math.max(max, file.getPath().length());
                    }
                }
                for (File file : set) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isDirectory()) {
                        if (isSystem(file)) continue; // skip System folder
                        String x = indent + '/' + truncateTree(file);
                        printX(_STDOUT, x);
                        max = Math.max(max, file.getPath().length());
                        if (recursive) {
                            max = Math.max(max, treeImpl(file,
                                    indent + " ", isFile, recursive));
                        }
                    }
                }
            } else {
                if (isMatch(input)) {
                    if (isFile) {
                        String x = indent + "| " + truncateTree(input);
                        printX(_STDOUT, x);
                    }
                    max = input.getPath().length();
                }
            }
        }
        return max;
    }

    /**
     * ls - list segments
     *
     * @param input Path to start listing
     */
    public int ls(String input, Object... x) {
        Initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isSimple = checkOption(SIMPLE, true);
        boolean isComma = checkOption(COMMA, true);
        boolean isUnit = checkOption(UNIT, true);
        boolean is260 = checkOption(L260, true);
        boolean isNo260 = checkOption(Lno260, true);
        boolean isRecursive = checkOption(noRECURSIVE, false);
        boolean isAttr = isSimple || isComma || isUnit;
        String args = ana.virtualPath + listOptions();
        messageTitle("ls", args);
        int rc = checkPath(in, FLAG_ROOT | FLAG_DIRECTORY);
        if (rc <= STATUS_INFORMATION) {
            Set<File> set = new TreeSet<>();
            lsImpl(set, in.toFile(), is260, isNo260, isRecursive);
            for (File file : set) { // パスを切り詰めないで表示
                String ls = lsAttr(file, isAttr);
                printX(_STDOUT, ls);
            }
            messageCYAN("Number of processed:", set.size());
        }
        close();
        return rc;
    }

    private void lsImpl(Set<File> set, File input,
                        boolean is260, boolean isNo260, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (input.isDirectory()) {
            File[] files = input.listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isDirectory()) {
                        if (recursive) {
                            //noinspection ConstantValue
                            lsImpl(set, file, is260, isNo260, recursive);
                        }
                    } else {
                        lsSelect260(set, file, is260, isNo260);
                    }
                }
            } else {
                lsSelect260(set, input, is260, isNo260);
            }
        }
    }

    private String lsAttr(File input, boolean isAttr) {
        StringBuilder sb = new StringBuilder(128);
        if (isAttr) {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yy/MM/dd HH:mm");
            String daytime = sdf.format(input.lastModified());
            String len = lsLength(input.length());
            sb.append(daytime).append('\t');
            sb.append(len).append('\t');
        }
        return sb.append(sprintX(_STDOUT, input)).toString();
    }

    /**
     * Output long path
     *
     * @param is260   Output paths longer than 260
     * @param isNo260 Output paths under 260
     */
    private void lsSelect260(Set<File> set, File input,
                             boolean is260, boolean isNo260) {
        if (isMatch(input)) {
            if (is260) {
                if (as260(input)) set.add(input);
            } else if (isNo260) {
                if (!as260(input)) set.add(input);
            } else {
                set.add(input);
            }
        }
    }

    private static final int MAX_PATH = 260;

    private static boolean as260(File input) {
        return input.getPath().length() > MAX_PATH;
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static final String NUMBER_UNIT = " KMGTPEZY";

    private String lsLength(long len) {
        if (hasOption(COMMA)) {
            return String.format("%,d", len);
        }
        if (hasOption(UNIT)) {
            for (int i = 0; i < NUMBER_UNIT.length(); i++) {
                if (len < 1024) {
                    String ch = Character.toString(NUMBER_UNIT.charAt(i));
                    return String.format("%3d %s", len, ch);
                }
                len /= 1024;
            }
        }
        return Long.toString(len);
    }

    private static final int MAX_PATH_UNIT_SIZE = 80;
    private static final int PathUnitHalfSize = MAX_PATH_UNIT_SIZE / 2;
    private static final boolean usePathName = true; // Tree でファイル名を表示

    /**
     * 切り詰めたファイル名・IDを返す (returns a truncated path/filename)
     */
    private String truncateTree(File input) {
        StringBuilder sb = new StringBuilder(MAX_PATH_UNIT_SIZE + 16);
        sb.append(sprintX(_STDOUT, input, usePathName)).append(' ');
        String path = input.getPath();
        String name = input.getName();
        String info = name.length() + "/" + path.length();
        String maxPathID = as260(input) ? " *" : "";
        sb.append(maxPathID.isEmpty() ? info :
                color(MAGENTA, info + maxPathID));
        return sb.toString();
    }

    /**
     * パスを切り詰める
     *
     * @param type [Default value: path value] else, path name
     */
    private static String truncatePath(File input, boolean... type) {
        String path = applySlashSeparator(input, type);
        int len = path.length();
        if (MAX_PATH_UNIT_SIZE < len) {
            return path.substring(0, PathUnitHalfSize) + "…" +
                    path.substring(len - PathUnitHalfSize);
        }
        return path;
    }

    /**
     * パス区切り文字を変換
     *
     * @param type [Default value: path value] else, path name
     */
    private static String applySlashSeparator(File input, boolean... type) {
        String path = type.length == 0 ?
                input.getPath() : input.getName();
        if (File.pathSeparatorChar != '/')
            return path.replace('\\', '/');
        return path;
    }

    static String color(String color, String message) {
        return color + message + RESET;
    }

    private static void messageTitle(String name, Object arg) {
        System.out.println(color(YELLOW, name) + ' ' + arg);
    }

    private static void messageCYAN(String name, Object arg) {
        System.err.println(color(CYAN, name) + ' ' + arg);
    }

    private static String throwMessage(String name, Object arg) {
        return color(RED, name) + ' ' + arg;
    }

    // ルートは指定できません
    private static void rootCannotBeSpecified(Path path) {
        System.err.println(color(RED,
                "Root folder cannot be specified: " +
                        applySlashSeparator(path.toFile())));
    }

    // 入力と出力は同じファイル
    private static void inputAndOutputAreSameFile(Path path) {
        System.err.println(color(RED,
                "Input and output are the same file: " +
                        applySlashSeparator(path.toFile())));
    }

    // パスが存在しません
    private static void noSubfolderExist(Path path) {
        System.err.println(color(MAGENTA,
                "Path does not defined: " +
                        applySlashSeparator(path.toFile())));
    }

    private static final String RESET = "\033[0m";
    private static final String RED = "\033[91m";       // error
    @SuppressWarnings("unused")
    private static final String GREEN = "\033[92m";
    private static final String YELLOW = "\033[93m";    // title
    static final String BLUE = "\033[94m";
    private static final String MAGENTA = "\033[95m";   // warning
    private static final String CYAN = "\033[96m";      // information

    private static final Pattern WILD_CARD_ALL = Pattern
            .compile(".*");

    static class Analysis {
        private static final Pattern SPLIT_PATH = Pattern
                .compile("^\\s*(.*[/\\\\])*(.*)?\\s*$");
        final String virtualPath; // path / WildCard
        final Path path; // ワイルドカードを除いた実際のパス
        final Pattern regex; // ワイルドカード
        final boolean alwaysTrue; // ワイルドカードが常に真かどうか


        Analysis(String input) {
            String path = input.trim();
            String wild = "";
            Pattern regex = WILD_CARD_ALL;
            Matcher m = SPLIT_PATH.matcher(input);
            if (m.matches()) {
                String g2 = getValue(m.group(2).trim());
                if (hasWildcard(g2)) {
                    path = getValue(m.group(1)).trim();
                    wild = g2;
                    regex = mkWildcard(g2);
                }
            }
            if (path.isEmpty()) path = "./";
            this.path = Path.of(path); // .normalize();
            this.virtualPath = path +
                    (wild.isEmpty() ? "" : NanoTools.color(NanoTools.BLUE, wild));
            this.regex = regex;
            alwaysTrue = WILD_CARD_ALL.equals(regex);
        }

        /**
         * hasWildcard - ワイルドカード指定かどうかを返す
         * - wildcard: * ? | or .$
         */
        private static boolean hasWildcard(String wild) {
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                if (0 <= "*?|".indexOf(c)) return true;
            }
            return wild.endsWith(".");
        }

        /**
         * mkWildcard - 大小文字を区別せずにマッチする (Case-sensitive)
         * ワイルドカード (* ? .) 以外の記号は、正規表現として使用できる
         * e.g. /[fb]*.txt|*.log/
         *
         * @param wild wildcard
         */
        private static Pattern mkWildcard(String wild) {
            StringBuilder sb = new StringBuilder(256);
            if (wild.endsWith("."))
                wild = wild.substring(0, wild.length() - 1); // Delete the last .
            if (wild.isEmpty()) wild = "*"; // Default value
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                switch (c) {
                    case '*' -> sb.append(".*");
                    case '?' -> sb.append('.');
                    case '.' -> sb.append("\\.");
                    default -> sb.append(c);
                }
            }
            return Pattern.compile(sb.toString(), Pattern.CASE_INSENSITIVE);
        }

    }

    record Redirect(boolean defined, String name,
                    String rid, String file, boolean isErr) {
    }

    private static String getValue(String x) {
        return x == null ? "" : x;
    }
}