/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.runtime;

import org.jetbrains.annotations.Nullable;
import plus.io.Command;
import plus.io.Io;
import plus.io.IoHelper;
import plus.io.TextReader;
import plus.spawn.system.UtilInterface;
import plus.util.NumHelper;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Built-in - AWK 組み込み関数の実装.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public final class RunHelper {

    /**
     * サブプロセスの終了値 (何らかの失敗により終了値を取得できない).
     */
    private static final int CSTAT_ABEND_CODE = 128;

    /**
     * サブプロセスの終了値 (プロセス強制終了シグナルを受信した).
     */
    private static final int CSTAT_CNCEL_CODE = 129;

    /**
     * 空の文字列配列.
     */
    private static final String[] EMPTY_STRING_ARRAY = {};

    /**
     * プラットフォームは Windows?.
     */
    private static final boolean IS_WINDOWS = "\r\n".equals(System
            .getProperty("line.separator"));

    /**
     * group(2) argument index.
     */
    private static final int REGX_GRP2_ARGUMENTINDEX = 2;

    /**
     * group(3) flags,width,precision.
     */
    private static final int REGX_GRP3_FLAGS = 3;

    /**
     * group(4) conversion.
     */
    private static final int REGX_GRP4_CONVERSION = 4;

    /**
     * 書式文字 %g 変換結果の指数を表す正規表現.
     */
    private static final Pattern RX_SPRINTF_EXPONENTIAL = Pattern
            .compile("(.+e[-+])(\\d+)$");

    /**
     * 書式文字列 を表す正規表現.
     */
    private static final Pattern RX_SPRINTF_FORMAT = Pattern
            .compile("(%%)|%(?:(\\d*)\\$)?([^a-zA-Z]*)([a-zA-Z])");

    /**
     * 書式文字列 %g に整形するための正規表現.
     */
    private static final Pattern RX_SPRINTF_G = Pattern.compile("\\.?0+$");

    /**
     * 書式文字 %g 変換結果の後ろ'0'文字 を表す正規表現.
     */
    private static final Pattern RX_SPRINTF_ZTERMINATE_DECIMAL = Pattern
            .compile("\\d\\.\\d*0+$");

    /**
     * コマンドを空白文字で分解するための正規表現.
     */
    private static final Pattern RX_SYSTEM_SPLIT = Pattern.compile("\\s+");

    /**
     * 文字列を結合して返す. (REMIND NULは、削除)
     */
    public static String concat(CharSequence sep,
                                final CharSequence convfmt, Object[] args) {
        CharSequence fmt = (null == convfmt) ? // 数値から文字列への変換書式 (既定値は %.6g)
                BuiltInVar.CONVFMT.toString() : convfmt;
        StringBuilder sb = new StringBuilder();
        for (Object x : args) {
            sb.append(toString(fmt, x)).append(sep);
        }
        sb.setLength(sb.length() - sep.length());
        return sb.toString();
    }

    /**
     * テキスト行入力 (関数としての実装).
     * RT: RS にマッチし入力レコード区切子となった入力テキストを設定.
     */
    @Nullable
    public static synchronized String getlineImpl( // SYNC.
                                                   TextReader reader) throws IOException {
        String line;
        String rs = BuiltInVar.RS.toString();
        line = reader.readLine(rs); // 要求する入力レコードセパレータ (既定値は改行)
        if (null != line) {
            BuiltInVar.RT.put(reader.getRT()); // RTを設定
        }
        return line;
    }

    /**
     * sの中の tの位置をまた tを含まない場合は、0を返す.
     * <p>
     * (USE; IGNORECASE)
     */
    public static int indexOf(CharSequence s, CharSequence t) {
        String x = toString(null, s);
        String y = Objects.toString(t, "");
        int ignoreCase = BuiltInVar.IGNORECASE.intValue(); // 英大小文字の区別指示子
        return 1 + ((0 == ignoreCase) ?
                x.indexOf(y) : x.toLowerCase().indexOf(y.toLowerCase()));
    }

    /**
     * オブジェクトの長さ または 数を返す.
     * <p>
     * 数値: 数値を文字列に変換したときの文字列の数
     */
    public static int length(final Object s) {
        int len = 0;
        if (s instanceof Object[]) { // Variable Arguments: 可変長引数
            len = ((Object[]) s).length;

        } else if (s instanceof Map<?, ?>) { // Array (Map): 連想配列
            len = ((Map<?, ?>) s).size();

        } else if (s instanceof CharSequence) { // 文字列
            // CharBuffer, String, StringBuffer, StringBuilder, StringNumber
            len = ((CharSequence) s).length();

        } else if (s instanceof Number) { // 数値を文字列に変換したときの文字列の数
            len = toString(null, s).length();

        } else if (s instanceof List<?>) {
            len = ((List<?>) s).size() - 1; // (Array は 1 から始まる)

        } else if (s instanceof Collection<?>) { // List, Queue, Set
            len = ((Collection<?>) s).size();

        } else if (s instanceof Iterator<?>) {
            for (Iterator<?> x = (Iterator<?>) s; x.hasNext(); x.next()) {
                len++;
            }
        } else if (s instanceof Enumeration<?>) {
            for (Enumeration<?> x = (Enumeration<?>) s; x
                    .hasMoreElements(); x.nextElement()) {
                len++;
            }
        } else if (null != s) {
            len = s.toString().length(); // その他
        }
        return len;
    }

    //*           group(1)         (2)        (3)         (4)
    //* %[argumentIndex$][flags][width][.precision]conversion
    //* "^(?:(\\d*)\\$)?([^a-zA-Z]*)([a-zA-Z])(.*)", Pattern.DOTALL);

    /**
     * [%helper%] 'NL'を、プラットフォーム依存の改行'ORS'に置き換える.
     */
    public static String replaceNL2ORS(String x, String ors) {
        return x.replaceAll("\r\n|\r|\n", ors); // '\r\n' を変換
    }

    /**
     * [%helper%] 現在実行中のスレッドを一時的にスリープさせる.
     */
    public static int sleep(double second) {
        if (!Thread.currentThread().isInterrupted()) {
            double seconds = second * 1000.0;
            long millis = (long) seconds;
            int nanos = (int) ((seconds - millis) * 1000000);
            try {
                if (0 == nanos) {
                    Thread.sleep(millis);
                } else {
                    Thread.sleep(millis, nanos);
                }
                return 0; // OK
            } catch (InterruptedException e) { // 待機中に割り込みが発生した
                Thread.currentThread().interrupt(); // ひき逃げ
            }
        }
        return 1; // Interrupted
    }

    /**
     * 文字列(s)を正規表現(r)で区切って分割し結果(StringNumber)を配列(a)に設定する.
     *
     * @return StringNumber
     */
    public static int split(String s, Map<String, Object> a, String... r) {
        String fs = (0 == r.length) ? null : r[0];
        a.clear();
        String[] arr = split2Array(s, fs);
        int len = arr.length;
        for (int i = 0; len > i; i++)
            a.put(Integer.toString(i + 1), NumHelper.toStringNumber(arr[i]));
        return len;
    }

    /**
     * 文字列 s を FS で区切って分解した配列を返す.
     */
    public static String[] split2Array(String s, String f) {
        if ((null != s) && (0 < s.length())) { // 空文字でなければ
            String fs = f;
            if (f == null) {
                fs = BuiltInVar.FS.toString();
                if (" ".equals(fs)) {
                    fs = "\\s+";
                    s = s.trim(); // FS=" " なら前(後)空白を削除
                }
            } else if (fs.isEmpty()) { // FS="" 空なら1文字毎に分解
                int i = s.length();
                String[] arr = new String[i];
                while (0 <= --i) {
                    arr[i] = Character.toString(s.charAt(i));
                }
                return arr;
            }
            String guard = "\0\1\2\3"; // NUL SOH STX ETX
            s = s + guard;
            // 1文字のセパレータは正規表現でないためエスケープする
            String fx = 1 < fs.length() ? fs :
                    (0 <= "[]\\".indexOf(fs.charAt(0)) ? // [メタ文字]
                            "[\\" + fs + ']' : '[' + fs + ']');
            Matcher m = RegExp.compile(fx).matcher(s);
            List<String> list = new ArrayList<>();
            int i;
            for (i = 0; m.find(i); i = m.end()) {
                list.add(s.subSequence(i, m.start()).toString());
            }
            if (i < s.length()) {
                list.add(s.subSequence(i, s.length()).toString()); // 残り文字列を複写
            } else {
                list.add(""); // 残り文字列がない場合はセパレータにマッチしている
            }
            for (i = list.size() - 1; i >= 0; i--) { // 番兵を削除
                String x = list.get(i);
                if (x.isEmpty()) { // trim
                    list.remove(i);
                } else {
                    if (x.endsWith(guard)) {
                        x = x.substring(0, x.length() - guard.length()); // trim
                        if (x.isEmpty()) list.remove(i);
                        else list.set(i, x);
                    }
                    break;
                }
            }
            return list.toArray(new String[0]);
        }
        return EMPTY_STRING_ARRAY;
    }

    /**
     * 指定された書式文字列および引数を使ってフォーマットされた文字列を返す.
     */
    public static String sprintf(Object... args) {
        int aglen = args.length;
        if (0 != aglen) {
            StringBuilder sb = new StringBuilder();
            String convfmt = null; // 数値から文字列への変換書式 (既定値は %.6g)
            String fmt = args[0].toString();
            Matcher m = RX_SPRINTF_FORMAT.matcher(fmt);
            // %[argumentIndex$][flags][width][.precision]conversion
            //      (%%)|(?:%(?:(\\d*)\\$)?([^a-zA-Z]*)([a-zA-Z]))
            // group(1)         (2)        (3)         (4)
            int sp = 0;
            for (int i = 0; m.find(); sp = m.end()) {
                sb.append(fmt, sp, m.start());
                if (null != m.group(1)) {
                    sb.append('%');
                } else {
                    i++; // argument index
                    int ix = (null == m.group(REGX_GRP2_ARGUMENTINDEX))
                            ? i : NumHelper.intValue(m
                            .group(REGX_GRP2_ARGUMENTINDEX));
                    Object x = ((0 < ix) && (aglen > ix)) ? args[ix] : "";
                    String fm = '%' + m.group(REGX_GRP3_FLAGS)
                            + m.group(REGX_GRP4_CONVERSION);
                    char f = Character.toLowerCase(m.group(
                            REGX_GRP4_CONVERSION).charAt(0));
                    try {
                        String ss;
                        if (0 <= "aefgdox".indexOf(f)) { // 数値
                            double dd = NumHelper.doubleValue(x);
                            if (0 <= "dox".indexOf(f)) { // 整数
                                ss = String.format(fm, (long) dd);
                            } else { // 実数
                                String ww = String.format(fm,
                                        dd);
                                if (IS_WINDOWS && (0 <= "efg".indexOf(f))) { // e+12 -> e+012
                                    Matcher me = RX_SPRINTF_EXPONENTIAL
                                            .matcher(ww);
                                    if (me.find()) {
                                        ww = me.group(1) + '0' + me.group(2);
                                    }
                                }
                                if (('g' == f)
                                        && RX_SPRINTF_ZTERMINATE_DECIMAL
                                        .matcher(ww).find()) {
                                    ww = RX_SPRINTF_G.matcher(ww).replaceFirst(
                                            "");
                                }
                                ss = ww;
                            }
                        } else if (0 <= "cs".indexOf(f)) { // 文字, 一般
                            if (null == convfmt) {
                                convfmt = BuiltInVar.CONVFMT.toString();
                            }
                            ss = String.format(fm, toString(convfmt, x));
                        } else {
                            ss = String.format(fm, x); // 日付および時刻表現
                        }
                        sb.append(ss);
                    } catch (UnknownFormatConversionException e) {
                        System.err.println("INFO: sprintf('" + m.group() + "','" + x + "')");
                        throw e;
                    }
                }
            }
            if (sp < fmt.length()) {
                sb.append(fmt.substring(sp)); // 残り文字列を複写
            }
            return sb.toString();
        }
        return "";
    }

    /**
     * ☆ 特定された時間を日付フォーマットで文字列化する.
     */
    public static String strftime(String fmt, long timestamp) {
        Date x = new Date(timestamp);
        try {
            return new SimpleDateFormat(fmt).format(x);
        } catch (IllegalArgumentException e) {
            return x.toString();
        }
    }

    /**
     * ☆ 文字列 s の部分文字列を返す.
     */
    public static String substr(CharSequence s, int i, int n) {
        int len = s.length();
        int idx = (0 >= i) ? 0 : ((len < i) ? len : (i - 1));
        int wid = (0 >= n) ? 0 : (Math.min(len, idx + n) - idx);
        return (0 > (len - wid - idx)) ? "" :
                s.subSequence(idx, idx + wid).toString();
    }

    /**
     * コマンドを実行しコマンドの終了値を返す.
     */
    public static int system(String command) {
        String cmd = command.trim();
        if (cmd.isEmpty()) { // fflush() の無い AWK 処理系の Tip.
            System.out.flush(); // 標準出力をフラッシュ
            System.err.flush(); // 標準エラー出力をフラッシュ
            return 0;
        }

        String[] arr = RX_SYSTEM_SPLIT.split(cmd);
        if ("exit".equals(arr[0])) { // コマンドの終了値を設定する
            //noinspection NonStrictComparisonCanBeEquality
            return (1 >= arr.length) ? 0 : NumHelper.intValue(arr[1]);

        } else if ("sleep".equals(arr[0])) {
            if (1 < arr.length) { // 現在実行中のスレッドを一時的にスリープさせる(秒)
                return sleep(NumHelper.doubleValue(arr[1]));
            }
            return 0;
        }

        int rc = CSTAT_ABEND_CODE; // 何らかの失敗により終了値を取得できない
        try {
            UtilInterface out = (UtilInterface)
                    Command.processWriter(cmd);
            if (out.hasInput()) {
                IoHelper.copyLiLne(Io.STDIN, (Writer) out);
            }
            out.close();
            rc = out.exitValue();

        } catch (InterruptedIOException e) { // 待機中に割り込みが発生した
            System.err.println("system: " + e + ": `" + cmd + '`');
            rc = CSTAT_CNCEL_CODE;
        } catch (IOException e) { // 入出力例外が発生した
            System.err.println("system: " + e + ": `" + cmd + '`');
            e.printStackTrace();
        } // else ひき逃げ
        return rc;
    }

    /**
     * [%helper%] オブジェクト(数値)を文字列に変換する.
     */
    public static String toString(final CharSequence convfmt,
                                  final Object x) {
        if (x instanceof String) return (String) x;
        if (x instanceof Number) {
            Number num = NumHelper.normalise((Number) x);
            if ((num instanceof Integer) || (num instanceof Long))
                return num.toString();
            CharSequence fmt = (null == convfmt) ? // 数値から文字列への変換書式 (既定値は %.6g)
                    BuiltInVar.CONVFMT.toString() : convfmt;
            return sprintf(fmt, x); // 数値の出力書式 (既定値は %.6g)
        } else if (x instanceof Boolean) {
            return ((Boolean) x) ? "1" : "0";
        }
        return (null == x) ? "" : x.toString();
    }
}