/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import plus.BiIO;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

/**
 * nanoShell - To the top of the world.
 *
 * @author kunio himei.
 */
public class NanoTools {
    private final ExecutorService pool = Executors.newWorkStealingPool(); // スレッドプール
    private final Set<File> set = new TreeSet<>(); // ※ 出力リスト
    private final List<Path> drop = new ArrayList<>(192); // ※ move
    private final Map<Path, FileTime> mod = new HashMap<>(192); // ※ 日時設定
    private final AtomicInteger atom = new AtomicInteger();
    private long start;
    private Analysis ana;
    private boolean isRemove;

    private int DISPLAY_WIDTH = 40; // 画面への表示幅
    private int MAX_PATH = 260; // 長いパスの警告を表示する敷居値
    private static final String DISPLAY_WIDTH_KEY = "DISPLAY_WIDTH"; // all
    private static final String MAX_PATH_KEY = "MAX_PATH"; // tree, ls

    /**
     * @param x command line arguments
     */
    private void initialize(String input, Object... x) {
        start = System.currentTimeMillis();
        ana = new Analysis(input, DISPLAY_WIDTH);
        isRemove = false; // never delete
        atom.set(0); // ※ 現在のスレッド数
        set.clear(); // ※ 出力リスト
        mod.clear(); // ※ 日時設定
        drop.clear(); // ※ フオルダ削除 move
        optAll.clear(); // そのモジュールで利用可能なオプション
        optIn.clear(); // 明に適用されたオプション
        optARGV.clear(); // ユーザ定義された(生)オプション
        for (Object o : x) {
            String[] opts = o.toString().trim().toLowerCase()
                    .split("\\s+");
            optARGV.addAll(List.of(opts));
        }
        optARGV.remove("");
        __STDOUT = defaultOUT;
        __STDERR = defaultERR;
        redirect(x);
    }

    private void close() {
        closeImpl(__STDOUT);
        closeImpl(__STDERR);
    }

    private void closeImpl(Redirect re) {
        try {
            BiIO.fflush(re.file);
            if (re.redirect)
                BiIO.close(re.file);
        } catch (IOException e) {
            throw new RuntimeException(exception("close", re.name));
        }
    }

    /**
     * 親フォルダを取得
     */
    private static Path getParent(Path path) {
        Path pa = Files.isDirectory(path) ? path : path.getParent();
        return (pa == null) ? path.getRoot() : pa;
    }

    /**
     * 出力パスが ./foo/ - '/'で終わるとき、フォルダを作成する
     */
    private Output createOutFolder(Path input, String output) {
        String src = output.trim();
        Path out = Path.of(src);
        Path in = getParent(input);
        boolean createHolder = false;
        if (src.endsWith("/")) {
            if (Files.notExists(out)) { // root X:/
                if (createFolder(in, out))
                    createHolder = true;
            }
        } else if (!Files.isDirectory(out)) {
            Path parent = getParent(out);
            if (parent != null && Files.notExists(parent)) {
                if (createFolder(in, parent))
                    createHolder = true;
            }
        }
        return new Output(out, createHolder);
    }

    /**
     * 非同期 I/O の完了 - Completion of asynchronous I/O.
     */
    private void finishAsyncIO(Progress pro, Path output) {
        try {
            while (atom.get() > 0) { // スレッドの終了を待ち合わせる
                //noinspection BusyWait
                Thread.sleep(100);
            }
        } catch (InterruptedException e) {
            // do nothing
        }
        pro.flush();
        finishAsyncIOImpl(pro, getParent(output));
    }

    // フォルダ更新日付を設定
    private void finishAsyncIOImpl(Progress pro, Path output) {
        if (isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(output)) {
                for (Path path : ds) {
                    if (Files.isDirectory(path)) {
                        if (isSystem(path)) continue; // skip System folder
                        finishAsyncIOImpl(pro, path);
                    }
                }
                if (removeStartFolder(output)) { // >= Time
                    pro.decrementFolder();
                } else {
                    FileTime time = mod.get(output);
                    if (time != null) {
                        // フォルダ作成,アクセス日時は、ここでコピーされる
                        Files.setLastModifiedTime(output, time);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("cleanup", e));
            }
        }
    }

    // move 入力側ファイルをドロップ
    private void finishAsyncDropFileMove(Path input) {
        for (Path path : drop) {
            removeSafetyFile(path); // move の in 側ファイルを削除
        }
        finishAsyncDropFileImpl(getParent(input));
    }

    // move 入力側をクリーンアップ、Time0フォルダは削除しない
    private void finishAsyncDropFileImpl(Path input) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (Files.isDirectory(path)) {
                        if (isSystem(path)) continue; // skip System folder
                        finishAsyncDropFileImpl(path);
                    }
                }
                removeNoTime0Folder(input); // Other than Time0
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("moveDrop", e));
            }
        }
    }

    private static final int STATUS_ERROR = 3;
    private static final int STATUS_WARNING = 2;
    private static final int STATUS_FILE = 1; // request file
    private static final int STATUS_INFORMATION = STATUS_FILE;
    private static final int STATUS_NORMAL = 0; // folder or file

    private static final int CHK_ROOT = 1;
    private static final int CHK_DIRECTORY = 2;
    private static final int CHK_FILE = 4;

    /**
     * @param input Check paths for deletion
     * @param flags Allow root specification ('/') (e.g. tree)
     * @return STATUS
     */
    private static int checkPath(Path input, int flags) {
        Path path = input.toAbsolutePath();
        if (Files.isDirectory(path)) {
            boolean isRoot = path.equals(path.getRoot());
            if (isRoot && (flags & CHK_ROOT) == 0) {
                rootCannotBeSpecified(input);
                return STATUS_ERROR; // ルートフォルダは指定できない
            }
            return STATUS_NORMAL;
        } else if ((flags & CHK_FILE) != 0) {
            return Files.exists(path) ?
                    STATUS_NORMAL : // ファイルが検出された
                    STATUS_FILE; // まだ作成されていない
        }
        wrongPathSpecified(input); // 間違ったパス指定
        return STATUS_WARNING;
    }

    /**
     * Parsing redirect definitions
     *
     * @param x command line arguments (生データ)
     */
    private void redirect(Object... x) {
        for (Object o : x) {
            String arg = o.toString().trim();
            int ix1 = arg.indexOf('>');
            if (ix1 < 0) continue;
            int ix2 = arg.indexOf('>', ix1 + 2); // skip >>
            if (ix2 >= 0) {
                ix2 = arg.lastIndexOf(' ', ix2);
                if (ix2 >= 0) {
                    redirectImpl(arg.substring(0, ix2));
                    redirectImpl(arg.substring(ix2 + 1));
                } else {
                    throw new IllegalArgumentException(
                            exception("redirect", arg));
                }
            } else {
                redirectImpl(arg);
            }
        }
    }

    // <- SPACE
    private static final String RE_QUOTE = "'";
    private static final String RE_ESCAPE_01 = "\\" + RE_QUOTE;
    private static final String RE_ESCAPE_02 = RE_QUOTE + RE_QUOTE;
    private static final String BOM_QUOTE_BE = "\uFEFF"; // UTF16-BE <- ''
    private static final Pattern IS_REDIRECT = Pattern
            .compile("([12]?)(>{1,2})\\s*(" + RE_QUOTE + "?)(.+)");

    private void redirectImpl(String redirect) {
        String src = redirect.trim()
                .replace(RE_ESCAPE_01, BOM_QUOTE_BE) // \'
                .replace(RE_ESCAPE_02, BOM_QUOTE_BE); // ''
        Matcher m = IS_REDIRECT.matcher(src);
        if (m.find()) {
            String rno = getValue(m.group(1));
            String rid = getValue(m.group(2));
            boolean brackets = !getValue(m.group(3)).isEmpty();
            String file = getValue(m.group(4));
            if (brackets) {
                int ix = file.lastIndexOf(RE_QUOTE); // '...'
                if (ix >= 0) { // 終端の ' を剥がす
                    file = file.substring(0, ix);
                } else {
                    throw new IllegalArgumentException(
                            exception("Paired <'> mistake", redirect));
                }
            }
            file = file.trim()
                    .replace(BOM_QUOTE_BE, RE_QUOTE);
            String name = rno + rid + file;
            Redirect re = new Redirect(true, name,
                    rid, file, rno.equals("2"));
            if (re.isErr) this.__STDERR = re;
            else this.__STDOUT = re;
            optIn.add(name); // Used in option display

//            messageGREEN("re " + input, file);
        }
    }

    private Redirect __STDOUT; // リダイレクト
    private Redirect __STDERR; //
    private static final Redirect defaultOUT =
            new Redirect(false, "", "", Io.STDOUT, false);
    private static final Redirect defaultERR =
            new Redirect(false, "", "", Io.STDERR, true);

    // Optional parameter variable
    private final Map<String, Object> optAll = new TreeMap<>(); // そのモジュールで利用可能なオプション
    private final Set<String> optIn = new TreeSet<>(); // 明に適用されたオプション
    private final Set<String> optARGV = new HashSet<>(32); // ユーザ定義された(生)オプション

    private boolean isLonger260(Path input) {
        return input.toString().length() > MAX_PATH;
    }

    private static final String CLEAN = "clean"; // remove
    private static final String COMMA = ","; // ls size
    private static final String FILE = "file"; // tree
    private static final String MX_260 = "260"; // ls
    private static final String MX_no260 = "no260"; // ls
    private static final String PATH = "path"; // ls path length
    private static final String PRO = "progress"; // copy, move, remove
    private static final String ROOT = "root"; // remove
    private static final String SIMPLE = "0"; // ls size
    private static final String SYNC = "sync"; // copy
    private static final String UNIT = "k"; // ls size
    private static final String noRECURSIVE = "noRecursive";
    private static final String noTIME = "noTime";

    /*
     * 大小文字を区別しないで前方一致で比較
     */
    private boolean applyOption(String name, boolean... value) {
        boolean val = value.length == 0; // 省略時は真
        String key = name.toLowerCase();
        String key2 = key.startsWith("no") ?
                '-' + key.substring(2) : key;
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            if (key.startsWith(x) || key2.startsWith(x)) {
                optIn.add(name);
                optAll.put(name, val);
                return val;
            }
        }
        optAll.put(name, !val);
        return !val;
    }

    private static final Pattern IS_SET = Pattern
            .compile("(\\w+)=([-+]?\\d+)");

    private int applyShellVariable(String name, int value) {
        String key = name.toLowerCase();
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            Matcher m = IS_SET.matcher(x);
            while (m.find()) {
                String g1 = getValue(m.group(1));
                if (key.startsWith(g1)) {
                    String g2 = getValue(m.group(2));
                    int val = Integer.parseInt(g2);
                    String var = name + '=' + g2;
                    optIn.add(BLUE + var + RESET);
                    return val;
                }
            }
        }
        optIn.add(name + '=' + value);
        return value;
    }

    private boolean getBooOption(String key) {
        if (optAll.containsKey(key)) {
            Object o = optAll.get(key);
            if (o instanceof Boolean e)
                return e;
        }
        throw new RuntimeException(exception("No option", key));
    }

    private void resetOptions(String... x) {
        for (String key : x) {
            if (optAll.containsKey(key)) {
                optAll.put(key, false);
                optIn.remove(key);
            } else {
                throw new RuntimeException(exception("No option", key));
            }
        }
    }

    private String listOptions() {
        StringBuilder sb = new StringBuilder(64);
        sb.append(BLUE);
        for (String x : optIn) {
            sb.append(' ').append(x);
        }
        return sb.append(RESET).toString();
    }

    // System フォルダをスキップする
    private static boolean isSystem(Path path) {
        String str = path.toString();
        return str.contains("System Volume Information") ||
                str.contains("$RECYCLE.BIN");
    }

    private static boolean isSystem(File path) {
        return isSystem(path.toPath());
    }

    // ワイルドカードにマッチング
    private boolean isMatch(Path path) {
        return isMatch(path.toFile());
    }

    private boolean isMatch(File path) {
        if (path.exists() && !path.isDirectory()) {
            if (ana.alwaysTrue) return true;
            String name = path.getName();
            return ana.regex.matcher(name).matches();
        }
        return false;
    }

    /**
     * コマンド起動(start)以降に作成さた空フォルダを削除
     */
    private boolean removeStartFolder(Path path) {
        if (Files.isDirectory(path)) {
            boolean started = getModifiedTime(path).toMillis() >= start;
            return started && removeSafetyFolder(path);
        }
        return false;
    }

    /**
     * Time0 の空フォルダ以外を削除 - Other than Time0.
     */
    private boolean removeNoTime0Folder(Path path) {
        if (Files.isDirectory(path)) {
            boolean time0 = getModifiedTime(path).toMillis() != 0;
            return time0 && removeSafetyFolder(path);
        }
        return false;
    }

    /**
     * throw を発生させない (フォルダが空でない場合の対策)
     */
    private boolean removeSafetyFolder(Path path) {
        File file = path.toFile();
        return isRemove && !isSystem(path) &&
                Files.isDirectory(path) && file.delete();
    }

    /**
     * throw を発生させない (フォルダが空でない場合の対策)
     */
    private boolean removeSafetyFile(Path path) {
        File file = path.toFile();
        return isRemove && !isSystem(path) &&
                file.isFile() && file.delete();
    }

    /**
     * ファイルサイズを取得
     */
    private long getFileSize(Path path) {
        try {
            return Files.size(path);
        } catch (IOException e) {
            throw new RuntimeException(exception("getFileSize", e));
        }
    }

    /**
     *　更新日時を取得
     */
    private FileTime getModifiedTime(Path path) {
        try {
            return Files.getLastModifiedTime(path);
        } catch (IOException e) {
            throw new RuntimeException(exception("getModifiedTime", e));
        }
    }

    // 更新日時を設定
    private void setModifiedTime(Path path, FileTime time) {
        try {
            Files.setLastModifiedTime(path, time);
        } catch (IOException e) {
            throw new RuntimeException(exception("setModifiedTime", e));
        }
    }

    /**
     * 出力側のホルダを作成し、目印として Time0 日時を設定
     *
     * @param input null を許容する
     */
    private boolean createFolder(Path input, Path output) {
        try {
            FileTime time = getModifiedTime(input);
            mod.put(output, time);
            if (Files.notExists(output)) {
                Files.createDirectories(output); // ホルダを作成
                if (time.toMillis() == 0) {
                    setModifiedTime(output, time); // Time0 日時を設定
                }
                return true;
            }
        } catch (IOException e) {
            throw new RuntimeException(exception("Make directory", e));
        }
        return false;
    }

    private static final boolean _STDOUT = true;
    private static final boolean _STDERR = false;

    /**
     * @param stdout [Default value: hasSTDOUT] else, hasSTDERR
     */
    private void printX(boolean stdout, String x) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        BiIO.print(re.rid, re.file, x);
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, USE_PATH_NAME
     */
    @SuppressWarnings("SameParameterValue")
    private void printX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.redirect) {
            printX(stdout, applySlashSeparator(file, type));
        } else {
            printX(stdout, pack(file, type));
        }
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value: path value] else, USE_PATH_NAME
     */
    @SuppressWarnings("SameParameterValue")
    private String sprintX(boolean stdout, File file, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.redirect) {
            return applySlashSeparator(file, type);
        } else {
            return pack(file, type);
        }
    }

    /**
     * set - シェル変数の設定.
     */
    public int set(Object... x) {
        initialize("./", x);
        boolean hasARGV = optARGV.size() > 0;
        DISPLAY_WIDTH = applyShellVariable(DISPLAY_WIDTH_KEY, DISPLAY_WIDTH);
        MAX_PATH = applyShellVariable(MAX_PATH_KEY, MAX_PATH);
        String args = hasARGV ? listOptions() : "";
        messageTitle("set", args);
        if (!hasARGV) {
            for (String k : optIn) {
                System.out.println(k);
            }
        }
        messageFinish("Number of processed", optIn.size(), "");
        return STATUS_NORMAL;
    }

    /**
     * clean - クリーンアップ.
     */
    public int clean(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isClean = applyOption(CLEAN);
        boolean isTime = applyOption(noTIME, false);
        int flags = isTime ? CLN_MOD_TIME : 0; // 日時を設定
        if (isClean) flags |= CLN_EMPTY_FOLDER; // 空のフォルダを削除
        String args = ana.virtualPath + ' ' + listOptions();
        messageTitle("clean", args);
        int[] cf = new int[2];
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            cleanupFolderImpl(in, cf, true, flags);
        }
        messageFinish("Number of processed",
                cf[clean_FILE], cf[clean_FOLDER]);
        return STATUS_NORMAL;
    }

    private static final int CLN_MOD_TIME = 1; // 日時を設定 (flags)
    private static final int CLN_EMPTY_FOLDER = 2; // 空のフォルダを削除

    private static final int clean_FILE = 0; // cf のインデックス
    private static final int clean_FOLDER = 1;

    private long cleanupFolderImpl(Path input, int[] cf,
                                   boolean warning, int flags) {
        if (isSystem(input)) return 0; // skip System folder
        File inputFile = input.toFile();
        boolean isTime = (flags & CLN_MOD_TIME) != 0;
        boolean isEmpty = (flags & CLN_EMPTY_FOLDER) != 0;
        long modTime = 0;
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    long time;
                    if (Files.isDirectory(path)) {
                        time = cleanupFolderImpl(path, cf, warning, flags);
                        if (warning && !isEmpty && time == 0 && Files.isDirectory(path))
                            messageMAGENTA("Empty folder", pack(path.toFile()));
                        cf[clean_FOLDER] += 1;
                    } else {
                        time = getModifiedTime(path).toMillis();
                        cf[clean_FILE] += 1;
                    }
                    if (modTime < time) modTime = time;
                }
                if (Files.isDirectory(input)) { // 更新日時を設定
                    long time2 = getModifiedTime(input).toMillis();
                    if (isTime && time2 != modTime) {
                        FileTime fileTime = FileTime.fromMillis(modTime);
                        setModifiedTime(input, fileTime);
                    }
                    if (isEmpty && removeNoTime0Folder(input)) { // Other than Time0
                        if (warning)
                            messageMAGENTA("remove", pack(inputFile));
                        cf[clean_FOLDER] -= 1;
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("cleanup", e));
            }
        }
        return modTime;
    }

    /**
     * copy the file.
     * - removeはルートの親フォルダを削除するため、copyのフォルダ数 +1 となる
     * (※ PATCH, copy側で補正する)
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int copy(String input, String output, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        Output cf = createOutFolder(in, output);
        Path out = cf.path;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isSync = applyOption(SYNC);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' +
                pack(output) + listOptions();
        messageTitle("copy", args);
        if (in.equals(out)) { // same path
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        int ro = checkPath(out, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        ri = Math.max(ri, ro);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDERR, isPro);
            if (isSync) {
                copySync(pro, in, out); // synchronous copy
                pro.flush();
                if (pro.fileNumber > 0) {
                    if (pro.printX) {
                        for (File sync : set) {
                            printX(_STDERR, sync);
                        }
                    }
                    messageCYAN("Synchronized", pro.fileNumber);
                }
            }
            pro = new Progress(_STDOUT, isPro);
            if (cf.createHolder) pro.incrementFolder(); // ※ PATCH
            copyImpl(pro, in, out, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (File copy : set) {
                    printX(_STDOUT, copy);
                }
            }
            finishAsyncIO(pro, out);
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void copyImpl(Progress pro,
                          Path input, Path output, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    Path newOut = output.resolve(path.getFileName());
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            if (createFolder(path, newOut)) // ※
                                pro.incrementFolder();
                            //noinspection ConstantValue
                            copyImpl(pro, path, newOut, recursive);
                        }
                    } else if (isMatch(path)) {
                        atomicCopy(pro, path, newOut);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("copy", e));
            }
        } else if (isMatch(input) && Files.exists(input)) {
            atomicCopy(pro, input, output); // ファイル名指定で呼ばれたとき
        }
    }

    /**
     * Synchronous copy
     */
    private void copySync(Progress pro, Path input, Path output) {
        Map<String, File> map = new HashMap<>(512);
        if (isSystem(input) || isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) { // path
            File[] files = output.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    map.put(file.getName(), file);
                }
            }
        }
        if (Files.exists(output)) {
            File file = output.toFile();
            map.put(file.getName(), file);
        }
        if (Files.isDirectory(input)) { // input
            File[] files = input.toFile().listFiles();
            if (null != files) {
                for (File file : files) {
                    if (isSystem(file)) continue; // skip System folder
                    String name = file.getName();
                    Path newIn = file.toPath();
                    Path newOut = output.resolve(name);
                    map.remove(name);
                    if (file.isDirectory()) {
                        copySync(pro, newIn, newOut);
                    }
                }
            }
        }
        if (Files.exists(input)) {
            String name = input.toFile().getName();
            map.remove(name);
        }
        for (File file : map.values()) {
            copySyncRemove(pro, file.toPath()); // Synchronous
        }
    }

    /**
     * Synchronous copy - 指定されたパスを無条件に削除
     *
     * @param input 　file or folder
     */
    private void copySyncRemove(Progress pro, Path input) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        copySyncRemove(pro, path);
                        pro.incrementFolder();
                    } else if (isMatch(path)) {
                        long size = getFileSize(path);
                        if (removeSafetyFile(path)) {
                            pro.addFile(path, size);
                        }
                    }
                }
                removeSafetyFolder(input); // ※親フォルダを削除

            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("sync", e));
            }
        } else if (Files.exists(input)) {
            long size = getFileSize(input);
            if (removeSafetyFile(input)) // ファイル名指定で呼ばれたとき
                pro.addFile(input, size);
        }
    }

    /**
     * move the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int move(String input, String output, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        Output cf = createOutFolder(in, output);
        Path out = cf.path;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' +
                pack(output) + listOptions();
        messageTitle("move", args);
        if (in.equals(out)) { // same path
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        int ro = checkPath(out, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        ri = Math.max(ri, ro);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDOUT, isPro);
            if (cf.createHolder) pro.incrementFolder(); // ※ PATCH
            moveImpl(pro, in, out, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (File move : set) {
                    printX(_STDOUT, move);
                }
            }
            finishAsyncIO(pro, out);
            finishAsyncDropFileMove(in);
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void moveImpl(Progress pro,
                          Path input, Path output, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    Path newOut = output.resolve(path.getFileName());
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            if (createFolder(path, newOut)) // ※
                                pro.incrementFolder();
                            //noinspection ConstantValue
                            moveImpl(pro, path, newOut, recursive);
                        }
                    } else if (isMatch(path)) {
                        atomicMove(pro, path, newOut);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("move", e));
            }
        } else if (isMatch(input) && Files.exists(input)) {
            atomicMove(pro, input, output); // ファイル名指定で呼ばれたとき
        }
    }

    /**
     * 単一のファイルを差分コピー (出力側に存在しない、日付時刻・サイズが異なる場合)
     */
    private void atomicCopy(Progress pro, Path input, Path output) {
        Action act = isAtomicSameFile(input, output);
        output = act.path;
        if (output == null ||
                act.action == ACT_SAME_FILE) return;

        atom.getAndIncrement();
        pool.submit(new AsyncIO(pro, _COPY, act.action, input, output));
    }

    /**
     * ファイルをターゲット・ファイルに移動するか、そのファイル名を変更
     */
    private void atomicMove(Progress pro, Path input, Path output) {
        Action act = isAtomicSameFile(input, output);
        output = act.path;
        if (output == null) return;

        atom.getAndIncrement();
        pool.submit(new AsyncIO(pro, _MOVE, act.action, input, output));
        if (act.action == ACT_SAME_FILE) // move を Skip
            drop.add(input); // 入力ファイルを削除依頼
    }

    //////////////////////////////////////////////////////////////////////
    // Action.                          action     path
    private static final int ACT_SKIP = -2;      // null
    private static final int ACT_SAME_PATH = -1; // null
    private static final int ACT_SAME_FILE = 0;  // path (use,move)
    private static final int ACT_ACCEPT = 1;     // path
    private static final boolean _COPY = true;   // use Async io
    private static final boolean _MOVE = false;

    private static final Action ACTION_HAS_SKIP = new Action(ACT_SKIP, null);
    private static final Action ACTION_HAS_SAME_PATH = new Action(ACT_SAME_PATH, null);

    // FAT の時間誤差を除外
    private static final long FAT_TIME_ERROR = 2000; // 2sec.

    /**
     * ファイル属性が同一かどうかの判定
     *
     * @param input  パス(ファイル名)
     * @param output パス(ファイル名) or フォルダ
     * @return Action 状態コード、出力パス(null: 要件に該当しないとき、skip)
     */
    private Action isAtomicSameFile(
            Path input, Path output) {
        if (!Files.exists(input) || Files.isDirectory(input))
            throw new RuntimeException(
                    exception("File does not exist", input));
        if (!isMatch(input) || isSystem(input)) // wildcard mismatch
            return ACTION_HAS_SKIP; // wildcard unmatched, skip System folder
        if (Files.isDirectory(output)) {  // out がフォルダならファイル名を設定
            output = output.resolve(input.getFileName());
        }
        if (input.equals(output)) {
            inputAndOutputAreSamePath(input);
            return ACTION_HAS_SAME_PATH;
        }

        Path outParent = getParent(output); // out の親を取得
        if (outParent != null && !Files.exists(outParent)) {
            Path inParent = getParent(input); // in の親を取得
            if (inParent != null) {
                createFolder(inParent, outParent); // ※
            }
        }
        if (Files.exists(output) && !Files.isDirectory(output) &&
                input.getFileName().equals(output.getFileName())) {
            long iSize = getFileSize(input); // Check file attributes
            long oSize = getFileSize(output);
            long iMod = getModifiedTime(input).toMillis() / FAT_TIME_ERROR;
            long oMod = getModifiedTime(output).toMillis() / FAT_TIME_ERROR;
            if (iSize == oSize && iMod == oMod) { // FAT(誤差2sec) <> NTFS
//                messageGREEN("Same file", input);
                return new Action(ACT_SAME_FILE, output);
            }
        }
        return new Action(ACT_ACCEPT, output);
    }

    /**
     * remove - 指定されたパス以下にあるワイルドカードに一致するファイルを削除する
     * - Delete files matching wildcards under the specified path
     * - Path, Files version
     * - removeはルートの親フォルダを削除するため、copyのフォルダ数 +1 となる
     * (※ PATCH, copy側で補正する)
     *
     * @param input Requires at least one sub-path
     */
    public int remove(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isRoot = applyOption(ROOT);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("remove", args);
        int check = CHK_DIRECTORY | CHK_FILE;
        if (isRoot) check |= CHK_ROOT;
        int ri = checkPath(in, check);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDOUT, isPro);
            removeImpl(pro, in, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (File remove : set) {
                    if (!remove.isDirectory())
                        printX(_STDOUT, remove);
                }
            }
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void removeImpl(Progress pro, Path input,
                            boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            removeImpl(pro, path, recursive);
                        }
                    } else if (isMatch(path)) {
                        long size = getFileSize(path);
                        if (removeSafetyFile(path)) { // ファイルを削除
                            pro.addFile(path, size);
                        }
                    }
                }
                if (removeNoTime0Folder(input)) { // Other than Time0
                    pro.incrementFolder();
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("remove", e));
            }
        } else if (isMatch(input) && Files.exists(input)) {
            long size = getFileSize(input);
            if (removeSafetyFile(input)) // ファイル名指定で呼ばれたとき
                pro.addFile(input, size);
        }
    }

    /**
     * tree - File version.
     *
     * @param input Path to start listing
     */
    public int tree(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isFile = applyOption(FILE);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("tree", args);
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            int max = treeImpl(in.toFile(), "", isFile, isRecursive);
            messageFinish("MAX_PATH", max, " chars");
        }
        close();
        return ri;
    }

    @SuppressWarnings("ConstantValue")
    private int treeImpl(File input, String indent, boolean isFile, boolean recursive) {
        if (isSystem(input)) return 0; // skip System folder
        int max = 0;
        if (input.isDirectory()) {
            File[] files = input.listFiles();
            if (null != files) {
                TreeSet<File> set = new TreeSet<>(List.of(files));
                for (File file : set) { // ファイル処理
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isFile() && isMatch(file)) {
                        if (isFile) {
                            String x = indent + "| " + packTree(file);
                            printX(_STDOUT, x);
                        }
                        max = Math.max(max, file.getPath().length());
                    }
                }
                for (File file : set) { // フォルダ処理
                    if (isSystem(file)) continue; // skip System folder
                    if (file.isDirectory()) {
                        String x = indent + '/' + packTree(file);
                        printX(_STDOUT, x);
                        max = Math.max(max, file.getPath().length());
                        if (recursive) {
                            max = Math.max(max, treeImpl(file,
                                    indent + " ", isFile, recursive));
                        }
                    }
                }
            } else if (isMatch(input)) {
                if (isFile) {
                    String x = indent + "| " + packTree(input);
                    printX(_STDOUT, x);
                }
                max = input.getPath().length();

            }
//        } else { // ワーニングメッセージ出力済み
        }
        return max;
    }

    /**
     * ls - list segments.
     *
     * @param input Path to start listing
     */
    public int ls(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isPath = applyOption(PATH);
        boolean isSimple = applyOption(SIMPLE);
        boolean isComma = applyOption(COMMA);
        boolean isUnit = applyOption(UNIT);
        boolean is260 = applyOption(MX_260);
        boolean isNo260 = applyOption(MX_no260);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        boolean isAttr = isPath || isSimple || isComma || isUnit;
        if (isPath) {
            resetOptions(SIMPLE, COMMA, UNIT); // オプションの無効化
        }
        String args = ana.virtualPath + listOptions();
        messageTitle("ls", args);
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            lsImpl(in, is260, isNo260, isRecursive);
            for (File file : set) { // パスを切り詰めないで表示
                String ls = lsAttr(file, isAttr);
                printX(_STDOUT, ls);
            }
            messageFinish("Number of processed", set.size(), "");
        }
        close();
        return ri;
    }

    private void lsImpl(Path input,
                        boolean is260, boolean isNo260, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            lsImpl(path, is260, isNo260, recursive);
                        }
                    } else {
                        lsSelect260(path, is260, isNo260);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("ls", e));
            }
//        } else { // ワーニングメッセージ出力済み
        }
    }

    private String lsAttr(File input, boolean isAttr) {
        StringBuilder sb = new StringBuilder(256);
        if (isAttr) {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yy/MM/dd HH:mm");
            long time =getModifiedTime(input.toPath()).toMillis();
            String daytime = sdf.format(time);
            String len = lsLength(
                    getBooOption(PATH) ?
                            input.getPath().length() : // path length
                            input.length()); // file size
            sb.append(daytime).append('\t');
            sb.append(len).append('\t');
        }
        String path = sprintX(_STDOUT, input);
        return sb.append(path).toString();
    }

    /**
     * Output long path
     *
     * @param is260   Output paths longer than 260
     * @param isNo260 Output paths under 260
     */
    private void lsSelect260(Path input,
                             boolean is260, boolean isNo260) {
        if (isMatch(input)) {
            File in = input.toFile();
            if (is260) {
                if (isLonger260(input)) set.add(in);
            } else if (isNo260) {
                if (!isLonger260(input)) set.add(in);
            } else {
                set.add(in);
            }
        }
    }

    private String lsLength(long len) {
        if (getBooOption(PATH)) {
            return String.format("%, 7d", len); // max 32k, 6 char
        }
        if (getBooOption(COMMA)) {
            return String.format("%,d", len);
        }
        if (getBooOption(UNIT)) {
            return formatUnit(len);
        }
        return Long.toString(len);
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static final String NUMBER_UNIT = "BKMGTPEZY";

    /**
     * @return 前後空白に挟まれているため、注意！
     */
    private static String formatUnit(long length) {
        double len = length;
        for (int i = 0; i < NUMBER_UNIT.length(); i++) {
            if (len < 1024.) {
                String unit = Character.toString(NUMBER_UNIT.charAt(i));
                return String.format("%3.1f %s", len, unit);
            }
            len /= 1024.;
        }
        return Long.toString((long) len);
    }

    private static final boolean USE_PATH_NAME = true; // Tree でファイル名を表示

    /**
     * 切り詰めたファイル名・IDを返す (returns a truncated path/filename)
     */
    private String packTree(File input) {
        StringBuilder sb = new StringBuilder(DISPLAY_WIDTH + 32);
        sb.append(sprintX(_STDOUT, input, USE_PATH_NAME)).append(' ');
        String path = input.getPath();
        String name = input.getName();
        String info = name.length() + "/" + path.length();
        String maxPathID = isLonger260(input.toPath()) ? " *" : "";
        sb.append(maxPathID.isEmpty() ? info :
                color(MAGENTA, info + maxPathID));
        return sb.toString();
    }

    /**
     * パスを切り詰める
     *
     * @param type [Default value: path value] else, USE_PATH_NAME
     */
    private String pack(File file, boolean... type) {
        String str = applySlashSeparator(file, type);
        return pack(str, DISPLAY_WIDTH);
    }

    private String pack(String source) {
        return pack(source, DISPLAY_WIDTH);
    }

    /**
     * Analysis コンストラクタから呼ばれるため　static 必須
     * ※ ターミナルでは漢字は、全角表示される(下記の式では対応できない)
     *
     * @param DISPLAY_WIDTH 画面への表示幅
     */
    private static String pack(String source, int DISPLAY_WIDTH) {
        int len = source.length();
        if (len > DISPLAY_WIDTH) {
            int half = DISPLAY_WIDTH / 2;
            return source.substring(0, half) + "…" +
                    source.substring(len - half + 1);
        }
        return source;
    }

    /**
     * パス区切り文字を変換
     *
     * @param type [Default value: path value] else, USE_PATH_NAME
     */
    private static String applySlashSeparator(File input, boolean... type) {
        if (type.length != 0)
            return input.getName(); // これは '\'を含まない
        String path = input.getPath();
        return File.separatorChar == '/' ? path : // UNIX
                path.replace('\\', '/'); // Windows
    }

    //////////////////////////////////////////////////////////////////////
    // Message
    static String color(String color, String message) {
        return color + message + RESET;
    }

    private static void messageTitle(String name, Object arg) {
        System.out.println(color(YELLOW, name + ' ') + arg);
    }

    private void messageFinish(String name, int arg1, Object arg2) {
        String cyan = color(CYAN, name + ": ");
        String ela = elapsedTime(System.currentTimeMillis() - start);
        String str1 = String.format(", %,d", arg1);
        String str2;
        if (arg2 instanceof Long e)
            str2 = " (" + formatUnit(e).trim() + ')'; // need trim
        else if (arg2 instanceof Integer e)
            str2 = String.format(" (%,d)", e);
        else str2 = arg2.toString();
        String foo = cyan + ela + str1 + str2;
        System.err.println(foo);
    }

    private static String elapsedTime(long millis) {
        Calendar cl = Calendar.getInstance();
        cl.setTimeInMillis(millis); // FIXME GMTより9時間進んでるはず！
        cl.set(Calendar.ZONE_OFFSET, -6 * 60 * 60 * 1000);
        SimpleDateFormat sdf = new SimpleDateFormat("H:m:s.SSS");
        return sdf.format(cl.getTime());
    }

    @SuppressWarnings("SameParameterValue")
    private static void messageCYAN(String name, Object arg) {
        System.err.println(color(CYAN, name + ": ") + arg);
    }

    @SuppressWarnings("unused, SameParameterValue")
    private static void messageGREEN(String name, Object arg) { // debug
        System.err.println(color(GREEN, name + ": ") + arg);
    }

    @SuppressWarnings("unused")
    private static void messageBLUE(String name, Object arg) {
        System.err.println(color(BLUE, name + ": ") + arg);
    }

    @SuppressWarnings("SameParameterValue")
    private static void messageMAGENTA(String name, Object arg) {
        System.err.println(color(MAGENTA, name + ": ") + arg);
    }

    private static void messageRED(String name, Object arg) {
        System.err.println(color(RED, name + ": ") + arg);
    }

    private String exception(String name, String str) {
        return color(RED, name + ": ") + str;
    }

    @SuppressWarnings("SameParameterValue")
    private String exception(String name, Path path) {
        return color(RED, name + ": ") + pack(path.toFile());
    }

    private String exception(String name, Throwable e) {
        return color(RED, name + ": ") + e;
    }

    // ルートは指定できません
    private static void rootCannotBeSpecified(Path path) {
        messageRED("Root folder cannot be specified",
                applySlashSeparator(path.toFile()));
    }

    // 入力と出力は同じファイル
    private static void inputAndOutputAreSamePath(Path path) {
        messageRED("Input and path are the same path",
                applySlashSeparator(path.toFile()));
    }

    // 間違ったパス指定
    private static void wrongPathSpecified(Path path) {
        messageMAGENTA("Wrong path specified",
                applySlashSeparator(path.toFile()));
    }

    private static final String RESET = "\033[m";       // \033[0m
    private static final String RED = "\033[91m";       // error
    private static final String GREEN = "\033[92m";     // debug
    private static final String YELLOW = "\033[93m";    // title
    private static final String BLUE = "\033[94m";      // options
    private static final String MAGENTA = "\033[95m";   // warning
    private static final String CYAN = "\033[96m";      // information

    //////////////////////////////////////////////////////////////////////
    // Analysis
    private static class Analysis {
        private static final Pattern IS_PATH_HAS_FILE_UNIX = Pattern
                .compile("(.*/)*([^/]*)$");
        private static final Pattern IS_PATH_HAS_FILE_WIN = Pattern
                .compile("(.*[/\\\\])*([^/\\\\]*)$");
        private static final Pattern SPLIT_PATH = // / で終了するかどうか
                File.separatorChar == '/' ?
                        IS_PATH_HAS_FILE_UNIX :
                        IS_PATH_HAS_FILE_WIN;
        private static final Pattern WILD_CARD_ALL = Pattern
                .compile(".*");
        private final String virtualPath; // path / WildCard
        private final Path path; // ワイルドカードを除いた実際のパス
        private final Pattern regex; // ワイルドカード
        private final boolean alwaysTrue; // ワイルドカードが常に真かどうか

        private Analysis(String input, int DISPLAY_WIDTH) {
            String path = input.trim();
            String wild = "";
            Pattern regex = WILD_CARD_ALL;
            Matcher m = SPLIT_PATH.matcher(path);
            if (m.matches()) {
                String g2 = getValue(m.group(2).trim());
                if (hasWildcard(g2)) {
                    path = getValue(m.group(1)).trim();
                    wild = g2;
                    regex = mkWildcard(g2);
                }
            }
            if (path.isEmpty()) path = "./";
            this.path = Path.of(path);
            this.virtualPath = pack(path, DISPLAY_WIDTH) +
                    (wild.isEmpty() ? "" : NanoTools.color(NanoTools.BLUE, wild));
            this.regex = regex;
            alwaysTrue = WILD_CARD_ALL.equals(regex);
        }

        /**
         * hasWildcard - ワイルドカード指定かどうかを返す
         * - wildcard: * ? | endsWith .
         */
        private static boolean hasWildcard(String wild) {
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                if (0 <= "*?|".indexOf(c)) return true;
            }
            return wild.endsWith(".");
        }

        /**
         * mkWildcard - 大小文字を区別せずにマッチする (Case-sensitive)
         * ワイルドカード (* ? .) 以外の記号は、正規表現として使用できる
         * e.g. /[fb]*.txt|*.log/
         *
         * @param wild wildcard
         */
        private static Pattern mkWildcard(String wild) {
            StringBuilder sb = new StringBuilder(128);
            if (wild.endsWith("."))
                wild = wild.substring(0, wild.length() - 1); // Delete the last .
            if (wild.isEmpty()) wild = "*"; // Default value
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                switch (c) {
                    case '*' -> sb.append(".*");
                    case '?' -> sb.append('.');
                    case '.' -> sb.append("\\.");
                    default -> sb.append(c);
                }
            }
            return Pattern.compile(sb.toString(), Pattern.CASE_INSENSITIVE);
        }

    }

    /**
     * Progress (Verbose)
     */
    private class Progress {
        private final PrintStream printStream;
        private final boolean enable;
        private final boolean printX; // printXを使用するかどうか
        private volatile boolean hasPrint; // printしたかどうか
        private volatile int folderNumber; // number of holders
        private volatile int fileNumber; // number of files
        private volatile long fileSize; // size of files (bytes)
        private volatile File file;

        private Progress(boolean stdout, boolean enable) {
            this.printStream = stdout ? System.out : System.err;
            this.enable = enable;
            Redirect re = stdout ? __STDOUT : __STDERR; // printXの活性化
            if (re.redirect) this.printX = true; // リダイレクトのとき
            else this.printX = !enable; // ディセーブルのとき
            set.clear(); // ※ 出力リスト
        }

        private void flush() {
            if (enable && atom.intValue() <= 0) {
                print();
                if (hasPrint)
                    printStream.println();
            }
        }

        private void print() {
            if (enable && file != null) {
                printStream.print(this);
                hasPrint = true;
            }
        }

        private static final long ELAPSED = 500; // ms.
        private volatile long elapse = System.currentTimeMillis();

        private synchronized void incrementFolder() {
            folderNumber += 1;
        }

        private synchronized void decrementFolder() {
            folderNumber -= 1;
        }

        private synchronized void addFile(Path input, Path output) {
            addFile(input, getFileSize(output));
        }

        private synchronized void addFile(Path input, long size) {
            this.file = input.toFile();
            if (printX) set.add(this.file); // ※ 出力リスト
            fileSize += size;
            fileNumber += 1;
            long now = System.currentTimeMillis();
            if ((now - elapse) > ELAPSED) {
                elapse = now;
                print();
            }
        }

        // \033[nK 	カーソルより後ろを消去
        // \r       先頭に移動
        // \033[nA  上にn移動
        @Override
        public String toString() {
            String elapsed = elapsedTime(System.currentTimeMillis() - start);
            String progress = String.format("%s %,d(%,d): ",
                    elapsed, fileNumber, folderNumber);
            return color(BLUE, progress) +
                    pack(file) + "\033[K\r";
        }
    }

    /**
     * Async I/O.
     */
    private class AsyncIO implements Runnable {
        private static final int RETRY = 5; // number of retries
        private final Progress pro;
        private final boolean copy;
        private final int action;
        private final Path input;
        private final Path output;
        private Throwable ex;

        private AsyncIO(Progress pro, boolean copy, int action,
                        Path input, Path output) {
            this.pro = pro;
            this.copy = copy;
            this.action = action;
            this.input = input;
            this.output = output;
        }

        public void run() {
            if (copy) copy();
            else
                move();
            atom.decrementAndGet();
        }

        private void copy() {
            for (int i = RETRY; i > 0; i--) { // retry
                try {
                    // move と異なり同一ファイル判定が有効なようだ！
                    Files.copy(input, output, COPY_ATTRIBUTES, REPLACE_EXISTING);
                    pro.addFile(input, output);
                    return;
                } catch (IOException e) {
                    messageMAGENTA("copy", pack(input.toFile()));
                    ex = e;
                }
            }
            throw new RuntimeException(exception("copy", ex));
        }

        private void move() {
            for (int i = RETRY; i > 0; i--) { // retry
                try {
                    if (action != ACT_SAME_FILE) { // 出力先は同一ファイル
                        Files.move(input, output, REPLACE_EXISTING);
                    }
                    // input はパスが必要なため、削除されても良い
                    pro.addFile(input, output);
                    return;
                } catch (IOException e) {
                    messageMAGENTA("move", pack(input.toFile()));
                    ex = e;
                }
            }
            throw new RuntimeException(exception("move", ex));
        }
    }

    //////////////////////////////////////////////////////////////////////
    // Action
    private record Action(int action, Path path) {
    }

    // Output
    private record Output(Path path, boolean createHolder) {
    }

    // Redirect
    private record Redirect(boolean redirect, String name,
                            String rid, String file, boolean isErr) {
    }

    private static String getValue(String x) {
        return x == null ? "" : x;
    }
}