/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import plus.util.Debug;

/**
 * Parser - Type NumHelper.
 *
 * @author kunio himei.
 */
final class Type {

    /**
     * Don't let anyone instantiate this class.
     */
    private Type() {
        super();
    }

    /**
     * ネストしたリストを剥がす.
     */
    @SuppressWarnings("unchecked")
    static Object unWrapList(Object e) {
        Object x = e;
        while (true) {
            if ((x instanceof Object[]) && (1 == ((Object[]) x).length)) {
                x = ((Object[]) x)[0]; // Tail recursive call
            } else if ((x instanceof LexArray)
                    && (1 == ((LexArray<Object>) x).size())) {
                x = ((LexArray<Object>) x).get(0); // Tail recursive call
            } else {
                return x;
            }
        }
    }

    /**
     * ラップした 配列オブジェクトを返す.
     */
    @SuppressWarnings("unchecked")
    public static Object[] castWrapArray(Object e) {
        Object x = unWrapList(e);
        if (x instanceof LexArray) {
            return ((LexArray<Object>) x).toArray();
        } else if (x instanceof Object[]) {
            return (Object[]) x;
        }
        return new Object[]{x};
    }

    /**
     * Node 属性を返す.
     */
    public static int getNodeType(Object e) {
        Object x = e;
        while (true) {
            if ((x instanceof Object[]) && (0 == ((Object[]) x).length)) {
                return Flags.T00VOID;
            } else if (x instanceof Node.Ass) {
                return ((Node.Ass) x).nType;
            } else if (x instanceof Node.Call) {
                Node.Call xx = (Node.Call) x;
                return Symbols.findType(xx.name) ? Symbols.getType(xx.name)
                        : xx.nType;
            } else if (x instanceof Node.Func) { // 関数定義
                return ((Node.Func) x).nType;
            } else if ((x instanceof Node.If)
                    && (Keyword.SyyQIF == ((Node.If) x).id)) {
                Node.If xx = (Node.If) x;
                return getNodeType(xx.left) | getNodeType(xx.right);
            } else if (x instanceof Node.Stmt) { // exit, return.
                return ((Node.Stmt) x).nType;
            } else if (x instanceof Node.YyCall) { // `#invoke`
                return ((Node.YyCall) x).nType;
            } else if (x instanceof Node.YyVariable) { // Arr, NAME.
                return Symbols.getType(((Node.YyVariable) x).name);
            } else if (x instanceof Node.YyeValue) { // Bb, Bc, Cb, Cc.
                return ((Node.YyeValue) x).nType;
            } else if (x instanceof Term.YyValue) { // NUMBER, REGEXP, STRING.
                return ((Term.YyValue) x).nType;
            } else if (x instanceof Object[]) {
                x = ((Object[]) x)[0];
            } else if (x instanceof Node.YyNop || // 注釈 (Annotation).
//                    x instanceof Keyword ||
                    x instanceof String ||
                    null == x) {
                return Flags.T00VOID;
            } else {
                System.err.println("Type.getNodeType: " + x.getClass() + " " + x);
                throw new IllegalStateException("unmatch: " + Debug.objectOf(x));
            }
        }
    }

    /**
     * 空かどうかを返す.
     */
    static boolean isEmpty(Object a) {
        return 0 == castWrapArray(a).length;
    }
}