/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.util;

import org.jetbrains.annotations.NotNull;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * [%helper%] Built-in NumHelper.
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public final class NumHelper {

    //* 復帰値: 数値, FALSE(0)
    private static final Integer C_INTEGER0 = 0;
    //* 復帰値: 数値, TRUE(1)
    private static final Integer C_INTEGER1 = 1;
    //* Hex decimal 2 進数.
    private static final int TO_BIN = 2;
    //* Octal decimal 8 進数.
    private static final int TO_OCT = 8;
    //* Hex decimal 16 進数.
    private static final int TO_HEX = 16;

    //* 10 進数値を表す正規表現(decimal digit).
    private static final Pattern RX_DECIMAL_NUMBER = Pattern
            .compile("^([-+]?(?:\\d+[.]?\\d*|[.]\\d+)"
                    + "(?:[eE][-+]?\\d+)?)" + "[dflDFL]?"
                    + "(.*)");
    //* 16 進数値を表す正規表現(hex digit).
    private static final Pattern RX_HEX_NUMBER = Pattern
            .compile("^0[Xx]([0-9a-fA-F]+)(.*)");
    //* 8 進数値を表す正規表現(octal digit).
    private static final Pattern RX_OCT_NUMBER = Pattern
            .compile("^0([0-7]+)(.*)");
    //* 2 進数値を表す正規表現(binary digit).
    private static final Pattern RX_BIN_NUMBER = Pattern
            .compile("^0[Bb]([01]+)(.*)");

    //* RegExp: 数値を含むかどうかを返す正規表現.
    private static final Pattern RX_HAS_NUMBER = Pattern.compile("\\d+");
    //* RegExp: 数値を識別する正規表現.
    private static final Pattern RX_NUMBER = Pattern.compile(
            "^([-+]?(?:\\d*[.]?\\d*)(?:[dflDFL][-+]?\\d+)?)$");

    /**
     * 数値に一致するかどうかを返す.
     */
    public static boolean matchNumber(String x) {
        return RX_HAS_NUMBER.matcher(x).find() && RX_NUMBER.matcher(x).find();
    }

    /**
     * 文字列 を評価して Number を返す.
     */
    public static Number parseNumber(CharSequence x) {
        return Objects.requireNonNull(parseNumberImpl(x, false));
    }

    /**
     * 文字列 を評価して Number を返す.
     */
    public static Number parseNumberFull(CharSequence x) {
        return parseNumberImpl(x, true);
    }

    /**
     * 文字列を評価して Number を返す.
     */
    private static Number parseNumberImpl(CharSequence x, boolean isFull) {
        if (0 < x.length()) {
            if ('0' == x.charAt(0)) {
                Matcher m = RX_HEX_NUMBER.matcher(x); // ^0[Xx]([0-9a-fA-F]+)
                if (m.find() && (!isFull || m.group(2).isEmpty())) {
                    return Long.parseLong(m.group(1), TO_HEX);
                }
                m = RX_BIN_NUMBER.matcher(x); // ^0[Bb]([01]+)
                if (m.find() && (!isFull || m.group(2).isEmpty())) {
                    return Long.parseLong(m.group(1), TO_BIN);
                }
                m = RX_OCT_NUMBER.matcher(x); // ^0([0-7]+)
                if (m.find() && (!isFull || m.group(2).isEmpty())) {
                    return Long.parseLong(m.group(1), TO_OCT);
                }
            }
            Matcher m = RX_DECIMAL_NUMBER.matcher(x);
            if (m.find() && (!isFull || m.group(2).isEmpty()))
                return Double.parseDouble(m.group(1));
        }
        return (isFull) ? null : C_INTEGER0; // default value
    }

    /**
     * StringNumber を返す.
     */
    public static Object toStringNumber(Object value) {
        String x = Objects.toString(value, "");
        if (matchNumber(x)) {
            try {
                return NumHelper.normalise(Double.parseDouble(x));
            } catch (final NumberFormatException e) {
                // do nothing.
            }
        }
        return x;
    }

    /**
     * int を返す.
     */
    public static int intValue(Object x) {
        return (int) doubleValue(x);
    }

    /**
     * double を返す.
     */
    public static double doubleValue(Object x) {
        if (x instanceof Number) return ((Number) x).doubleValue();
        return numberValue(x).doubleValue();
    }

    /**
     * 正規化した数値を返す.
     */
    public static Number numberValue(Object x) {
        if (null == x)
            return C_INTEGER0;
        if (x instanceof Boolean) {
            return ((Boolean) x) ? C_INTEGER1 : C_INTEGER0;
        }
        if (x instanceof Number)
            return normalise((Number) x);
        Number xx = parseNumberImpl(x.toString(), false);
        if (null == xx) return C_INTEGER0;
        return normalise(xx);
    }

    /**
     * Integer、Long または Double に正規化した数値を返す.
     */
    public static Number normalise(double d) {
        long ll = (long) d;
        int ii = (int) d;
        if (0 != (d - ll))
            return d;
        if (0 != (ll - ii)) // NOTE 3項演算子では、機能しない.
            return ll;
        return ii;
    }

    /**
     * Integer または Double に正規化した数値を返す.
     */
    @NotNull
    public static Number normalise(Number number) {
        return normalise(number.doubleValue());
    }
}