/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#[macro_use]
extern crate lazy_static;
use chrono::{DateTime, Local};
use std::thread;
use std::time::Duration;
use std::env;
use std::path::Path;
use regex::Regex;

mod atomic;
mod daemon;
mod files;
mod iomod;
mod thmod;

const INPUT_: &str = "_IN";
const OUTPUT_: &str = "_OUT";

const RE_THREAD: &str = r"^-\d+$";
/**
 * Initialize - Command line parameter analysis
 */
fn initialize() -> (String, String, bool, i32) {
    lazy_static! { // (Regex は一度だけコンパイルされる)
        static ref RE: Regex = Regex::new(RE_THREAD).unwrap();
    }
    let args: Vec<String> = env::args().collect();
    let len = args.len();
    if len < 3 {
        let message = iomod::red("入出力フォルダが省略されています".to_string());
        eprintln!("{}: {:?}", message, args);
    }
    let _input: &str = if len > 2 { &args[1] } else { INPUT_ };
    let _output: &str = if len > 3 { &args[2] } else { OUTPUT_ };
    let input: &Path = Path::new(_input);
    let output: &Path = Path::new(_output);
    iomod::mkdir(input);
    iomod::mkdir(output);

    let mut is_copy: bool = true;
    let mut thread: i32 = 3;
    for i in 3..len {
        let argi = &args[i];
        if argi.starts_with("-") {
            if argi == "-c" {
                is_copy = true;
            } else if argi == "-m" {
                is_copy = false;
            } else if RE.is_match(argi) {
                let tmp: i32 = args[i].parse().unwrap();
                if tmp != 0 {
                    thread = 0 - tmp;
                }
            } else {
                let message = iomod::red("オプションエラー".to_string());
                eprintln!("{}: {:?}", message, argi);
            }
        }
    }
    let copymove: &str = if is_copy { "-c[opy]" } else { "-m[ove]" };

    if !input.is_dir() {
        let message = iomod::red("フォルダではありません".to_string());
        eprintln!("{}: {:?}", message, _input);
    }
    if !output.is_dir() {
        let message = iomod::red("フォルダではありません".to_string());
        eprintln!("{}: {:?}", message, _output);
    }
    let _start_time: DateTime<Local> = Local::now();
    println!("{}", _start_time);
    println!("{}: {}", iomod::blue("Input Folder"), _input);
    println!("{}: {}", iomod::blue("Output Folder"), _output);
    print!("{}: {}, ", iomod::blue("Action"), copymove);
    println!("{}: -{}", iomod::blue("Thread"), thread);
    (_input.to_string(), _output.to_string(), is_copy, thread)
}

const POLLING_INTERVAL: u64 = 10; // ms.
/**
 * main
 */
fn main() {
    let (input, output, is_copy, thread) = initialize();
    thmod::initialize();
    for _i in 0..thread {
        daemon::spawnx();
        // println!("spawn: {}", _i + 1);
    }
    let _ = files::get_fils(input.clone(), output, is_copy);
    thmod::terminator();
    while !thmod::is_eof() {
        thread::sleep(Duration::from_millis(POLLING_INTERVAL));
    }
    println!();
    if !is_copy {
        thmod::progress_fin("Delete the input folder");
        let _ = files::remove_dir(input);
    }
    thmod::progress_fin("Finish");
    run();
}

// Test
fn run() {
    atomic::_run();
    // iomod::_run();
}

/*
Rust には2つの文字列型があります。
&str - 文字列スライスとも呼ばれるプリミティブな文字列型。
String - 標準ライブラリの提供する文字列型。文字列操作などに使う。
ざっくり理解すると借用が &str で、所有権があるのが String と覚えておくとよさげです。
*/
fn _strings() {
    // 文字列リテラルは &str
    let _a: &str = "hello";

    // String -> &str
    let _b: &str = String::from("hello").as_str();

    // String の初期化
    let _c: String = String::from("hello");

    // &str -> String
    let _d: String = "hello".to_string();

    // String -> &str
    let _e: &str = _c.as_str();

    // String -> &str
    let _e: &str = &_c;

    // Mutable variable
    let mut _d: String = String::new();
    _d += "ABC";
    _d += "def";
    println!("mut String {:?}", _d);
    assert_eq!(_d, "ABCdef");

    let _ch1: char = _a.chars().nth(0).unwrap();
    let _ch2: &str = &_a[..3];

    // .contains()
    if _a.find("l").is_some() {
        println!("'l' is in the string!");
    }
}
