/**
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use std::sync::atomic::AtomicI32;
use std::sync::Mutex;
use std::time::SystemTime;

use crate::atomic;
use crate::files::DD;
use crate::iomod;

lazy_static! { // App start time
    // https://ytyaru.hatenablog.com/entry/2020/12/15/000000
    static ref _START_TIME: SystemTime = std::time::SystemTime::now();
}

static _SEQ_NO: AtomicI32 = AtomicI32::new(0);
static _CUR_NO: AtomicI32 = AtomicI32::new(0);
static _IS_EOF: AtomicI32 = AtomicI32::new(0);

static mut PRINT: Mutex<i32> = Mutex::new(1); // print lock
static mut LOCK: Mutex<i32> = Mutex::new(2); // STACK lock
static mut STACK: Vec<DD> = Vec::new(); // Vector

// static mut MUTEX: OnceCell<Mutex<i32>> = OnceCell::new();
// static mut ONCE_CELL: OnceCell<Vec<DD>> = OnceCell::new();

pub fn push(dd: DD) {
    unsafe {
        let _lock = LOCK.lock().unwrap();
        let _push: () = STACK.push(dd);
    }
}
pub fn pop() -> Option<DD> {
    unsafe {
        let _lock = LOCK.lock().unwrap();
        let _pop: Option<DD> = STACK.pop();
        _pop
    }
}

pub fn get() -> Option<DD> {
    let _pop = pop();
    // match _pop {
    //     Some(x) =>  Some(x),
    //     None => None,
    // }
    _pop
}
pub fn put(dd: DD) {
    atomic::atomic_add(&_SEQ_NO, 1); // sequence number
    atomic::atomic_add(&_CUR_NO, 1); // current number
    push(dd.clone());
}

pub fn decrement_cur_number() {
    atomic::atomic_add(&_CUR_NO, -1); // current number
}
/// Whether it is in an idle loop.
pub fn is_eof() -> bool {
    let eof: bool = atomic::atomic_bool_get(&_IS_EOF);
    let cur: i32 = atomic::atomic_get(&_CUR_NO); // current number
    eof && cur <= 0
}

pub fn progress(dd: DD) {
    let seq = atomic::atomic_get(&_SEQ_NO); // sequence number
    let cur = atomic::atomic_get(&_CUR_NO); // current number
    let ela = elapsed_time(); // 経過時間
    let cya = iomod::cyan(&format!("{} {}/{}", ela, cur, seq));
    // \e[nK カーソルより後ろを消去
    // \e[nA 上にn移動
    // print!("{}: {}\x1b[K\r", cya, dd.input); // NG
    unsafe {
        let _print = PRINT.lock().unwrap();
        println!("{}: {}\x1b[K\x1b[1A", cya, dd.input);
    }
}
pub fn progress_fin(message: &str) {
    let seq = atomic::atomic_get(&_SEQ_NO); // sequence number
    let cur = atomic::atomic_get(&_CUR_NO); // current number
    let ela = elapsed_time(); // 経過時間
    let cya = iomod::cyan(&format!("{} {}/{}", ela, cur, seq));
    unsafe {
        let _print = PRINT.lock().unwrap();
        println!("{}: {}", cya, message);
    }
}
// https://ytyaru.hatenablog.com/entry/2020/12/15/000000
// as_millis, as_micros, as_nanos
pub fn elapsed_time() -> String {
    let now: SystemTime = std::time::SystemTime::now();
    if let Ok(epoch) = now.duration_since(*_START_TIME) {
        let min: u64 = epoch.as_secs() / 60_u64;
        let sec: u64 = epoch.as_secs() % 60_u64;
        let mil: &str = &format!("{:09}", epoch.as_nanos())[..3];
        return format!("{}:{:02}.{}", min, sec, mil);
    }
    "".to_string() // start > now
}

const STACK_CAPACITY: usize = 1024;

pub fn initialize() {
    let _ = elapsed_time(); // Initialize start time

    unsafe {
        let _lock = LOCK.lock().unwrap(); // キャパシティの設定
        STACK.reserve(STACK_CAPACITY);
        // println!("STACK_CAPACITY: {}", STACK.capacity());
    }
}
pub fn terminator() {
    atomic::atomic_bool_set(&_IS_EOF, true); // EOF マークの設定
}

//  https://doc.rust-jp.rs/book-ja/ch19-01-unsafe-rust.html
// unsafe Rustでできること
// ・可変で静的な変数にアクセスしたり変更すること
// ・生ポインタを参照外しすること
// ・unsafeな関数やメソッドを呼ぶこと
// ・unsafeなトレイトを実装すること
/*
Rustにunsafeな分身がある理由は、根本にあるコンピュータのハードウェアが本質的に
unsafeだからです。Rustがunsafeな処理を行わせてくれなかったら、特定の仕事を行えません。
Rustは、低レベルなシステムプログラミングを許可する必要があります。
直接OSと相互作用したり、独自のOSを書くことさえもそうです。
低レベルなシステムプログラミングに取り組むことは、言語の目標の1つなのです。
*/
// 可変で静的な変数を定義
static mut _TAX: f32 = 0.1;
fn _test() {
    unsafe {
        _TAX = 0.08; // use of mutable static
        println!("Price: {}", _TAX * 300.0);
    }
}
