/**
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * cutSpace - Optimize the Space and Semicolon.
 *
 * @author kunio himei.
 */
class Space {

    // Language start (attribute) tag.
    private static final Pattern LANG_START_ATTRIBUTE_TAG = Pattern.compile(
            "<(?:script|style)[^>]+?>", Pattern.CASE_INSENSITIVE);

    private static final Pattern CUT_LAST_SEMICOLON = Pattern.compile( // ;} -> }
            "[\\s]*;[\\s]*}[\\s]*");

    private String optCSS = CSS_SELECTOR; // Inside of selector
    private String optHTML = HTML_OUTSIDE_TAG; // Outside of <>
    private boolean isStateTransitionProhibited; // 状態遷移禁止 (HTML埋め込みモード)

    /**
     * Space compression.
     * -  Optimize language start tag.
     */
    String space(String input, boolean deprecated) {
        Matcher m = LANG_START_ATTRIBUTE_TAG.matcher(input);
        if (m.find()) { // If the attribute is specified
            // NOTE HTML以外の言語を開始するということは、HTMLをリセットして良い.
            //  Starting a language other than HTML may reset HTML.
            optHTML = HTML_OUTSIDE_TAG;
            String tag = html(m.group(), deprecated);
            input = m.replaceFirst(tag);
            optHTML = HTML_OUTSIDE_TAG;
        }
        if (Lang.is(Lang.SCRIPT)) { // SCRIPT
            return script(input);

        } else if (Lang.is(Lang.CSS)) { // CSS
            return css(input);
        }
        return html(input, deprecated); // HTML
    }

    /**
     * script. ///////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Cut semicolon, ;} -> }
     *
     * @param input input text (trim)
     * @return source text (trim)
     */
    @NotNull
    private String script(@NotNull String input) {
        final int capacity = input.length() + 7;
        StringBuilder sb = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input.trim());
        String src = t2.source();
        String tpl = t2.template();
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                optimizeSpace(sb, SCRIPT_OPTIMIZE, c);
            } else
                sb.append(c);
        }
        Matcher m = CUT_LAST_SEMICOLON.matcher(sb); // ;} -> }
        if (m.find())
            return m.replaceAll("}").trim();
        return sb.toString().trim();
    }

    /**
     * css. //////////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Cut semicolon, ;} -> }
     *
     * @param input input text (trim)
     * @return source text (trim)
     */
    @NotNull
    private String css(@NotNull String input) {
        final int capacity = input.length() + 7;
        StringBuilder sb = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input.trim());
        String src = t2.source();
        String tpl = t2.template();
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                if (!isStateTransitionProhibited) {
                    if ('[' == c) optCSS = CSS_ATTRIBUTE_SELECTOR; // [Attribute]
                    else if (']' == c) optCSS = CSS_SELECTOR;
                    else if ('{' == c) optCSS = CSS_ATTRIBUTE; // {Attribute:value;}
                    else if ('}' == c) optCSS = CSS_SELECTOR;
                    else if ('@' == c) optCSS = CSS_MEDIA_QUERIES; // @media(){}
                }
                optimizeSpace(sb, optCSS, c);
            } else
                sb.append(c);
        }
        Matcher m = CUT_LAST_SEMICOLON.matcher(sb); // ;} -> }
        if (m.find())
            return m.replaceAll("}").trim();
        return sb.toString().trim();
    }

    /**
     * html. /////////////////////////////////////////////////////////////
     * - Remove the white space around the control code.
     * - Remove whitespace around html block tags.
     * - Optimize Type="CSS", media="screen and (max-width:480px)"
     * - Cut semicolon, Type="foo;", onContextMenu="return false;"
     * - Deprecated: <tag foo=""  bar=""> → <tag foo=""bar="">
     *
     * @param input input text
     * @return source text (rTrim)
     */
    @NotNull
    private String html(@NotNull String input, boolean deprecated) {
        final int capacity = input.length() + 7;
        StringBuilder sb = new StringBuilder(capacity);
        StringBuilder type = new StringBuilder(capacity);
        T2TPL t2 = new Template().parse(input);
        String src = t2.source() + '\r'; // Guard, Required if ending with a string
        String tpl = t2.template() + '\r';
        String save = optCSS;
        isStateTransitionProhibited = true; // Inline mode
        for (int i = 0; i < src.length(); i++) {
            char t = tpl.charAt(i);
            char c = src.charAt(i);
            if (Template.DEL != t) {
                if (!type.isEmpty()) { // Optimize CSS in tag <tag Type = "CSS">
                    optCSS = CSS_ATTRIBUTE;
                    String css = css(type.substring(1, type.length() - 1));
                    if (Tools.endsWith(css, ';')) { // Cut last semicolon
                        css = css.substring(0, css.length() - 1);
                    }
                    char qt = type.charAt(0);
                    sb.append(qt).append(css).append(qt);
                    type.setLength(0);
                }
                if ('<' == c)
                    optHTML = deprecated ? HTML_INSIDE_DEPRECATED : HTML_INSIDE_TAG;
                optimizeSpace(sb, optHTML, c);
                if ('>' == c) optHTML = HTML_OUTSIDE_TAG;
            } else
                type.append(c);
        }
        optCSS = save;
        isStateTransitionProhibited = false;
        return cutSpace(Tools.rTrim(sb)); // Remove '\r'
    }

    /**
     * Optimize the space. ////////////////////////////////////////////////
     *
     * @param sb       I/O
     * @param optimize OPTIMIZE control characters
     * @param ch       Additional character
     */
    private static void optimizeSpace(@NotNull StringBuilder sb, String optimize, char ch) {
        char c = Tools.toWhiteSpace(ch);
        if (!sb.isEmpty() &&
                0 <= optimize.indexOf(c)) { // if Optimize target
            char front = Tools.charAtLast(sb); // To White Space
            // The front is the character to be optimized
            if (0 <= optimize.indexOf(front)) {
                if (' ' == c) return; // Do not add whitespace
            }
            if (' ' == front) {
                if (' ' == c) return; // Do not add double space
                sb.setLength(sb.length() - 1); // Remove front whitespace
            }
        }
        sb.append(c);
    }

    //////////////////////////////////////////////////////////////////////
    private static final Pattern HTML_EXTRACT_TAG = Pattern.compile(
            "[\\s]*(</?" + "([a-zA-Z]+[1-6]?)" + "[^>]*?>" + ")[\\s]*");
    //                      group(2):Tag name

    /**
     * html: ブロック要素(インライン以外)の周辺の空白を削除する。
     * - Remove whitespace around block elements (other than inline).
     */
    @NotNull
    private static String cutSpace(CharSequence s) {
        Matcher m = HTML_EXTRACT_TAG.matcher(s);
        while (m.find()) {
            String g1 = m.group(1); // </?Tag...>
            if (!m.group().equals(g1)) {
                String tag = '|' + m.group(2).toLowerCase() + '|';
                if (!HTML_INLINE_ELEM.contains(tag)) {
                    s = s.subSequence(0, m.start()) +
                            g1 + s.subSequence(m.end(), s.length());
                    m.reset(s);
                }
            }
        }
        return s.toString();
    }

    // https://developer.mozilla.org/ja/docs/Web/HTML/Inline_elements
    @SuppressWarnings("SpellCheckingInspection")
    private static final String HTML_INLINE_ELEM =
            "|" +
                    "a|" +          // ※ リンクのアンカー
                    "abbr|" +       // ※ 略語（全般）
//                  "acronym|" +    // ✗ 略語（頭字語）（Abolished、ref.abbr）
                    "audio|" +      // （視覚的なコントロールがある場合）
                    "b|" +          // ※ 太字（注意）
                    "bdi|" +        // ※ 書字方向分離
                    "bdo|" +        // ※ 書字方向
                    "big|" +        // ※ 大きめの文字
//                  "br|" +         // ✗ 改行 （Treated as a block element）
                    "button|" +     // ※ ボタン
                    "canvas|" +     // グラフィックキャンバス
                    "cite|" +       // ※ 引用元（出典・参照先）
                    "code|" +       // ※ ソースコード
                    "data|" +       // ※ コンテンツの断片を機械可読な翻訳にリンク
                    "datalist|" +   // 他のコントロールで利用可能な値を表現
                    "del|" +        // ※ 削除済み文字列
                    "dfn|" +        // ※ 定義する用語
                    "em|" +         // ※ 強調（斜体）
                    "embed|" +      // 埋め込み外部コンテンツ
                    "i|" +          // ※ イタリック体
                    "iframe|" +     // インラインフレーム
                    "img|" +        // 画像の埋め込み
                    "input|" +      // フォームの部品
                    "ins|" +        // ※ 文書に追加されたテキストの範囲
                    "kbd|" +        // ※ 文字入力を表す行内の文字列の区間
                    "label|" +      // 部品とラベルの関連付け
                    "map|" +        // イメージマップ
                    "mark|" +       // ※ 参照や記述の目的で目立たせたる文字列
                    "meter|" +      // 既知の範囲内のスカラー値、または小数値
                    "noscript|" +   // スクリプトの実行がブラウザーで無効にされている場合に表示する
                    "object|" +     // オブジェクトの埋め込み
                    "output|" +     // サイトやアプリが計算結果やユーザー操作の結果を挿入することができるコンテナー
                    "picture|" +    // 画像要素
                    "progress|" +   // タスクの進捗状況を表示
                    "q|" +          // ※ 短文の引用（インライン）
                    "ruby|" +       // ベーステキストの上、下、隣に描画される小さな注釈
                    "s|" +          // ※ 取り消し線
                    "samp|" +       // ※ 出力サンプル
                    "script|" +     // スクリプト
                    "select|" +     // セレクトボックス
                    "slot|" +       // ウェブコンポーネントのスロット
                    "small|" +      // ※ 小さめの文字
                    "span|" +       // ※ 範囲の指定（インライン）
                    "strong|" +     // ※ より強い強調（重要）
                    "sub|" +        // ※ 下付き文字
                    "sup|" +        // ※ 上付き文字
                    "svg|" +        // 二次元グラフィックスを XMLで記述
                    "template|" +   // コンテンツテンプレート
                    "textarea|" +   // 複数行の入力欄
                    "time|" +       // 特定の時の区間
                    "tt|" +         // ※ 等幅フォント
                    "u|" +          // ※ 非言語的注釈（下線）
                    "var|" +        // ※ 変数・引数
                    "video|" +      // 映像
                    "wbr|";         // ※ 改行可能位置

    //////////////////////////////////////////////////////////////////////
    // https://developer.mozilla.org/ja/docs/Web/CSS/CSS_Selectors
    // CSS : selector{attribute:value;}
    // -  子結合 >、一般兄弟 ~、隣接兄弟 +、E:not(s)、
    // -  属性セレクタ E[foo]、E[foo="bar"]、E[foo~="bar"]、E[foo|="en"]、
    // -  E[foo^="bar"]、E[foo$="bar"]、E[foo*="bar"]
    // h1, h2 : 1つのCSSで複数のセレクタを対象にする(カンマ区切り、セレクターリスト)
    // .p1 .c2 : 複数のセレクタで階層構造の対象を絞り込む(空白区切り、子孫セレクタ) ※
    // .ｃ1.ｃ2 : 複数のセレクタで対象を絞り込む(区切り文字なし) <p class="ｃ1 ｃ2"> ※
    // ul > li : 特定の要素の直下にある要素を>区切りで指定する（子セレクタ）
    // p ~ span : <p> 要素の後にある <span> 要素をすべて選択（一般兄弟結合）
    // p + span : <p> 要素の後にすぐに続く <span> 要素をすべて選択（隣接兄弟結合）
    // col || td : <col> 要素のスコープに所属するすべての <td> 要素を選択（列結合）
    // a[href][title] : 複数の属性セレクタで対象を絞り込む([ ])
    // a:link[class="p1"] : 擬似クラスと属性セレクタで絞り込む
    // p[class="p2"]::before : 属性セレクタと擬似要素を指定する
    // a:link::before : 擬似クラスと擬似要素を指定する
    //
    // Remove whitespace at both ends (両端の空白を削除)
    // NOTE ※ "." Is not defined in the selector
    //  because the meaning is different depending on the presence or absence of blanks.
    //  ※ 空白の有無で意味が異なるため、”.”は、セレクタに定義しない.
    private static final String SCRIPT_OPTIMIZE = " ;{}[]()<>" + Template.SCRIPT_OPERATOR;
    private static final String CSS_SELECTOR = " ,:>~+|()[]{}";
    private static final String CSS_ATTRIBUTE_SELECTOR = " ()[=~|^$*]{}"; // [Attr Selector]
    private static final String CSS_ATTRIBUTE = " ,:;!#/(=<>)[]{}";
    private static final String CSS_MEDIA_QUERIES = CSS_ATTRIBUTE; // @media{}
    private static final String HTML_OUTSIDE_TAG = " ";
    private static final String HTML_INSIDE_TAG = " =>"; // <tag foo="" bar="">
    private static final String HTML_INSIDE_DEPRECATED = " =>\"'"; // <tag foo=""bar="">
}