/**
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Parser - 構文解析.
 *
 * @author kunio himei.
 */
public class Parser {

    private static final char C_UTF_BOM = '\uFFFE'; // 削除マーク
    private static final String S_UTF_BOM = String.valueOf(C_UTF_BOM);
    public static final char C_DELETE = '✔'; // U+2713
    private static final char NL = '\n';
    private final StringBuilder buffer = new StringBuilder(512);
    public String debug; // デバッグ出力
    public int cutSpace; // 空白削除数
    private int indentLength; // 先頭空白(インデント)の長さ
    private boolean isPre; // Pre タグの中

    // 先頭(左端)空白
    private static final Pattern INDENT = Pattern.compile(
            "^(\\s*)");
    private static final Pattern HTML_EXTRACT_TAG = Pattern.compile(
            "(\\s*)(<(/?)" + "([a-zA-Z]+[1-6]?)" + "[^<>]*" + "(>?))(\\s*)");
//                 (g1:S)(g2(g3:End)(g4:TagName     )                (g5))(g6:S)

    /**
     * parse: 空白削除パーサー
     * - Remove whitespace around block-level tags.
     */
    public String parse(String input) {
        Matcher m = INDENT.matcher(input); // 先頭(左端)空白を抽出
        indentLength = m.find() ? m.end(1) : 0;
        String result = space(join(input));
        debug = result.replace(C_UTF_BOM, C_DELETE);
        String output = result.replace(S_UTF_BOM, "");
        cutSpace += result.length() - output.length(); // 空白削除数
        return output;
    }

    /**
     * join: 改行で生き別れになったブロックレベルタグを結合する
     */
    private String join(String input) {
        buffer.setLength(0);
        buffer.append(input);
        Matcher m = HTML_EXTRACT_TAG.matcher(buffer);
        while (m.find()) {
            String tagName = getBlockElement(m.group(4));
            if (!tagName.isEmpty()) { // ブロックレベル要素なら
                boolean hasEnd = !getValue(m.group(5)).isEmpty();
                if (!hasEnd) { // > 抜け
                    String in = Main.getLine();
                    if (in == null) break;
                    buffer.append(' ').append(in.trim()); // Join
                    System.out.println(Main.lineNumber + ": " + m.group(2));
                    m.reset(buffer); // retry
                }
            }
        }
        return buffer.toString();
    }

    /**
     * space: ブロックレベルタグの周辺空白を削除する.
     */
    private String space(String input) {
        buffer.setLength(0);
        boolean hasIndent;
        Matcher m = HTML_EXTRACT_TAG.matcher(input);
        while (m.find()) {
            String replacement;
            String tagName = getBlockElement(m.group(4));
            if (!tagName.isEmpty()) { // ブロックレベル要素なら
                // 前後空白
                // 左空白
                String left = getValue(m.group(1)); // 左空白
                // 右空白
                String right = getValue(m.group(6)); // 右空白
                hasIndent = indentLength == m.end(1); // インデントか？

                if (Main.isDebug) {
                    System.out.println(Main.lineNumber + ": " +
                            hasIndent + "\t" + m.end(1) +
                            "\t'" + m.group() + "'");
                }
                if (tagName.equals("pre")) {
                    isPre = getValue(m.group(3)).isEmpty();
                    if (isPre && !hasIndent && !left.isEmpty())
                        left = S_UTF_BOM;
                }
                if (!isPre) { // <pre> の外なら (</pre> を含む)
                    if (!hasIndent && !left.isEmpty())
                        left = S_UTF_BOM;
                    if (!right.isEmpty())
                        right = S_UTF_BOM;
                }
                replacement = left + m.group(2) + right;
            } else {
                replacement = m.group(); // inline-level element.
            }
            m.appendReplacement(buffer, replacement);
        }
        m.appendTail(buffer);
        return buffer.append(NL).toString();
    }

    // タグ名(ブロック)の取得
    private static String getBlockElement(String tag) {
        String tagName = tag.toLowerCase();
        if (HTML_BLOCK_ELEMENT_SET.contains(tagName)) {
            return tagName;
        }
        return "";
    }

    // https://developer.mozilla.org/ja/docs/Web/HTML/Block-level_elements
    // NOTE ※ 空白を無視するかどうかで追加したタグで、ブロック要素でないものを含む.
    //  Tags added depending on whether to ignore whitespace,
    //  including those that are not block elements.
    @SuppressWarnings("SpellCheckingInspection")
    private static final List<String> HTML_BLOCK_ELEMENT_LIST = Arrays.asList(
            // ※ は空白に関してブロック要素のように振る舞うタグ
            "address",      // 連絡先情報
            "article",      // 記事コンテンツ
            "aside",        // 本論から外れたコンテンツ
            "blockquote",   // 長い「ブロック」の引用
            "br",           // ※ (改行)
            "dd",           // 定義リストで用語を説明
            "details",      // 折りたたみウィジェット
            "dialog",       // ダイアログボックス
            "div",          // 文書の一部
            "dl",           // 定義リスト
            "dt",           // 定義語リストの用語
            "fieldset",     // フィールドセットのラベル
            "figcaption",   // 図表のキャプション
            "figure",       // キャプション(figcaption)を伴うメディアコンテンツをグループ化
            "footer",       // セクションまたはページのフッター
            "form",         // 入力フォーム
            "h1",           // 見出しレベル 1-6
            "h2",           //
            "h3",           //
            "h4",           //
            "h5",           //
            "h6",           //
            "header",       // セクションまたはページのヘッダー
            "hgroup",       // 見出し情報をグループ化
            "hr",           // 水平線 (区切り線)
//          "img",          // ※ (イメージ表示)
            "li",           // リストの項目
            "main",         // この文書で固有の中心的なコンテンツを含む
            "nav",          // ナビゲーションのリンクを含む
            "ol",           // 番号付きリスト
            "pre",          // 整形済みテキスト
            "p",            // 段落
            "section",      // ウェブページのセクション
            "table",        // 表
            "td",           // ※ (表のデータ)
            "th",           // ※ (表のヘッダ)
            "tr",           // ※ (表の行部分)
            "title",        // ※ (文書の題名)
            "ul");          // 番号なしリスト
    private static final Set<String> HTML_BLOCK_ELEMENT_SET =
            new HashSet<>(HTML_BLOCK_ELEMENT_LIST);

    /**
     * 6. Right trim. (String)
     */
    public static String rTrim(String s) {
        int length = s.length();
        int end = length;
        while (0 <= --end)
            if (' ' < s.charAt(end)) break;
        end += 1;
        return end < length ? s.substring(0, end) : s;
    }

    // regex が null を返す時の対策
    private static String getValue(String val) {
        return val == null ? "" : val;
    }
}