package jp.co.powerbeans.powerql.dao;

import jp.co.powerbeans.powerql.POQLManager;
import jp.co.powerbeans.powerql.POQLTransaction;
import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;

/**
 * POQLBaseDAO, POQLDynaViewBaseDAO の抽象クラス。<BR>
 * 全てのDAOで共通のプロパティ、処理を定義する。
 *
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * @author A.Monden
 */
abstract class CoreDAO {

    /** PowerQLマネージャ */
    private POQLManager manager;
    
    /** Beanクラス */
    private final Class beanClass;

    /** 単一検索結果が空の場合に null を返すかどうか */
    private boolean returnNull;

    /** 例外発生時に例外をスローするかどうか */
    private boolean throwException;

    /** 例外リスナー */
    private POQLExceptionListener exceptionListener;

    /** rtrim フラグ. true 行う, false 行わない */
    private boolean rtrim; 

    /**
     * 全DAO共通コンストラクタ
     * @param manager PowerQLマネージャー
     * @param bean_class 検索結果取得Bean
     */
    public CoreDAO(POQLManager manager, Class bean_class) {
        
        this.manager = manager;
        this.beanClass = bean_class;
        this.exceptionListener = manager.getExceptionListener();
    }

    /**
     * Beanクラスを取得
     * @return beanClass
     */
    public Class getBeanClass() {
        return beanClass;
    }

    /**
     * PowerQLマネージャーを取得
     * @return manager
     */
    public POQLManager getManager() {
        return manager;
    }

    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうかのフラグを取得する。<BR>
     * @return returnNull true nullで返す。false 空のBeanインスタンスで返す。
     */
    public boolean isReturnNull() {
        return returnNull;
    }

    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか設定する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @param returnNull returnNull を設定します。true nullで返す。false 空のBeanインスタンスで返す。
     */
    public void setReturnNull(boolean returnNull) {
        this.returnNull = returnNull;
    }

    /**
     * DBアクセス時のExceptionを処理
     * @param e 発生Exception
     */
    protected void onException(Exception e) {
    
        if (exceptionListener != null) {
            // 例外リスナーが設定されていれば通知
            exceptionListener.onException(e);
        }
        
        if (throwException) {
            // 例外スローフラグが有効ならスローする
            throw new RuntimeException(e);
        }
    }

    /**
     * 例外発生時に例外をRuntimeExceptionでラップしてスローするかどうか
     * 設定する。
     * 
     * @param throw_exception true:例外をスロー, false:スローしない
     */
    public void setThrowException(boolean throw_exception) {
        this.throwException = throw_exception;
    }

    /**
     * 例外発生時に例外をRuntimeExceptionでラップしてスローするかどうかの
     * フラグを取得。
     * 
     * @return throwException true:例外をスロー, false:スローしない
     */
    public boolean isThrowException() {
        return throwException;
    }

    /**
     * Beanクラスをインスタンス化して返す
     * @return Beanクラスインスタンス
     */
    protected Object createBean() {
        try {
            return beanClass.newInstance();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * もし検索結果なし、あるいはException発生時に
     * 空のBeanを返す設定であれば空のBeanを生成して返す。
     * nullを返す設定であればnullを返す。
     * @return 空のBean or null
     */
    protected Object createSafeBean() {
        if (returnNull) {
            return null;
        } else {
            try {
                return beanClass.newInstance();
            } catch (Exception e) {
                onException(e);
            }
            return null;
        }
    }

    /**
     * トランザクションをクローズ
     * 
     * @param bqlTrn
     */
    protected void close(POQLTransaction bqlTrn) {
        
        try {
            if (bqlTrn != null && bqlTrn.getConnection() != null
                    && !bqlTrn.getConnection().isClosed()) {
                
                bqlTrn.close();
            }
            if (bqlTrn != null) {
                bqlTrn = null;
            }
        } catch (Exception e) {
            onException(e);
        }
    }

    /**
     * @return Returns the exceptionListener.
     */
    public POQLExceptionListener getExceptionListener() {
        return exceptionListener;
    }
    /**
     * @param exceptionListener The exceptionListener to set.
     */
    public void setExceptionListener(POQLExceptionListener exceptionListener) {
        this.exceptionListener = exceptionListener;
    }
	/**
	 * @return rtrim を戻します。
	 */
	public boolean isRtrim() {
		return rtrim;
	}
	/**
	 * @param rtrim rtrim を設定。
	 */
	public void setRtrim(boolean rtrim) {
		this.rtrim = rtrim;
	}
}
