package jp.co.powerbeans.powerql.dao;

import java.util.Collection;

import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;
import jp.co.powerbeans.powerql.exceptions.POQLResultEmptyException;
import jp.co.powerbeans.powerql.exceptions.POQLResultMultiException;

/**
 * <p>タイトル: POQLDAO</p>
 * <p>説明: 汎用DAOメソッドを提供するインターフェース。<BR>
 * 通常ビジネスロジックは POQLManager#createDAO(Class) により
 * テーブル、ビューと関連したDAOを取得し、各メソッドで
 * DBにアクセスする。<BR><BR>
 * 
 * POQLDAOの基本的理念は、例外をcatchしたい時だけcatchする、という考えに基づいている。<BR>
 * そのためSQLエラーなどの例外発生時にビジネスロジック側に例外をスローするかどうかを
 * 動的に変更できる（デフォルトはスローしない。setThrowExcpetion(false) を実行したのと同じ状態）。<BR><BR>
 * 
 * 例外をスローさせるには、DAOを生成したあと、<BR>
 * setThrowException(true); <BR>
 * を実行する。その後各DBアクセスメソッドを実行しSQLエラー、排他エラー等が発生すると、<BR>
 * 各メソッドは例外をRuntimeExceptionでラップしてスローする。<BR><BR>
 * 
 * 例外をスローしない時にSQLエラー等が発生した場合の戻り値は、<BR>
 * insert,update,deleteがつくメソッドは0<BR>
 * find1By,findByPrimaryKeyメソッドはnull<BR>
 * その他のfindメソッドは Object[0]<BR>
 * を返す。<BR><BR>
 * 
 * POQLDAOを利用するビジネスロジック側では必要な場合だけ例外をcatchし、
 * 例外の種類に応じた処理を実装する。<BR>
 * <BR>
 * <pre>
 * <b>POQLDAOを生成するまでに必要な前準備</b>
 * <ol>
 * <li>DBMSにテーブルを作成（以下の例ではTrnWork)
 * <li>作成したテーブルと同じ名称のJavaBeanクラスを作成。(パッケージはどこでもOK,大文字小文字は一致しなくても構いません）
 * <li>JDBCドライバをクラスパスに追加
 * <li>サーブレット、あるいは各ビジネスロジッククラスの継承元（基底）クラスの
 *     static {} 内などの初期化処理部分で POQLManager を生成
 * <li>ビジネスロジック内でPOQLManagerからDAOを生成しコミット
 * </ol>
 * <hr>
 * 階層1) WEBアプリケーション
 * ブラウザ<->WEB層<->コントローラ<->ビジネスロジック   <->   POQLDAO <-> DB
 *                                  ┗同一トランザクションで処理可能┛
 * 
 * 階層2) GUIアプリケーション
 * UI層<->イベントディスパッチャ<->ビジネスロジック  <->  POQLDAO <-> DB
 *                               ┗同一トランザクションで処理可能┛
 * <hr>
 * 
 * 例1）   // 排他チェックを行い、1メソッドでコミット、またはロールバックし、例外をスローする
 *         POQLManager bm = new POQLManager("org.postgresql.Driver",
 *               "jdbc:postgresql:bqltest", "trans",
 *               "t79rp90", "POQL_SEQUENCE");
 * 
 *         POQLDAO dao = bm.createDAO(TrnWork.class); // DAO生成
 *         dao.setExclusiveCheckField("updateDate"); // 排他チェックを行う場合のチェックプロパティ名
 *         dao.setSingleCallMethod(true); // 1メソッドでコミット or ロールバックする
 *         dao.setThrowException(true); // 例外をスローする
 * 
 *         TrnWork tw = new TrnWork();
 *         tw.setWorkOID(300);
 *         tw.setWorkName("Teacher");
 *         try {
 *          // 追加実行
 *          dao.create(tw); // setThrowException(true) を実行したため例外をcatchする
 *         } catch(RuntimeException e) {
 *            if (e instance of POQLExclusiveException) {
 *              // 排他エラー
 *            } else if (...) {
 *              // その他のエラー
 *            }
 *         }
 * 
 * 
 * 例2)    // 排他チェックなし、1メソッドでコミット、またはロールバックし、例外をスローしない
 *         POQLManager bm = new POQLManager("org.postgresql.Driver",
 *               "jdbc:postgresql:bqltest", "trans",
 *               "t79rp90", "POQL_SEQUENCE");
 * 
 *         POQLDAO dao = bm.createDAO(MstPref.class);
 *         // try catch する必要は無い。存在しない場合は null を返す
 *	       MstPref m = (MstPref)dao.findByPrimaryKey("01"); // プライマリキーで検索
 *
 *
 * 例3）   // 排他チェックを行い、明示的にコミット、またはロールバックし、例外をスローする
 *         POQLManager bm = new POQLManager("org.postgresql.Driver",
 *               "jdbc:postgresql:bqltest", "trans",
 *               "t79rp90", "POQL_SEQUENCE");
 * 
 *         POQLDAO dao = bm.createDAO(TrnWork.class); // DAO生成
 *         dao.setExclusiveCheckField("updateDate"); // 排他チェックを行う場合のチェックプロパティ名
 *         dao.setSingleCallMethod(false); // 明示的にでコミット or ロールバックする
 *         dao.setThrowException(true); // 例外をスローする
 * 
 *         TrnWork tw = new TrnWork();
 *         tw.setWorkOID(300);
 *         tw.setWorkName("Teacher");
 *         try {
 *            // 更新実行
 *            dao.update(tw); // setThrowException(true) を実行したため例外をcatchする
 *            // 他のビジネスロジック
 *            // もしここでビジネスロジックの例外が発生した場合は
 *            // 何かしらの例外をスローしてトランザクションをrollbackすることが可能。
 * 
 *            // ..
 *            .. 処理を完了したのでコミット
 *            dao.commit();
 * 
 *         } catch(RuntimeException e) {
 *            // ロールバック
 *            dao.rollback();
 * 
 *            if (e instance of POQLExclusiveException) {
 *              // 排他エラー
 *            } else if (...) {
 *              // その他のエラー
 *            }
 *         } finally {
 *            // dao.setSingleCallMethod(false) をコールしたため明示的に close が必要
 *            dao.close();
 *         }
 *
 * <b>デフォルト値</b>
 *   getExclusiveCheckField() = null
 *   getSingleCallMethod() = true
 *   getThrowException() = false
 * </pre>
 * 　
 * 
 * </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/16</p>
 * @author 門田明彦
 * @version $Revision: 1.3 $
 */

public interface POQLDAO {

    /**
     * プライマリキーで検索<BR>
     * プライマリキーが複数フィールドで構成されている場合は
     * キーと同数Object[]を渡す。
     * 
     * 
     * @param obj
     *            プライマリキーの値
     * @return 検索結果Bean
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     */
    public Object findByPrimaryKey(Object obj);

    /**
     * NUMBER型プライマリキーで検索(内部で Integer(id) に変換)
     * 
     * @param id
     *            プライマリキーの値
     * @return 検索結果Bean
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     */
    public Object findByPrimaryKey(int id);

    /**
     * 全検索
     * @return 検索結果Bean
     */
    public Collection findByAll();

    /**
     * OrderByを指定して全検索
     * @return 検索結果Bean
     */
    public Collection findByAll(String order_by);

    /**
     * where 条件を指定して1レコードを取得
     * 
     * @param where
     *            検索条件
     * @return 検索結果
     */
    public Object find1By(String where);

    /**
     * where 条件を指定して複数レコードを取得する。<BR>
     * 
     * @param where
     *            検索条件
     * @return 検索結果
     */
    public Collection findBy(String where);
    
    /**
     * where 条件とソート条件を指定して複数レコードを取得する。<BR>
     * 
     * @param where
     *            検索条件
     * @param order_by ソート条件
     * @return 検索結果
     */
    public Collection findBy(String where, String order_by);
    
    /**
     * プライマリキーフィールドで一致するレコードを更新する。<BR>
     * 排他制御チェックが有効の場合は、レコードの排他チェック用フィールドを
     * SELECT FOR UPDATE で取得してbeanの対応フィールドと比較し、
     * 一致しなければロールバックしてExceptionをスローする。<BR>
     * 排他チェック用フィールドはプロパティファイルで指定する。
     * 値が "" の場合は排他チェックを行わない。<BR>
     * DAO固有の設定にしたい場合はこのメソッドを呼び出す前に
     * setExclusiveCheckField メソッドで指定する。<BR><BR>
     * 
     * 複数のレコードをまとめて更新する場合は bean の配列か
     * Collection を引数に渡す。
     * @param bean 更新する値を持つbean
     * @return 更新レコード数
     */
    public int update(Object bean);
    
    /**
     * columnsで指定したフィールドの値で
     * プライマリキーフィールドで一致するレコードを更新する。<BR>
     * 排他制御チェックが有効の場合は、レコードの排他チェック用フィールドを
     * SELECT FOR UPDATE で取得してbeanの対応フィールドと比較し、
     * 一致しなければロールバックしてExceptionをスローする。<BR>
     * 排他チェック用フィールドはプロパティファイルで指定する。
     * 値が "" の場合は排他チェックを行わない。<BR>
     * DAO固有の設定にしたい場合はこのメソッドを呼び出す前に
     * setExclusiveCheckField メソッドで指定する。<BR><BR>
     * 
     * 複数のレコードをまとめて更新する場合は bean の配列か
     * Collection を引数に渡す。
     * @param bean 更新する値を持つbean
     * @param columns 値指定カラム名
     * @return 更新レコード数
     */
    public int update(Object bean, String[] columns);    
    
    /**
     * DAOで複数のupdateメソッドを呼び出すかどうか設定する。<BR>
     * 単一呼び出しモードでは sql 実行毎にcommitする。<BR>
     * 複数呼び出しモードでは sql 実行直後はcommitせず、POQLDAO#commit メソッドで明示的にコミットする。<BR>
     * findメソッドを利用する場合はこのメソッドは利用する必要は無い。<BR>
     * 既にトランザクションが存在する状態でこのメソッドを実行すると、IllegalStateException をスローする。
     * @param singleCallMethod true 単一呼び出しモード(デフォルト), false 複数メソッド呼び出しモード
     */
    public void setSingleCallMethod(boolean singleCallMethod); 
    
    /**
     * 単一メソッド呼び出しモードかどうか判定
     * @return singleCallMethodMode true 単一メソッド呼び出しモード, false 複数メソッド呼び出しモード
     */
    public boolean isSingleCallMethod();
    
    /**
     * 複数呼び出しモードでコミットしていない変更に対してコミットする。<BR>
     * 単一呼び出しモードでは何も実行しない。
     */
    public void commit();
    
    /**
     * 複数呼び出しモードでコミットしていない変更をロールバックする。<BR>
     * 単一呼び出しモードでは何もしない。
     */
    public void rollback();

    /**
     * 複数呼び出しモードで残っているコネクションをクローズする。
     * 単一呼び出しモードではfind,update等のメソッド内でコネクションをクローズしているため
     * このメソッドを実行しても何も行わない。
     */
    public void close();
    
    /**
     * commit と close を実行する。
     */
    public void commitAndClose();
    
    /**
     * 排他チェックフィールド名を指定。<BR>
     * 排他チェックを行わない場合は "" か null を設定する。
     * デフォルトは null。
     * @param exclusiveCheckField 排他チェックフィールド名
     */
    public void setExclusiveCheckField(String exclusiveCheckField);

    /**
     * 排他チェックフィールド名を取得
     * @return exclusiveCheckField 排他チェックフィールド名
     */
    public String getExclusiveCheckField();

    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかをDAO個別に指定
     * @param throw_exception
     */
    public void setThrowException(boolean throw_exception);

    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかのフラグを取得
     * @return throwException
     */
    public boolean isThrowException();
    
    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか取得する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @return returnNull, true nullで返す。false 空のBeanインスタンスで返す。
     */
    public boolean isReturnNull();
    
    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか設定する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @param returnNull returnNull を設定します。true nullで返す。false 空のBeanインスタンスで返す。
     */
    public void setReturnNull(boolean returnNull);
    
    /**
     * レコードを追加する。<BR>
     * テーブルに値自動インクリメントが有効になっている場合は
     * それらDBMSの機能を利用してレコードを追加する。(INSERT文で値を指定しない)<BR><BR>
     * もし bean の対応するプロパティに値が設定されていても、
     * DBの対応カラムの属性が値自動インクリメント属性になっている場合は
     * 値を無視してINSERTを行う。
     * 
     * 強制自動インクリメントは以下のDBMS,属性のみ解釈可能。<BR>
     * DB2 GENERATED ALWAYS <BR>
     * MySQL AUTO NUMBER<BR>
     * 
     * @param bean 追加するレコードのデータを持つBean
     * @return 追加件数
     */
    public int create(Object bean);    
    
    /**
     * columnsで指定したカラムの値のみ設定してレコードを追加する。<BR>
     * テーブルに値自動インクリメントが有効になっている場合は
     * それらDBMSの機能を利用してレコードを追加する。(INSERT文で値を指定しない)<BR><BR>
     * もし bean の対応するプロパティに値が設定されていても、
     * DBの対応カラムの属性が値自動インクリメント属性になっている場合は
     * 値を無視してINSERTを行う。
     * 
     * 強制自動インクリメントは以下のDBMS,属性のみ解釈可能。<BR>
     * DB2 GENERATED ALWAYS <BR>
     * MySQL AUTO NUMBER<BR>
     * 
     * @param bean 追加するレコードのデータを持つBean
     * @param columns 値指定カラム名
     * @return 追加件数
     */
    public int create(Object bean, String[] columns);

    /**
     * レコードを追加する。プライマリキーが1つで数値型の場合は
     * PowerQLのシーケンステーブルから次のシーケンス値を取得して格納する。<BR>
     * レコード格納後、bean の該当プロパティに格納したシーケンス値を設定する。
     * @param bean  追加するレコードのデータを持つBean
     * @return 追加件数
     */
    public int createByPowerQLSeq(Object bean);

    /**
     * テーブルの全レコードを物理削除
     * @return 削除件数
     */
    public int removeByAll();  

    /**
     * where 条件を指定してレコードを物理削除
     * @return 削除件数
     */
    public int removeBy(String where);  

    /**
     * プライマリキーを指定してレコードを物理削除。<BR>
     * 複数のレコードを削除する場合は、Beanの配列かCollectionを引数に渡す。<BR><BR>
     * 排他チェックが有効な場合は排他チェックフィールドを参照して
     * 排他チェックを行う。<BR>
     * @param key プライマリキーの値
     * @return 削除件数
     */
    public int removeByPrimaryKey(Object key);  

    /**
     * プライマリキーを指定してレコードを物理削除。<BR>
     * 内部で removeByPrimaryKey(new Integer(key)) を実行している。
     * @param key プライマリキーの値
     * @return 削除件数
     */
    public int removeByPrimaryKey(int key);

    /**
     * Beanを指定してレコードを削除。<BR>
     * プライマリキーが複数フィールドの場合は removeByPrimaryKey の代わりに
     * こちらの remove メソッドを利用する。<BR>
     * 内部では削除時に全プライマリキーを参照して削除している。<BR><BR>
     * 複数のレコードを削除する場合は、Beanの配列かCollectionを引数に渡す。<BR><BR>
     * 排他チェックが有効な場合は排他チェックフィールドを参照して
     * 排他チェックを行う。<BR>
     * @param bean 削除するレコードのデータを持つBean
     * @return 削除件数
     */
    public int remove(Object bean);

    /**
     * 全レコード数を取得
     * @return 全レコード数
     */
    public int countByAll();
    
    /**
     * where 条件を指定してレコード数を取得
     * @param where 検索条件
     * @return 一致レコード数
     */
    public int countBy(String where);

    /**
     * 例外リスナークラスを返す
     * @return 例外リスナー
     */
    public POQLExceptionListener getExceptionListener();
    
    /**
     * 例外リスナークラスを設定する。<BR>
     * Managerのリスナーが設定されている場合は上書きされる。
     * @param exceptionListener 例外リスナー
     */
    public void setExceptionListener(POQLExceptionListener exceptionListener);
    
    /**
     * rtrimフラグを返す
     * @return rtrim を戻します。
     */
    public boolean isRtrim();

    /**
     * rtrimフラグを設定する
     * @param rtrim rtrimフラグ
     */
    public void setRtrim(boolean rtrim);
    
    /**
     * 最後に実行したSQLを出力する。
     * SQL内の?は変換しない
     * @return sql
     */
    public String getLastSQL();

    /**
     * 最後に実行したSQLのバインド値を出力する。
     * @return バインド値
     */
    public Object[] getLastBindValues();
    
    /**
     * 自動タイムスタンプ格納カラム名を設定
     */
    public void setAutoTimestampColName(String[] auto_timestamp_colname);
    
    /**
     * JTA等の外部トランザクション管理モードフラグ, true JTA利用, false JTA非利用(default)
     * @return useJTATransaction を戻します。
     */
    public boolean isUseJTATransaction();
    
    /**
     * JTA等の外部トランザクション管理モードフラグを設定<BR>
     * true JTA利用, false JTA非利用(default)
     * @param useJTATransaction useJTATransaction を設定。
     */
    public void setUseJTATransaction(boolean useJTATransaction);

	/**
	 * テーブルをDROPする
	 * @param DROP するTABLE
	 */
	public void dropTable(Class bean_class);
	
	
}