package jp.co.powerbeans.powerql.dao;

import java.util.Collection;

import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;

/**
 * <p>タイトル: POQLDynaViewDAO</p>
 * <p>説明: 動的にVIEWを作成し検索するメソッドを提供する汎用POQLDAOインターフェース。<BR>
 * ビジネスロジックは POQLManager#createViewDAO(Class, Class[], String) により
 * プログラム内部で動的なVIEW SQLを作成し
 * DBにアクセスする。<BR>
 * ここで言う動的なVIEWは、DAO生成時にCREATE VIEWを実行して生成するVIEWではなく、
 * DAO内部に結合するテーブル、結合条件を利用して生成したSQLを指す。
 * そのため一般的なVIEWを利用する場合は POQLViewDAO を利用すること。
 * <BR>
 * <BR>
 * MySQL4.1等VIEWを実装していないDBMSではこのインターフェースを利用することで
 * 大幅にコーディング量を減らすことができる。<BR><BR>
 * 
 * POQLDynaViewDAOは検索のみで、追加、更新、削除を行うメソッドは提供していない。<BR>
 * <BR>
 * <pre>
 * 例）  // MstCity,MstPrefテーブルを結合してPrefCityクラスで結果を複数件取得
 *       POQLManager bm = new POQLManager("org.postgresql.Driver",
 *               "jdbc:postgresql:bqltest", "trans",
 *               "t79rp90", "POQL_SEQUENCE");
 * 
 *       // DAO 生成
 *       POQLDynaViewDAO dao = bm.createViewDAO(PrefCity.class, // 結果取得モデル
 *               new Class[] { MstCity.class, MstPref.class },	      // 結合モデル(テーブル)
 *               "MstPref.prefCd = substring(MstCity.cityCd,1,2)");   // 結合条件
 * 
 *       // 検索
 *       Collection c = dao.findBy("regionName Like '%大通東%'");
 * 
 *       // c には PrefCity モデルインスタンスのリストが格納されている。
 * 
 *
 * デフォルト値
 *   getThrowException() = false
 *   isReturnNull() = true
 *   
 * </pre>
 * 　
 * 
 * </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/16</p>
 * @author 門田明彦
 * @version $Revision: 1.1 $
 */
public interface POQLDynaViewDAO {

    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかをDAO個別に指定
     * @param throw_exception
     */
    public void setThrowException(boolean throw_exception);

    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかのフラグを取得
     * @return throwException
     */
    public boolean isThrowException();
    
    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか取得する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @return returnNull, true nullで返す。false 空のBeanインスタンスで返す。
     */
    public boolean isReturnNull();
    
    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか設定する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @param returnNull returnNull を設定します。true nullで返す。false 空のBeanインスタンスで返す。
     */
    public void setReturnNull(boolean returnNull);
    
    /**
     * 条件を指定して検索
     * @param where where SQL
     * @return 検索結果。複数
     */
    public Collection findBy(String where);

    /**
     * 条件を指定して検索
     * @param where where SQL
     * @param order_by ORDER BY SQL
     * @return 検索結果。複数
     */
    public Collection findBy(String where, String order_by);

    /**
     * 条件を指定して検索
     * @param where where SQL
     * @return 検索結果。1件以外はエラー
     */
    public Object find1By(String where); 


    /**
     * 全レコードを検索
     * @return 検索結果。複数
     */
    public Collection findByAll();
    
    /**
     * 条件を指定して検索
     * @param where where SQL
     * @param order_by ORDER BY SQL
     * @return 検索結果。1件以外はエラー
     */
    public Object find1By(String where, String order_by); 

    /**
     * 全レコード数を取得
     * @return 全レコード数
     */
    public int countByAll();
    
    /**
     * where 条件を指定してレコード数を取得
     * @param where 検索条件
     * @return 一致レコード数
     */
    public int countBy(String where);

    /**
     * 例外リスナークラスを返す
     * @return 例外リスナー
     */
    public POQLExceptionListener getExceptionListener();
    
    /**
     * 例外リスナークラスを設定する。<BR>
     * Managerのリスナーが設定されている場合は上書きされる。
     * @param exceptionListener 例外リスナー
     */
    public void setExceptionListener(POQLExceptionListener exceptionListener);

}
