package jp.co.powerbeans.powerql.vendor;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;

import jp.co.powerbeans.powerql.ColumnProperty;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.POQLUtil;
import jp.co.powerbeans.powerql.exceptions.POQLException;

import org.apache.oro.text.perl.Perl5Util;


/**
 * <p>タイトル: DBDependQLStatementOracle</p>
 * <p>説明: DBDependQLStatement の Oracle実装。</p>
 * 
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/10</p>
 * @author 門田明彦
 * @version $Revision: 1.5 $
 */
public class DBDependQLStatementOracle extends DBDependQLStatementSupport {

	private static final String DB_TYPE = "Oracle";

  /** typmapping */
  private static Hashtable db2javaTypeMap = new Hashtable();

  /** type mapping data key */
  private static Object[][] db2javaMap = { { "TINYINT", Short.class }, {
      "TINYINT\\([0-9]+\\)", Short.class }, {
      "SMALLINT", Integer.class }, {
      "SMALLINT\\([0-9]+\\)", Integer.class }, {
      "MEDIUMINT", Integer.class }, {
      "MEDIUMINT\\([0-9]+\\)", Integer.class }, {
      "INT", Integer.class }, {
      "INT\\([0-9]+\\)", Integer.class }, {
      "INTEGER", Integer.class }, {
      "INTEGER\\([0-9]+\\)", Integer.class }, {
      "BIGINT", Long.class }, {
      "BIGINT\\([0-9]+\\)", Long.class }, {

      "NUMERIC\\([0-9]+,[0-9]+\\)", Double.class }, {
      "DECIMAL\\([0-9]+,[0-9]+\\)", Double.class }, {
      "REAL\\([0-9]+,[0-9]+\\)", Float.class }, {
      "DOUBLE PRECISION\\([0-9]+,[0-9]+\\)", Double.class }, {

      "DATE", Timestamp.class }, {
      "DATETIME", Timestamp.class }, {
      "TIMESTAMP", Timestamp.class }, {
      "TIMESTAMP\\([0-9]+\\)", Timestamp.class }, {
      "TIME", Timestamp.class }, {

      "TINYBLOB", String.class }, {
      "TINYTEXT", String.class }, {
      "BLOB", String.class }, {
      "TEXT", String.class }, {
      "MEDIUMBLOB", String.class }, {
      "MEDIUMTEXT", String.class }, {
      "LONGBLOB", String.class }, {
      "LONGTEXT", String.class }, {

      "CHAR", String.class }, {
      "VARCHAR", String.class }, {
      "CHAR\\([0-9]+\\)", String.class }, {
      "VARCHAR\\([0-9]+\\)", String.class }, };

  /** type mapping data key */
  private static Object[][] java2dbMap = { { Byte.class, "TINYINT" }, {
      Short.class, "TINYINT" }, {
      Character.class, "TINYINT" }, {
      Integer.class, "INT" }, {
      Long.class, "BIGINT" }, {
      Double.class, "DOUBLE PRECISION" }, {

      String.class, "VARCHAR" }, {
      Timestamp.class, "DATE" }
  };
  /** type mapping */
  private static Hashtable java2dbTypeMap = new Hashtable();

  static {
    // typeMap.put(Oracle Type (regex), Java type)
    for (int i = 0; i < db2javaMap.length; i++) {
      db2javaTypeMap.put(db2javaMap[i][0], db2javaMap[i][1]);
    }

    // typeMap.put(Java type, MySQL type)
    for (int i = 0; i < java2dbMap.length; i++) {
      java2dbTypeMap.put(java2dbMap[i][0], java2dbMap[i][1]);
    }
  }



	/** regexp lib */
	private static Perl5Util perl = new Perl5Util();

	/** Oracld > JDBC type mapping */
	private static Hashtable typeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] typeArray =
		{ { "NUMBER\\([0-9]+\\)", Integer.class }, {
			"NUMBER\\([0-9]+,[0-9]+\\)", Double.class }, {
			"CHAR\\([0-9]+\\)", String.class }, {
			"VARCHAR2\\([0-9]+\\)", String.class }, {
			"NUMBER", Integer.class }, {
			"DATE", Timestamp.class }, {
            "CHAR", String.class }, {
            "NVARCHAR2", String.class }, {
            "LONG", Long.class }, {
            "RAW", byte[].class }, {
            "ROWID", String.class }, {
            "UROWID", String.class }, {
            "NCHAR", String.class }, {
            "UROWID", String.class }, {
            "CLOB", String.class }, {
            "NCLOB", String.class }, {
            "BLOB", byte[].class }, {
            "VARCHAR2", String.class }, };

	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < typeArray.length; i++) {
			typeMap.put(typeArray[i][0], typeArray[i][1]);
		}
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getColumnPropertyList()
	 */
	public ArrayList getColumnPropertyList(String table, String schema)
		throws SQLException, POQLException {

    ArrayList list = new ArrayList();
    String sql =
      "SELECT "
        + " user_tab_columns.column_name, "
        + " data_type, "
        + " data_length, "
        + " nullable, "
        + " data_default, "
        + " C.PKEY_NAME "
        + " FROM "
        + "     user_tab_columns, "
        + "  (SELECT"
        + "    user_constraints.CONSTRAINT_NAME AS PKEY_NAME,"
        + "    user_constraints.CONSTRAINT_TYPE,"
        + "    user_cons_columns.CONSTRAINT_NAME,"
        + "    user_cons_columns.COLUMN_NAME "
        + "   FROM"
        + "    user_constraints,"
        + "    user_cons_columns"
        + "    WHERE"
        + "    user_constraints.table_name      = user_cons_columns.table_name(+)  AND"
        + "    user_constraints.constraint_name = user_cons_columns.constraint_name(+)  AND"
        + "    upper(user_constraints.table_name) = upper('"
        + table
        + "') AND CONSTRAINT_TYPE = 'P' ) C "
      //					+ "    user_constraints.CONSTRAINT_TYPE = 'P') C"
      +" WHERE "
        + "     upper(table_name) = upper('"
        + table
        + "') AND "
        + " user_tab_columns.column_name = C.COLUMN_NAME(+) "
        + " ORDER BY "
        + "     column_id";

    ResultSet result = statement.executeQuery(sql.toString());

    // 一時保管用Hash
    HashMap colMap = new HashMap();

    while (result.next()) {
      // 各カラム情報を取得
      // 1つのフィールドに複数のキーを設定していた場合
      // フィールドを複数列で取得してしまうので
      // フィールドが重複しないように一旦 colMap に入れながらリスト化

      ColumnProperty cp = new ColumnProperty();
      cp.setName(result.getString("column_name"));

      if (colMap.containsKey(cp.getName())) {
        continue;
      }

      cp.setTypeAsDB(result.getString("data_type"));
      cp.setNullable(result.getString("nullable").equals("Y"));
      cp.setDefaultValue(result.getObject("data_default"));
      String pkey_name = result.getString("PKEY_NAME");
      cp.setPrimaryKey(pkey_name != null && pkey_name.length() > 0);
      cp.setType(getJavaTypeFromDBType(cp.getTypeAsDB()));

      colMap.put(cp.getName(), cp);
    }
    result.close();

    // list化
    for (Iterator it = colMap.values().iterator(); it.hasNext();) {
      list.add((ColumnProperty)it.next());
    }

    return list;
	}

	/**
	 * getJavaTypeFromDBType<BR>
	 * DB(Oracle)のカラムの型からJAVAの型をClassで取得
	 * @param type_as_DB DB(Oracle)の型
	 * @return JAVAの型
	 */
	private Class getJavaTypeFromDBType(String type_as_DB)
		throws POQLException {

		String upper = type_as_DB.toUpperCase();

		if (typeMap.containsKey(upper)) {
			// 完全一致
			return (Class) typeMap.get(upper);

		} else {
			// 全パターンと正規表現比較
			for (int i = 0; i < typeArray.length; i++) {
				if (perl.match(upper, (String) typeArray[i][0])) {
					return (Class) typeArray[i][1];
				}
			}

		}

		// 一致する型無し
		throw new POQLException(
			"not recoganize db TYPE(" + DB_TYPE + ") " + type_as_DB);
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#isExistTable(java.lang.String)
	 */
	public boolean isExistTable(String table_name) throws SQLException {

		String sql =
			"SELECT table_name FROM user_tables WHERE UPPER(TABLE_NAME) = UPPER('"
				+ table_name
				+ "')";
		ResultSet result = statement.executeQuery(sql.toString());
		boolean is_exist = false;

		while (result.next()) {
			int c = 1;

			String table = result.getString(c++);

			if (table != null
				&& table_name != null
				&& table.toUpperCase().equals(table_name.toUpperCase())) {
				// 存在
				is_exist = true;
				break;
			}
		}

		result.close();
		return is_exist;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#createTable(java.lang.String, jp.co.powerbeans.powerql.ColumnProperty[])
	 */
	public void createTable(String table_name, ColumnProperty[] cp)
		throws POQLException, SQLException {

      if (cp == null || cp.length == 0) {
        throw new POQLException("ColumnProperty value is invalid:" + cp);
      }

      StringBuffer sql = new StringBuffer("CREATE TABLE " + table_name + "(");

      // column
      for (int i = 0; i < cp.length; i++) {
        sql.append(getJava2DBType(cp[i]) + ", ");
      }

      // pk
      sql.append(
        "constraint pk_"
          + table_name
          + " primary key ("
          + cp[0].getName()
          + ") ");
      sql.append(" )");

      statement.executeUpdate(sql.toString());
	}

  /**
   * getJava2DBType<BR>
   * ColumnProperty から CREATE SQL 用カラム定義を作成
   * @param property
   * @return CREATE SQL 用カラム定義を作成
   */
  private String getJava2DBType(ColumnProperty cp) throws POQLException {

    StringBuffer sql = new StringBuffer(cp.getName());
    if (!java2dbTypeMap.containsKey(cp.getType())) {
      // JAVAの型が設定されていない
      throw new POQLException(
        "cannot get type:"
          + cp.getType()
          + "(java 2 db) ["
          + DB_TYPE
          + "]");
    }

    sql.append(" " + (String) java2dbTypeMap.get(cp.getType()));

    if (cp.getSize() > 0) {
      sql.append("(" + cp.getSize() + ")");
    }

    if (cp.getDefaultValue() != null
      && cp.getDefaultValue().toString().length() > 0) {
      sql.append(" DEFAULT " + cp.getDefaultValue());
    }

    if (!cp.isNullable()) {
      sql.append(" not null");
    }

    sql.append(" ");
    return sql.toString();
  }

  /**
   * getSelectSQL<BR>
   * パラメータの条件からSQLを作成。
   * Oracleはgroup by がある場合はgroup by で指定したフィールドだけ SELECT 句に指定して取得。
   * @param tableName テーブル名
   * @param bpList BeanPropertyList
   * @param where where句
   * @param groupby groupby句
   * @param orderby orderby句
   * @return select SQL
   */
  public String getSelectSQL(String tableName, ArrayList bpList, String where, String groupby, String orderby) {

    StringBuffer sql = null;
    
    // group by がある場合はSELECT句で利用
    if (groupby != null && groupby.trim().length() > 0) {
      sql = new StringBuffer("SELECT " + groupby + " FROM " + tableName);
      
    } else {
      sql = new StringBuffer("SELECT * FROM " + tableName);
    }
    
    if (where != null && where.trim().length() > 0) {
      sql.append(" WHERE " + where.trim());
    }
    
    if (groupby != null && groupby.trim().length() > 0) {
      sql.append(" GROUP BY " + groupby.trim());
    }
    
    if (orderby != null && orderby.trim().length() > 0) {
      sql.append(" ORDER BY " + orderby.trim());
    }
    return sql.toString();
  }

    /*
     * (Javadoc なし)
     * 
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW()
     */
    public String getSQL_NOW() {
        return " SYSDATE ";
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW(java.lang.String)
     */
    public String getSQL_NOW(String column_type) {

        return getSQL_NOW();
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#formatDateVal(java.util.Date, java.lang.String)
     */
    public String formatDateVal(Date date, String column_type) {
        // Oracleは日付型のみ格納可能
        // ただしフォーマットはDateTimeにする
        
        if (column_type.equalsIgnoreCase("DATE")) {
            return "TO_DATE('" + POQLUtil.formatDateTime(date) + "','YYYY-MM-DD HH24:MI:SS')";
        } else {
            // 未対応のカラム型. とりあえず DateTimeにする
            return "'" + POQLUtil.formatDateTime(date) + "'";
        }
    }

//    /* (non-Javadoc)
//     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getBlobClass()
//     */
//    public Class getBlobClass() {
//        return super.getBlobClass("oracle.sql.BLOB");
//    }
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableAllSql()
	 */
	public String getSysSelectTableAllSql() {
		throw new RuntimeException("not support getSysSelectTableAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectSchemaAllSql()
	 */
	public String getSysSelectSchemaAllSql() {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}
	

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableBySchemaSql(java.lang.String)
	 */
	public String getSysSelectTableBySchemaSql(String schema) {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}
	
    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#escape(java.lang.String)
     */
    public String escape(String sql) {
    	// TODO Oracle のエスケープ文字は "
    	return super.escape(sql);
    }
    
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getGeneratedKey(jp.co.powerbeans.powerql.POQLPreparedStatement)
	 */
	public Object getGeneratedKey(POQLPreparedStatement st) throws SQLException {
		ResultSet key_rset = st.getGeneratedKeys();
		// Oracleは ROW ID 取得後, ROWIDを元に再度検索
		if (key_rset == null) {
			return Integer.valueOf(0);
		}
		
	    int id = 0;
		String select =
		    "select id from hoge where rowid = ?";
		PreparedStatement stmt2 = this.connection.prepareStatement(select);
		stmt2.setObject(1, key_rset.getObject(1));
		ResultSet rset2 = stmt2.executeQuery();
		if (rset2.next()) {
		    id = rset2.getInt(1);
		}
		return Integer.valueOf(id);
	}
}
