package jp.co.powerbeans.powerql;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import jp.co.powerbeans.powerql.dao.POQLAutoJoinDAO;
import jp.co.powerbeans.powerql.dao.POQLBaseDAO;
import jp.co.powerbeans.powerql.dao.POQLDAO;
import jp.co.powerbeans.powerql.dao.POQLDynaViewDAO;
import jp.co.powerbeans.powerql.dao.POQLMapBaseDAO;
import jp.co.powerbeans.powerql.dao.POQLMapDAO;
import jp.co.powerbeans.powerql.dao.POQLRawSqlDAO;
import jp.co.powerbeans.powerql.dao.POQLStandardDAO;
import jp.co.powerbeans.powerql.dao.POQLStandardDynaViewDAO;
import jp.co.powerbeans.powerql.dao.POQLStandardRawSqlDAO;
import jp.co.powerbeans.powerql.dao.POQLStandardViewDAO;
import jp.co.powerbeans.powerql.dao.POQLStanderdAutoJoinDAO;
import jp.co.powerbeans.powerql.dao.POQLViewDAO;
import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * <p>タイトル: POQLManager</p>
 * <p>説明: 
 * JDBC ドライバ,ユーザー、パスワード情報を保持し
 * POQLStatement を返す。<BR>
 * 通常アプリケーションで１つのインスタンスを保持する。
 * 
 * </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/07</p>
 * @author 門田明彦
 * @version $Revision: 1.8 $
 */
public class POQLManager implements IPOQLManagerIF {

	/** JDBC Driver */
	private String driverName = "";

	/** JDBC url */
	private String url = "";

	/** JDBC user */
	private String user = "";

	/** JDBC password */
	private String password = "";

	/** JDBC Driver */
	private Driver driver;

	/** Database Vendor */
	private int vendorType;

	/** auto Create Table */
	private boolean autoCreateTable = false;

	/** sequenceTable */
	private String sequenceTable;

	/** DataSource (null の場合はJDBCを利用, not null の場合は DataSourceを利用) */
	private DataSource ds;


	// --- static 

	/** Vendor UNKNOWN */
	public static int VENDOR_UNKNOWN = 0;

	/** Vendor Oracle */
	public static int VENDOR_ORACLE = 1;

	/** Vendor Mysql */
	public static int VENDOR_MYSQL = 2;

	/** Vendor DB2 */
    public static final int VENDOR_DB2 = 3;

    /** Vendor PostgreSQL */
    public static final int VENDOR_POSTGRESQL = 4; 
    
  /** Vedor HSQL Database Engine */
    public static final int VENDOR_HSQL = 5;

	/** Vendor PostagreSQL */
	//	public static int VENDOR_POSTAGRE_SQL = 3;

	/** Vendor HypersonicSQL */
	//	public static int VENDOR_HYPERSONIC_SQL = 4;
    
    /** ベンダー実装のBlobクラス */
//    private Class blobClass;

    /** 独自実装DAOクラスマップ　 (key=Beanクラス名, value=DAOクラス名) */
    private Map customDAO; 
    
    /** 例外リスナー */
    private POQLExceptionListener exceptionListener;

    /** 例外発生時に例外をスローするかどうか(POQLManager自身) */
    private boolean throwException;

    static final private Hashtable db2javaTypeMap;
    static 
    {
        db2javaTypeMap = new Hashtable();
        Object[][] db2javaMap = {
            {
                "INTEGER", int.class
            },{
                "INT", int.class
            },{
                "DOUBLE", double.class
            },{
                "DOUBLE PRECISION", double.class
            },{
                "FLOAT", double.class
            },{
                "VARCHAR", java.lang.String.class
            },{
                "VARCHAR_IGNORECASE", java.lang.String.class
            },{
                "CHAR", java.lang.String.class
            },{
                "CHARACTER", java.lang.String.class
            },{
              "LONGVARCHAR", java.lang.String.class
            },{
              "TEXT", java.lang.String.class
            },{
                "DATE", java.sql.Date.class
            },{
                "TIME", java.sql.Time.class
            },{
                "TIMESTAMP", java.sql.Timestamp.class
            },{
                "DATETIME", java.sql.Timestamp.class
            },{
                "DECIMAL", java.math.BigDecimal.class
            },{
                "NUMERIC", java.math.BigDecimal.class
            },{
                "BOOLEAN", boolean.class
            },{
                "BIT", java.lang.Boolean.class
            },{
                "TINYINT", byte.class
            },{
                "SMALLINT", short.class
            },{
                "BIGINT", long.class
            },{
                "REAL", double.class
            },{
                "BINARY", byte[].class
            },{
                "LONGVARBINARY", byte[].class
            },{
                "OTHER", java.lang.Object.class
            },{
                "OBJECT", java.lang.Object.class
            }
        };
        
        for(int i = 0; i < db2javaMap.length; i++) {
            db2javaTypeMap.put(db2javaMap[i][0], db2javaMap[i][1]);
        }
    }

    
    /**
     * コンストラクタ
     * @param driver_name ドライバクラスパス
     * @param url 接続URL
     * @param user 接続ユーザーID
     * @param password 接続パスワード
     * @throws POQLException
     */
    public POQLManager(
        String driver_name,
        String url,
        String user,
        String password)
        throws POQLException {
        super();

        initialize(
            driver_name,
            url,
            user,
            password,
            null,
            null);
   }
    
    /**
	 * コンストラクタ
	 * @param driver_name ドライバクラスパス
	 * @param url 接続URL
	 * @param user 接続ユーザーID
	 * @param password 接続パスワード
	 * @param sequence_table シーケンステーブル名
	 * @throws POQLException
	 */
	public POQLManager(
		String driver_name,
		String url,
		String user,
		String password,
		String sequence_table)
		throws POQLException {
		super();

		initialize(
			driver_name,
			url,
			user,
			password,
			sequence_table,
			null);
	}
	/**
	 * コンストラクタ
	 * @param driver_name ドライバクラスパス
	 * @param url 接続URL
	 * @param user 接続ユーザーID
	 * @param password 接続パスワード
	 * @param sequence_table シーケンステーブル名
	 * @param custom_dao 独自実装DAOのマップ
	 * @throws POQLException
	 */
	public POQLManager(
		String driver_name,
		String url,
		String user,
		String password,
		String sequence_table,
		Map custom_dao)
		throws POQLException {
		super();

		initialize(
			driver_name,
			url,
			user,
			password,
			sequence_table,
			custom_dao);
	}
	
    
    /**
     * コンストラクタ
     * (DataSource から生成)
     * @param ds DataSource
     */
    public POQLManager(DataSource ds) throws POQLException {

        initialize(ds, null, null); 
    }
    
    /**
     * コンストラクタ
     * (DataSource から生成)
     * @param ds DataSource
     * @param sequence_table シーケンステーブル名
     */
    public POQLManager(DataSource ds, String sequence_table) throws POQLException {

        initialize(ds, sequence_table, null); 
    }
	
	/**
	 * コンストラクタ
	 * (DataSource から生成)
	 * @param ds DataSource
	 * @param sequence_table シーケンステーブル名
	 * @param custom_dao
	 */
	public POQLManager(DataSource ds, String sequence_table, Map custom_dao) throws POQLException {

		initialize(ds, sequence_table, custom_dao); 
	}
	
	/**
	 * initialize<BR>
	 * @param ds
	 * @param vendor_type
	 * @param sequence_table
	 */
	private void initialize(DataSource ds, String sequence_table,
	        Map custom_dao) throws POQLException {
		
	    this.ds = ds;
		this.sequenceTable = sequence_table;

	    // ベンダータイプを判定
	    parseVendar();
	    
		Log.println("[POQLManager#initialize() start --->");
		
	    try {
            // customDAO をチェック
            validateCustomDAO(custom_dao);
        } catch (ClassNotFoundException e) {
            throw new POQLException(e);
        }

		// シーケンス管理テーブル作成
		createSeqTableSafe();
	}

	/**
	 * ベンダータイプを判定
     * @throws POQLException
     */
    private void parseVendar() throws POQLException {
	    POQLTransaction bt = getPowerQLTransaction();
	    Connection con = null;
        vendorType = POQLManager.VENDOR_UNKNOWN;
	    String db_name = "";
        try {
    	    con = bt.getConnection();
            db_name = con.getMetaData().getDatabaseProductName();
            String product_names[] = {"", "Oracle", "MySQL", "DB2", "PostgreSQL", "HSQL Database Engine"};
    	    for(int i = 1; i < product_names.length; i++) {
    	        if (db_name.startsWith(product_names[i])) {
    	            // 一致したのでベンダー決定
    	            this.vendorType = i;
    	            break;
    	        }
    	    }
    	    if (vendorType == POQLManager.VENDOR_UNKNOWN) {
    	        throw new POQLException("can't recoganize database product name:" + db_name);
    	    }
    	    
    	    Log.println("vender:" +db_name);
    	    
//    	    // Blob実装を見つけておく
//    	    DBDependQLStatement dep = DBDependQLStatementFactory.create(
//    	            con, this);
//    	    this.blobClass = dep.getBlobClass();
    	    
        } catch (SQLException e) {
            throw new POQLException(e);
        } finally {
            if (con != null) {
                try {
                    con.close();
                } catch (SQLException e1) {
                    e1.printStackTrace();
                }
            }
        }
    }

    /**
	 * initialize<BR>
	 * @param driver_name
	 * @param url
	 * @param user
	 * @param password
	 * @param sequence_table
	 * @throws POQLException
	 */
	private void initialize(
		String driver_name,
		String url,
		String user,
		String password,
		String sequence_table,
		Map custom_dao)
		throws POQLException {
		driverName = driver_name;
		this.url = url;
		this.user = user;
		this.password = password;
		this.sequenceTable = sequence_table;
		this.customDAO = custom_dao; 

		try {
			// ドライバをロード
			driver = (Driver) Class.forName(driver_name).newInstance();
			
		    // customDAO をチェック
		    validateCustomDAO(custom_dao);
			
		} catch (InstantiationException e) {
			throw new POQLException(e);
		} catch (IllegalAccessException e) {
			throw new POQLException(e);
		} catch (ClassNotFoundException e) {
			throw new POQLException(e);
//				"Driver Class not found:" + driverName + "\n" + e);
		}

	    // ベンダータイプを判定
	    parseVendar();
	    
        // シーケンス管理テーブル作成
        createSeqTableSafe();
	}

	/**
     * sequenceTable が null でなければシーケンステーブルを作成.
     * sequenceTable が null の場合は何もしない
     * @throws POQLException
     */
    private void createSeqTableSafe() throws POQLException {
        
        if (sequenceTable == null) {
            return;
        }
        
        if (isExistTable(sequenceTable)) {
            Log.println("[POQLManager#initialize() end <---");
            return;
        }

        Log.println("[POQLManager#initialize() creating SequenceTable");

        // 存在しないので作成
        createSequenceTable();

        Log.println("[POQLManager#initialize() end <---");
    }
    /**
	 * custom_dao の各クラスが正しいかチェックする。
	 * クラスが見つからない場合は ClassNotFoundException をスローする。
     * @param custom_dao
	 * @throws ClassNotFoundException
     */
    private void validateCustomDAO(Map custom_dao) throws ClassNotFoundException { 
        
        if (custom_dao == null) {
            return;
        }
        
        for(Iterator it = custom_dao.keySet().iterator(); it.hasNext();) {
            String bean = (String)it.next();
            String dao = (String)custom_dao.get(bean);
            
            Class.forName(bean);
            Class dao_class = Class.forName(dao);
            if (dao_class.isAssignableFrom(POQLDAO.class) ||
                dao_class.isAssignableFrom(POQLDynaViewDAO.class) ||
                dao_class.isAssignableFrom(POQLViewDAO.class)) {
                // OK
            } else {
                throw new IllegalArgumentException("CustomDAO:" + dao_class + 
                        " is not implement POQLDAO or POQLDynaViewDAO or POQLViewDAO");
            }
        }
        
    }

    /**
	 * createSequenceTable<BR>
	 * シーケンス管理テーブル作成
	 * 
	 */
	private void createSequenceTable() throws POQLException {
		POQLTransaction bt = getPowerQLTransaction();

		// カラム作成
		ColumnProperty[] cp = new ColumnProperty[2];
		
//		// カラム:tableID
//		cp[0] = new ColumnProperty();
//		cp[0].setName("tableID");
//		cp[0].setNullable(false);
//		cp[0].setType(Integer.class);
//		//cp[0].setTypeAsDB("NUMBER");
//		cp[0].setDefaultValue(new Integer(1));
//		cp[0].setPrimaryKey(true);

		// カラム:tableName
		cp[0] = new ColumnProperty();
		cp[0].setName("tableName");
		cp[0].setNullable(false);
		cp[0].setType(String.class);
		//cp[1].setTypeAsDB("NUMBER");
		cp[0].setDefaultValue("");
		cp[0].setPrimaryKey(true);
		cp[0].setSize(100);

		// カラム:sequence
		cp[1] = new ColumnProperty();
		cp[1].setName("sequence");
		cp[1].setNullable(false);
		cp[1].setType(Integer.class);
		//cp[1].setTypeAsDB("NUMBER");
		cp[1].setDefaultValue(new Integer(1));
		cp[1].setPrimaryKey(false);

		try {
			DBDependQLStatement dqs =
				DBDependQLStatementFactory.create(bt.getConnection(), bt.mgr);

		Log.println("[POQLManager#createSequenceTable() creating SequenceTable");
			dqs.createTable(sequenceTable, cp);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new POQLException(
				"cannot create table:" + sequenceTable,
				e);
		} finally {
			bt.close();
		}
		Log.println("[POQLManager#createSequenceTable() create SequenceTable ok");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#isExistTable(java.lang.String)
	 */
	public boolean isExistTable(String table_name) throws POQLException {
	
		POQLTransaction bt = getPowerQLTransaction();
		try {
			DBDependQLStatement dqs =
				DBDependQLStatementFactory.create(bt.getConnection(), bt.mgr);

			return dqs.isExistTable(table_name);
			
		} catch (SQLException e) {
			e.printStackTrace();
			throw new POQLException(
				"cannot check exist table:" + table_name,
				e);
		} finally {
			bt.close();
		}
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getDriverName()
	 */
	public String getDriverName() {
		return driverName;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getPassword()
	 */
	public String getPassword() {
		return password;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getUrl()
	 */
	public String getUrl() {
		return url;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getUser()
	 */
	public String getUser() {
		return user;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getPowerQLTransaction()
	 */
	public POQLTransaction getPowerQLTransaction() throws POQLException {

		if (ds != null) {
			// get connection form DataSource
			try {
				java.sql.Connection con = ds.getConnection();
				Log.println("connect. (DataSource): " + ds + " con: " + con);
				return new POQLTransaction(con, this);
			} catch (SQLException e) {
				throw new POQLException(e.getMessage() + "\nDataSource connect error :" + ds);
			}
		}

		// get connection form JDBC
		if (driver == null) {
			throw new POQLException("not loaded driver.");
		}

		Properties props = new Properties();
		props.put("user", user);
		props.put("password", password);
		try {
			java.sql.Connection con = driver.connect(url, props);
			Log.println("connect. user:" + user + " password:XXXX url:" + url + " con:" + con);
			return new POQLTransaction(con, this);

		} catch (SQLException e) {
		    e.printStackTrace();
			throw new POQLException(e.getMessage() + "\nJDBC connect error (check parameter:user " + user + ", url " + url, e);
		}
	}

////	/**
////	 * getPowerQLTransaction<BR>
////	 * user, password, url を渡してConnection を持つ POQLTransaction を作成して返す
////	 * @param user
////	 * @param password
////	 * @param url
////	 * @return
////	 */
////	public POQLTransaction getPowerQLTransaction(String user, String password, String url) throws POQLException {
////
////		if (driver == null) {
////			throw new POQLException("not loaded driver.");
////		}
////
////		Properties props = new Properties();
////		props.put("user", user);
////		props.put("password", password);
////		try {
////			java.sql.Connection con = driver.connect(url, props);
////			return new POQLTransaction(con, this);
////
////		} catch (SQLException e) {
////			throw new POQLException(e);
////		}
////	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getPowerQLTransaction(java.sql.Connection)
	 */
	public POQLTransaction getPowerQLTransaction(java.sql.Connection con) throws POQLException {

		return new POQLTransaction(con, this);
	}


	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getVendorType()
	 */
	public int getVendorType() {
		return vendorType;
	}

// まだ自動テーブル作成を実行していなためコメント
//	/**
//	 * isAutoCrateTable<BR>
//	 * @return 
//	 */
//	public boolean isAutoCreateTable() {
//		return autoCreateTable;
//	}
//
//	/**
//	 * setAutoCrateTable<BR>
//	 * @param b
//	 */
//	public void setAutoCreateTable(boolean b) {
//		autoCreateTable = b;
//	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getSequenceTable()
	 */
	public String getSequenceTable() {
		return sequenceTable;
	}

    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createDAO(java.lang.Class)
	 */
    public POQLDAO createDAO(Class c) {
        
        if (customDAO != null && customDAO.containsKey(c.getName())) {
            try {
                return (POQLDAO) ((Class)customDAO.get(c.getName())).newInstance();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        
        // 生成に失敗した場合、現在はPowerQLStandardDAOのみを返す。
        POQLBaseDAO daoimpl = new POQLStandardDAO(this,c);
        
        return daoimpl;
    }

    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createViewDAO(java.lang.Class, java.lang.Class[], java.lang.String)
	 */
    public POQLDynaViewDAO createViewDAO(Class c, Class[] table_beans, String join_sql) {
        
        
        if (customDAO != null && customDAO.containsKey(c.getName())) {
            try {
                return (POQLDynaViewDAO) ((Class)customDAO.get(c.getName())).newInstance();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        
        // 現在はPOQLStandardDynaViewDAOのみを返す。
        return new POQLStandardDynaViewDAO(this, c, table_beans, join_sql);
    }
    
    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createAutoJoinDAO(java.lang.Class, java.lang.Class[])
	 */
    public POQLAutoJoinDAO createAutoJoinDAO(Class c, Class[] table_beans) {
        
    	// 一定の法則にしたがって join_sql を作成する
    	// 基本的にtable_beans[0]のプロパティ名とtable_beans[1]のプロパティが後方一致し
    	// 末尾がIdであれば結合キーとする
    	// 結合イメージ
    	// table_beans[0].xxxId <-> table_beans[1].xxxId
    	// table_beans[0].yyyId <-> table_beans[2].yyyId
    	StringBuffer join = new StringBuffer();
    	Map parent_map = new HashMap();
    	List child_ids_list = new ArrayList();
    	
    	try {
			// 親テーブルのIDをMap化
			if (table_beans != null && table_beans.length > 0){
			    BeanInfo info = Introspector.getBeanInfo(table_beans[0]);
			    PropertyDescriptor[] properties = info.getPropertyDescriptors();

			    for (int i = 0; i < properties.length; i++) {
			        // プロパティ名とカラム名から一致するカラムを探し
			        // 一致したらBeanPropertyを生成してbp_listに追加
			        PropertyDescriptor p = properties[i];
			        String name = p.getName();
			        if ("class".equals(name) || !name.endsWith("Id")) {
			        	continue;
			        }
			        parent_map.put(name.toUpperCase(), null);
			    }
			}
			
			// 子テーブルをループして各IDのMapをリスト化
			for (int j = 1; j < table_beans.length; j++) {
				List ids = new ArrayList();
			    BeanInfo info = Introspector.getBeanInfo(table_beans[j]);
			    PropertyDescriptor[] properties = info.getPropertyDescriptors();

			    for (int i = 0; i < properties.length; i++) {
			        // プロパティ名とカラム名から一致するカラムを探し
			        // 一致したらBeanPropertyを生成してbp_listに追加
			        PropertyDescriptor p = properties[i];
			        String name = p.getName();
			        if ("class".equals(name) || !name.endsWith("Id")) {
			        	continue;
			        }
			        ids.add(new String[]{"" + j, name.toUpperCase()});
			    }
			    child_ids_list.add(ids);
			}
			
			// 親テーブルのIDに関連付け
			for (Iterator iter = child_ids_list.iterator(); iter.hasNext();) {
				List ids = (List) iter.next();
				for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
					String[] vals = (String[]) iterator.next();
					String class_index = vals[0];
					String child_id = vals[1];
					
					// 親テーブルの各IDと後方一致検索し一致したら格納
					for(Iterator pit = parent_map.keySet().iterator(); pit.hasNext();) {
						String pid = (String) pit.next();
						if (parent_map.get(pid) != null || !pid.endsWith(child_id)) {
							continue; // unmatch
						}
						// 一致したので格納
						parent_map.put(pid, "A" + class_index + "." + child_id);
						break;
					}
					
				}
			}
			
//			// 関連付けの定義どおりにSQLを作成
//			CPCache.put()
//			
//			
//			
//			join.append("SELECT * FROM "); TODO SELECT フィールド別名を作成する. class[1] 以降のフィールドは A1.teamName AS myTeamId_teamName (= pid_child_)
//			(あるいは検索時に)CPCacheを利用してフィールド名を取得する
			for (int i = 0; i < table_beans.length; i++) {
				if (i > 0) {
					join.append(",");
				}
				join.append(" " + POQLUtil.className2TableName(table_beans[i]) + (" A" + i) + " ");
			}
			join.append(" WHERE ");
			int n = 0;
			for(Iterator it = parent_map.keySet().iterator(); it.hasNext();) {
				String pid = (String) it.next();
				String child_id = (String) parent_map.get(pid);
				if (child_id == null) {
					// 関連が見つからないIDは無視
					continue;
				}
				if (n++ > 0) {
					join.append(" AND ");
				}
				join.append(" A0." + pid + " = " + child_id);
			}
			
	    	// TODO この関連(JOINSQL)をキャッシュ
			
		} catch (IntrospectionException e) {
			e.printStackTrace();
		}

        if (customDAO != null && customDAO.containsKey(c.getName())) {
            try {
                return (POQLAutoJoinDAO) ((Class)customDAO.get(c.getName())).newInstance();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        
        // 現在はPOQLStanderdAutoJoinDAOのみを返す。
        return new POQLStanderdAutoJoinDAO(this, c, table_beans, join.toString());
     }

    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createViewDAO(java.lang.Class)
	 */
    public POQLViewDAO createViewDAO(Class c) {
        
        if (customDAO != null && customDAO.containsKey(c.getName())) {
            try {
                return (POQLViewDAO) ((Class)customDAO.get(c.getName())).newInstance();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        
        // 現在はPowerQLStandardViewDAOのみ返す。
        return new POQLStandardViewDAO(this, c);
    }

    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createViewDAO(java.lang.Class, java.lang.String)
	 */
    public POQLDynaViewDAO createViewDAO(Class c, String sql) {
        if (customDAO != null && customDAO.containsKey(c.getName())) {
            try {
                return (POQLDynaViewDAO) ((Class)customDAO.get(c.getName())).newInstance();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        
        // 現在はPowerQLStandardViewDAOのみ返す。
        return new POQLStandardDynaViewDAO(this, c, sql);
    }
    
    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#createRawSqlDAO()
	 */
    public POQLRawSqlDAO createRawSqlDAO() {
//        if (customDAO != null && customDAO.containsKey(c.getName())) {
//            try {
//                return (POQLDynaViewDAO) ((Class)customDAO.get(c.getName())).newInstance();
//            } catch (InstantiationException e) {
//                e.printStackTrace();
//            } catch (IllegalAccessException e) {
//                e.printStackTrace();
//            }
//        }
        
        // 現在はPOQLStandardRawSqlDAOのみ返す。
        return new POQLStandardRawSqlDAO(this);
    }

	/**
	 * POQLMapDAOを生成して返す
	 * @param table_name 対象テーブル名
	 * @return POQLMapDAO
	 */
	public POQLMapDAO createMapDAO(String table_name ) {
		return new POQLMapBaseDAO(this, table_name );
	}
//    /**
//     * 値を格納したBLOB実装を返す
//     * @param value 初期値
//     * @return BLOBインスタンス
//     */
//    public Blob createBlob(String value) {
//        
//        if (value == null) {
//            return null;
//        }
//        return createBlob(value.getBytes());
//    }
//
//    /**
//     * バイト配列を指定してBLOB実装を返す
//     * @param bytes BLOB値
//     * @return BLOBインスタンス
//     */
//    private Blob createBlob(byte[] bytes) { 
//        
//        if (blobClass == null) {
//            return null;
//        }
//        
//        Blob blob = blobClass.newInstance();
//        // TODO Auto-generated method stub
//        return null;
//    }

    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#getExceptionListener()
	 */
    public POQLExceptionListener getExceptionListener() {
        return exceptionListener;
    }
    /* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#setExceptionListener(jp.co.powerbeans.powerql.exceptions.POQLExceptionListener)
	 */
    public void setExceptionListener(POQLExceptionListener exceptionListener) {
        this.exceptionListener = exceptionListener;
    }
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#isAutoCreateTable()
	 */
	public boolean isAutoCreateTable() {
		return autoCreateTable;
	}
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#setAutoCreateTable(boolean)
	 */
	public void setAutoCreateTable(boolean autoCreateTable) {
		this.autoCreateTable = autoCreateTable;
	}
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#initTable(java.lang.Class)
	 */
	public void initTable(Class c) throws POQLException {
		if (!POQLInitBean.class.isAssignableFrom(c)) {
			Log.println("initTable " + c.getClass() + " is not implements " + POQLInitBean.class.getName());
			return;
		}
		POQLInitBean obj;
		try {
			obj = (POQLInitBean) c.newInstance();
		} catch (InstantiationException e) {
			throw new POQLException(e);
		} catch (IllegalAccessException e) {
			throw new POQLException(e);
		}
		Object[] objs = obj.initialData();

		{
			if (obj.initialTruncate()) {
				POQLDAO dao = createDAO(c);
				dao.dropTable(c);
			}
		}
		{
			POQLDAO dao = createDAO(c);
			dao.setSingleCallMethod(false);
			for (int i = 0; i < objs.length; i++) {
				Object dummy_obj = dao.findByPrimaryKey(objs[i]);
				if (dummy_obj == null) {
					int num = dao.create(objs[i]);
					Log.debug(" initTable:insert num " + num);
				}
			}
			dao.commitAndClose();
		}
	}
	
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#findSysTableAll()
	 */
	public TableProperty[] findSysTableAll() {
		
		Connection con = null;
		try {
			POQLTransaction bt = getPowerQLTransaction();
			con = bt.getConnection();
			DBDependQLStatement dependst = DBDependQLStatementFactory.create(con, this);
			String sql = dependst.getSysSelectTableAllSql();
			
			POQLSimpleViewStatement st = bt.createSimpleViewStatement(TableProperty.class);
			return (TableProperty[]) st.select(sql).toArray(new TableProperty[]{});
			
		} catch (Exception e) {
			onException(e);
			return new TableProperty[]{};
		} finally {
			closeSafe(con);
		}
	}
	

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#findSysTableBySchema(java.lang.String)
	 */
	public TableProperty[] findSysTableBySchema(String schema) {
		Connection con = null;
    try {
      POQLTransaction bt = getPowerQLTransaction();
      con = bt.getConnection();
      java.sql.DatabaseMetaData metadata = con.getMetaData();
      // String[] usertables = {
      // "TABLE", "GLOBAL TEMPORARY", "VIEW"
      // };
      ResultSet result = metadata.getTables(null, schema, null, null);
      List result_list = new ArrayList();

      while (result.next()) {
        // "TABLE_TYPE" に "TABLE" が含まれる項目のみ取得
        // Pgsqlでは INDEX,SEQUENCEも取得されてします
        // SYSTEM TABLE は取得する
        String table_type = result.getString("TABLE_TYPE");
        if (table_type == null || table_type.indexOf("TABLE") == -1) {
          continue;
        }
        
        // 各検索結果をMappingに基づいてBeanに変換
        TableProperty tablep = new TableProperty();
        tablep.setTableName(result.getString(3));
        tablep.setSchema(result.getString(2));

        result_list.add(tablep);
      }

      result.close();
//    		
//    		return re
//			
//			
//			DBDependQLStatement dependst = DBDependQLStatementFactory.create(con, this);
//			String sql = dependst.getSysSelectTableBySchemaSql(schema);
//			
//			POQLSimpleViewStatement st = bt.createSimpleViewStatement(TableProperty.class);
//			Collection list = st.select(sql);
//			List newlist = new ArrayList();
//			// SQLのみで絞れない場合があるため再度schemaで絞る
//			for (Iterator it = list.iterator(); it.hasNext();) {
//				TableProperty m = (TableProperty) it.next();
//				System.out.println(m.getSchema());
//				if (schema.equals(m.getSchema())) {
//					newlist.add(m);
//				}
//			}
			return (TableProperty[]) result_list.toArray(new TableProperty[] {});

    } catch (Exception e) {
      onException(e);
      return new TableProperty[] {};
    } finally {
      closeSafe(con);
    }
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#findSysSchemaAll()
	 */
	public SchemaProperty[] findSysSchemaAll()  {
		Connection con = null;
		try {
			POQLTransaction bt = getPowerQLTransaction();
			con = bt.getConnection();
			java.sql.DatabaseMetaData metadata = con.getMetaData();
			ResultSet result = metadata.getSchemas();
	        List result_list = new ArrayList();
    		while (result.next()) {
    			// 各検索結果をMappingに基づいてBeanに変換
    			SchemaProperty m = new SchemaProperty();
    			m.setSchemaName(result.getString(1));
    			
    			result_list.add(m);
    		}
			return (SchemaProperty[]) result_list.toArray(new SchemaProperty[]{});
//			
//			DBDependQLStatement dependst = DBDependQLStatementFactory.create(con, this);
//			String sql = dependst.getSysSelectSchemaAllSql();
//			
//			POQLSimpleViewStatement st = bt.createSimpleViewStatement(SchemaProperty.class);
//			return (SchemaProperty[]) st.select(sql).toArray(new SchemaProperty[]{});
			
		} catch (Exception e) {
			onException(e);
			return new SchemaProperty[]{};
		} finally {
			closeSafe(con);
		}
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#findSysColumnBySchemaTable(java.lang.String, java.lang.String)
	 */
	public ColumnProperty[] findSysColumnBySchemaTable(String schema, String table) {
		Connection con = null;
		try {
			POQLTransaction bt = getPowerQLTransaction();
			con = bt.getConnection();
			java.sql.DatabaseMetaData metadata = con.getMetaData();
			Map pkmap = new HashMap();
	        List result_list = new ArrayList();
	        
			{
				// 1. PK取得
				ResultSet result = metadata.getPrimaryKeys(null, schema, table);
				while(result.next()) {
          
					pkmap.put(result.getString("COLUMN_NAME"), 
							result.getString("PK_NAME"));
				}
				result.close();
			}
 
			{
				// 2. 全カラム取得
				ResultSet result = metadata.getColumns(null, schema, table, null);
		        
	    		while (result.next()) {
	    			// 各検索結果をMappingに基づいてBeanに変換
	    			ColumnProperty cp = new ColumnProperty();
	    			cp.setName(result.getString("COLUMN_NAME"));
	    			cp.setDefaultValue(result.getString("COLUMN_DEF"));
	//    			cp.setForceAutoValue(result.getString("COLUMN_NAME"));
            cp.setNullable("YES".equals(result.getString("IS_NULLABLE")));
            cp.setNotNull(cp.isNullable() ? "" : "NOT NULL");
	    			cp.setSize(result.getInt("COLUMN_SIZE") > 0 ? 
	    					result.getInt("COLUMN_SIZE") : result.getInt("BUFFER_LENGTH"));
            cp.setTypeAsDB(result.getString("TYPE_NAME"));
	    			cp.setType((Class) db2javaTypeMap.get(cp.getTypeAsDB().toUpperCase()));
	    			cp.setName(result.getString("COLUMN_NAME"));
	    			cp.setComment(result.getString("REMARKS"));
	    			
	    			cp.setPrimaryKey(pkmap.containsKey(cp.getName()));
	    			if (cp.isPrimaryKey()) {
	    				cp.setPrimaryKeyName((String) pkmap.get(cp.getName()));
	    			}
	    			
	    			result_list.add(cp);
	    		}
	    		result.close();
			}

			return (ColumnProperty[]) result_list.toArray(new ColumnProperty[]{});
			
		} catch (Exception e) {
			onException(e);
			return new ColumnProperty[]{};
		} finally {
			closeSafe(con);
		}
	}

  
  /* (non-Javadoc)
 * @see jp.co.powerbeans.powerql.IPOQLManager#finsSysIndexBySchemaTable(java.lang.String, java.lang.String)
 */
  public IndexProperty[] finsSysIndexBySchemaTable(String schema, String table) {
    Connection con = null;
    try {
      POQLTransaction bt = getPowerQLTransaction();
      con = bt.getConnection();
      java.sql.DatabaseMetaData metadata = con.getMetaData();
      List result_list = new ArrayList();

      // 1. 全インデックス取得
      ResultSet result = metadata.getIndexInfo(null, schema, table, false, false);

      while (result.next()) {
        // 各検索結果をMappingに基づいてBeanに変換
        IndexProperty p = new IndexProperty();
        p.setSchema(result.getString("TABLE_SCHEM"));
        p.setTableName(result.getString("TABLE_NAME"));
        p.setIndexName(result.getString("INDEX_NAME"));
        p.setCatalog(result.getString("INDEX_QUALIFIER"));
        p.setColumnName(result.getString("COLUMN_NAME"));
        p.setNonUnique(result.getString("NON_UNIQUE"));
        p.setType(result.getShort("TYPE"));

        switch(p.getType()) {
        case DatabaseMetaData.tableIndexClustered: p.setTypeName("Clustered"); break;
        case DatabaseMetaData.tableIndexHashed: p.setTypeName("Hashed"); break;
        case DatabaseMetaData.tableIndexOther: p.setTypeName("Other"); break;
        case DatabaseMetaData.tableIndexStatistic: p.setTypeName("Statistic"); continue; // 追加しない

        default: p.setTypeName("");
        }
        
        result_list.add(p);
      }
      result.close();

      return (IndexProperty[]) result_list.toArray(new IndexProperty[] {});

    } catch (Exception e) {
      onException(e);
      return new IndexProperty[] {};
    } finally {
      closeSafe(con);
    }
  }
  
	/**
   * 安全にコネクションをクローズする
   * @param con 切断
   */
	private void closeSafe(Connection con) {
		try {
			if (con != null && !con.isClosed()) {
				con.close();
			}
		} catch (SQLException e) {
			onException(e);
		}
	}

	/**
   * 例外発生時処理
	 * @param e 発生した冷害
	 */
	private void onException(Throwable e) {
		if (this.exceptionListener != null) {
			this.exceptionListener.onException(e);
		}
		
        if (throwException) {
            // 例外スローフラグが有効ならスローする
            throw new RuntimeException(e);
        }
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#isThrowException()
	 */
	public boolean isThrowException() {
		return throwException;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.IPOQLManager#setThrowException(boolean)
	 */
	public void setThrowException(boolean throwException) {
		this.throwException = throwException;
	}
	
	public void dummy() {
		
	}

}
