package jp.co.powerbeans.powerql;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLExclusiveException;
import jp.co.powerbeans.powerql.exceptions.POQLResultEmptyException;
import jp.co.powerbeans.powerql.exceptions.POQLResultMultiException;
import jp.co.powerbeans.powerql.exceptions.POQLTableNotFoundException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * <p>タイトル: POQLStatement</p>
 * <p>説明:
 * SQL実行のためのStatement保持クラス
 *  </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/07</p>
 * @author 門田明彦
 * @version $Revision: 1.4 $
 */
public class POQLStatement extends POQLStatementSupport {

	/** Statement */
	private Statement statement;
    /** 最終実行 SQL  */
    private String lastSQL;
    /** 最終実行時 Bind Values  */
    private Object[] lastBindValues;

	/**
	 * コンストラクタ
	 * @param bean_class Beanクラス
	 * @param statement 生成済みStatement
	 * @param super_st POQLTransaction
	 */
	POQLStatement(
		Class bean_class,
		Statement statement,
		POQLTransaction super_tr)
		throws POQLException, POQLTableNotFoundException {
		bean = bean_class;
		this.statement = statement;
		this.super_tr = super_tr;
        this.lastSQL = "";

		// テーブル名作成
		tableName = POQLUtil.className2TableName(bean);

		// 渡されたBeanのSQLマッピングを作成
		try {
			createSQLMapping();
		} catch (SQLException e) {
			throw new POQLException(e);
		}
    // rtrim フラグ設定
    setRtrim(super_tr.isRtrim());
	}
	
	/**
	 * コンストラクタ
	 * @param bean_class Beanクラス
	 * @param statement 生成済みStatement
	 * @param super_st POQLTransaction
	 */
	POQLStatement(
		String table_name,
		Statement statement,
		POQLTransaction super_tr)
		throws POQLException, POQLTableNotFoundException {
//		bean = bean_class;
		this.statement = statement;
		this.super_tr = super_tr;
        this.lastSQL = "";

		// テーブル名作成
		tableName = table_name;

//		// 渡されたBeanのSQLマッピングを作成
//		try {
//			createSQLMapping();
//		} catch (SQLException e) {
//			throw new POQLException(e);
//		}
    // rtrim フラグ設定
    setRtrim(super_tr.isRtrim());
	}

	/**
	 * selectAll<BR>
	 * 全件検索し結果をBeanのListで返す
     * @return 検索結果
	 */
	public Collection selectAll() throws POQLException, SQLException {
	    return selectAll("");
	}
	
    /**
	 * 全件検索し結果をBeanのListで返す
     * @param order_by ソート順
     * @return 検索結果
     * @throws SQLException
     * @throws POQLException
     */
    public Collection selectAll(String order_by) throws SQLException, POQLException {
		StringBuffer sql = new StringBuffer("SELECT * FROM " + getTableName());
		if (order_by != null && order_by.length() > 0) {
		    sql.append(" ORDER BY " + order_by);
		}
		ArrayList result_list = new ArrayList();
		ResultSet result = statement.executeQuery(sql.toString());

		while (result.next()) {
			// 各検索結果をMappingに基づいてBeanに変換
			result_list.add(getBeanByResultSet(result));
		}

		result.close();
		return result_list;
    }


	/**
	 * select<BR>
	 * 検索条件を与えて検索した結果をBeanのListで返す
	 * @param where WHERE 句
	 * @return 検索結果
	 */
	public Collection select(String where)
		throws SQLException, POQLException {

		return select(where, "", "");
	}

	/**
	 * select<BR>
	 * 検索条件を与えて検索した結果をBeanのListで返す
	 * @param where WHERE 句
	 * @param groupby GROUP BY 句
	 * @return 検索結果
	 */
	public Collection select(String where, String groupby)
		throws SQLException, POQLException {

		return select(where, groupby, "");
	}

	/**
	 * select<BR>
	 * 検索条件を与えて検索した結果をBeanのListで返す
	 * @param where WHERE 句
	 * @param groupby GROUP BY 句
	 * @param orderby ORDER BY 句
	 * @return 検索結果
	 */
	public Collection select(String where, String groupby, String orderby)
		throws SQLException, POQLException {

      // 同一SQLではエラーとなるため各ベンダー毎にSQLを作成して取得
      DBDependQLStatement dqs =
        DBDependQLStatementFactory.create(statement.getConnection(), super_tr.mgr);
        
        // SQL 生成
	    String sql = dqs.getSelectSQL(getTableName(), bpList, where, groupby, orderby);
        
        return selectBySQL(sql);
	}

	/**
	 * insert<BR>
	 * １レコードをオブジェクトからINSERT。<BR>
	 * BLOB型は対応していないのでBLOB型に対応するには
	 *  POQLManager#createPOQLPreparedStatement_forInsert で
	 * 生成した POQLPreparedStatement を利用してINSERTすること。
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @return 更新数
	 */
	public int insert(Object o) throws POQLException, SQLException {
		return insert(o, true);
	}
	/**
	 * insert<BR>
	 * １レコードをオブジェクトからINSERT。<BR>
	 * BLOB型は対応していないのでBLOB型に対応するには
	 *  POQLManager#createPOQLPreparedStatement_forInsert で
	 * 生成した POQLPreparedStatement を利用してINSERTすること。
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @param autonum true:オートナンバーカラムをSQLで追加しない, false:オートナンバーカラムでもoの値を格納
	 * @return 更新数
	 */
	public int insert(Object o, boolean autonum) throws POQLException, SQLException {
    
	    validateBean(o);
	    
	    // DBMS依存処理クラス取得
//	    DBDependQLStatement dep = DBDependQLStatementFactory.create(
//	            super_tr.getConnection(), super_tr.mgr);

//		StringBuffer sql =
//		    new StringBuffer("INSERT INTO " + getTableName() + " (");
//
//		// フィールド指定
//		int c = 0;
//		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
//			// Property
//			BeanProperty bp = (BeanProperty) bpit.next();
//			
//			// DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
//			if (autonum && bp.getColumnProp().isNotUseInsertSql()) {
//			    continue;
//			}
//			// SQLに追加
//			if (c++ > 0) {
//				sql.append(",");
//			}
//			sql.append(bp.getColumnProp().getName());
//		}
//		
//		sql.append(") VALUES (");
//
//		// 各Property名を利用して値を取得し PreparedStatement#setXxx で格納
//		c = 0;
//		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
//			// Property
//			BeanProperty bp = (BeanProperty) bpit.next();
//			// カラム値取得
//			Object val = getValueByType(o, bp);
//			
//			// DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
//			if (autonum && bp.getColumnProp().isNotUseInsertSql()) {
//			    continue;
//			}
//			
//			// SQLに追加
//			if (c > 0) {
//				sql.append(",");
//			}
//			// 値を格納
////			sql.append(formatVal(val, bp.getColumnProp().getTypeAsDB(), dep));
//			sql.append("?");
//
//			c++;
//		}
//		sql.append(")");
//
//		if (c == 0) {
//			throw new POQLException("don't get Column");
//		}
		
		
		POQLPreparedStatement prepared_statement = this.super_tr.createPreparedStatement_forInsert(this.bean);
		return prepared_statement.insert(o);
//		return statement.executeUpdate(sql.toString());
	}
	
	/**
	 * delete<BR>
	 * 条件を指定せずにプライマリキーが一致するレコードを削除
	 * @param o 削除するBeanクラスインスタンス
	 * @return 削除件数
	 */
	public int delete(Object o) throws POQLException, SQLException {
    
    validateBean(o);
    
		StringBuffer sql =
			new StringBuffer(
				"DELETE FROM " + getTableName() + " WHERE ");

		String where = null;
		
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (bp.getColumnProp().isPrimaryKey()) {
				// Primary Key で更新
				Object val = getValueByType(o, bp);
				where = bp.getName() + " = " + formatVal(val);
				break;
			}
		}
		if (where == null) {
			throw new POQLException("cannot get Primary Key Column of :" + this.tableName);
		}
		sql.append(where);

		return statement.executeUpdate(sql.toString());
	}

	/**
	 * delete<BR>
	 * 条件を指定してレコードを削除
	 * @param where WHERE 句
	 */
	public int delete(String where) throws POQLException, SQLException {

		StringBuffer sql =
			new StringBuffer(
				"DELETE FROM " + getTableName());

		if (where != null && where.trim().length() > 0) {
		    sql.append(" WHERE " + where);
		}

		return statement.executeUpdate(sql.toString());
	}

	/**
	 * update<BR>
	 * 条件を指定せずにUPDATE
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @return 更新数
	 */
	public int update(Object o) throws POQLException, SQLException {

		return update(o, "");
	}

	/**
	 * update<BR>
	 * 条件を指定してUPDATE<BR>
	 * プライマリキー以外の全カラムを更新
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @param where プライマリキー条件に追加する WHERE 条件
	 * @return 更新数
	 */
	public int update(Object o, String where)
		throws POQLException, SQLException {

    validateBean(o);

		StringBuffer sql =
			new StringBuffer("UPDATE " + getTableName() + " SET ");

		// 各Property名を利用して値を取得
		int c = 0;
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (bp.getColumnProp().isPrimaryKey()) {
				// Primary Key は更新しない
				continue;
			}
			// カラム値取得
			Object val = getValueByType(o, bp);
			// SQLに追加
			if (c > 0) {
				sql.append(",");
			}
			sql.append(bp.getName() + "=" + formatVal(val));

			c++;
		}

		if (c == 0) {
			throw new POQLException("don't get Column");
		}

		// Primary Key WHERE 句を作成
		sql.append(" WHERE ");
		int pc = 0;
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (!bp.getColumnProp().isPrimaryKey()) {
				continue;
			}
			if (pc > 0) {
				sql.append(",");
			}
			Object val = getValueByType(o, bp);
			sql.append(bp.getName() + " = " + formatVal(val));
		}
		
		if (where != null && where.trim().length() > 0) {
			// 追加WHERE句
			sql.append(" AND " + where.trim());
		}

		return statement.executeUpdate(sql.toString());
	}

	/**
	 * getNextSequence<BR>
	 * このテーブルの次のシーケンスを取得<BR>
	 * <B>TODO 将来は Oracle Sequence, MySQL AUTONUMBER に対応させる</B>
	 * @return 新しいシーケンス番号
	 */
	public int getNextSequence() throws POQLException {

        String sequence_table = super_tr.mgr.getSequenceTable();
        if (sequence_table == null) {
            new POQLException("sequence_table is null.");
        }
		try {
			// クラス（テーブル）毎に同期
			synchronized (bean.getClass()) {

				// シーケンスレコード存在チェック
                StringBuffer sql_select =
					new StringBuffer(
						" SELECT sequence "
							+ " FROM "
							+ sequence_table
							+ " WHERE tableName = '"
							+ getTableName()
							+ "'");
                
                ArrayList cp_list = CPCache.getColumnPropertyList(super_tr.getConnection(), super_tr.mgr, getTableName());
                ColumnProperty cp = null;

                // 最初に出現したプライマリキーのカラムをシーケンスフィールドとして扱う
                for(Iterator it = cp_list.iterator(); it.hasNext();) {
                    
                    ColumnProperty cpw = (ColumnProperty)it.next();
                    if (cpw.isPrimaryKey()) {
                       cp = cpw;
                       break;
                    }
                }
                
                if (cp == null) {
                    throw new POQLException("finding seq PK. but PK not found");
                }
				
//				// 先頭のカラムをシーケンスとしてきめうちで取得
//				ColumnProperty cp = (ColumnProperty)CPCache.getColumnPropertyList(
//				        super_tr.getConnection(),
//				        super_tr.mgr,
//				        getTableName()).get(0);
							
//				ColumnProperty cp = (ColumnProperty)getColumnPropertyList().get(0);// 先頭がシーケンスフィールド固定
				
				// 対象テーブルから最大シーケンス取得SQL
				StringBuffer sql_select_max = 
					new StringBuffer(
						" SELECT MAX(" + cp.getName() + ")" + 
						" FROM " + getTableName());

				int check_c = 0; // シーケンステーブル数
				int now_seq = 0; // インサートするシーケンス
				int table_max_seq = 0; // テーブルシーケンスフィールド最大値

				// シーケンステーブルからシーケンス取得
				{
					ResultSet result_c =
						statement.executeQuery(sql_select.toString());
					if (result_c.next()) {
						check_c++;
						now_seq = result_c.getInt(1);
					}
					result_c.close();
				}

				// 既存テーブルから最大シーケンスを取得
				{
					ResultSet result =
						statement.executeQuery(sql_select_max.toString());

					if (result.next()) {
						table_max_seq = result.getInt(1);
					}

					result.close();
				}

				if (check_c == 0) {
					// シーケンスを新規作成
					now_seq = table_max_seq;
					if (check_c == 0) {
						// シーケンスレコードをINSERT
						StringBuffer sql_insert = 
							new StringBuffer(
							"INSERT INTO " + sequence_table + 
							" VALUES('" + getTableName() + "'," + now_seq + ")");
						
						statement.executeUpdate(sql_insert.toString());
					}
										
				} else {
					// シーケンステーブルと既存テーブルの最大値を比較して大きい値を利用					
					
					now_seq = now_seq > table_max_seq ? now_seq : table_max_seq;
				}

				// シーケンスインクリメント(UPDATE)
				boolean is_auto_commit =
					super_tr.getConnection().getAutoCommit();
				super_tr.setAutoCommit(false);
				
				StringBuffer sql_update =
					new StringBuffer(
						" UPDATE "
							+ sequence_table
							+ " SET sequence = " + (now_seq + 1)
//				+ " SET sequence = sequence + 1 "
							+ " WHERE tableName = '"
							+ getTableName()
							+ "'");

				int c = statement.executeUpdate(sql_update.toString());
				if (c == 0) {
					throw new POQLException("cannot update sequence");
				}

				// シーケンス取得
				ResultSet result =
					statement.executeQuery(sql_select.toString());

				int sequence_val = 0;
				if (result.next()) {
					sequence_val = result.getInt(1);
				}

				result.close();
				super_tr.commit();
				super_tr.setAutoCommit(is_auto_commit);
				return sequence_val;
			}
		} catch (SQLException e) {
			super_tr.rollback();
			e.printStackTrace();
			throw new POQLException(e);
		} finally {
			
		}

	}

	/**
	 * selectOne<BR>
	 * 単一オブジェクトだけ取得するものとしてSELECT
	 * @param where where 句SQL
	 * @return 検索結果。Beanクラスインスタンス。
	 */
	public Object selectOne(String where) throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException {

		return selectOne(where, "");
	}

	/**
	 * selectOne<BR>
	 * 単一オブジェクトだけ取得するものとしてSELECT
	 * @param where where 句SQL
	 * @param order_by order by句SQL
	 * @return 検索結果。Beanクラスインスタンス。
	 */
	public Object selectOne(String where, String order_by) throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException {
		Collection col = select(where, order_by);
		if (col.isEmpty()) {
			throw new POQLResultEmptyException("検索結果 0件");
		} else if (col.size() > 1) {
			throw new POQLResultMultiException("検索結果 " + col.size() + "件");
		}
		
		Object o = null;
		for(Iterator it = col.iterator(); it.hasNext();) {
			o = it.next(); 
		}
		return o;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.POQLStatementIF#close()
	 */
	public void close() throws POQLException {
		if (statement != null) {
			try {
				statement.close();
			} catch (SQLException e) {
				throw new POQLException(e);
			}
		}
	}

	/**
	 * StatementがNULL以外の場合はclose, NULLの場合は何もしない
	 * @param st Statement
	 */
	public static void closeSafe(POQLStatementIF st) {
		try {
			if (st != null) {
				st.close();
			}
		} catch (POQLException e) {
			e.printStackTrace();
		}
	}
	
    /**
     * 設定ファイルの フィールド名で排他チェックを行い、レコード取得時と同じ値なら更新する。<BR>
     * レコード取得時と異なる場合は排他エラーとし、POQLExclusiveException をスローする。
     * 
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @param where プライマリキー条件に追加する WHERE 条件
     * @return 更新数
     * @throws POQLExclusiveException
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     * @throws POQLException
     * @throws SQLException
     */
    public int updateExclusive(Object o, String where) throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException, POQLExclusiveException {
        return updateExclusive(o, where, "");
    }

    /**
     * exclusive_check_field フィールドで排他チェックを行い、レコード取得時と同じ値なら更新する。<BR>
     * レコード取得時と異なる場合は排他エラーとし、POQLExclusiveException をスローする。
     * 
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @param where プライマリキー条件に追加する WHERE 条件
     * @param exclusive_check_field 排他チェックフィールド名。null or "" の場合は排他チェックを行わない
     * @return 更新数
     * @throws POQLException
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     * @throws POQLExclusiveException
     * @throws SQLException
     */
    public int updateExclusive(Object o, String where, String exclusive_check_field) 
    throws SQLException, POQLException, 
    	POQLResultEmptyException, POQLResultMultiException, POQLExclusiveException {
        validateBean(o);
        
        // 0. WHERE句作成
		int pc = 0;
		StringBuffer where_sql = new StringBuffer();
		
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (!bp.getColumnProp().isPrimaryKey()) {
				continue;
			}
			if (pc > 0) {
			    where_sql.append(",");
			}
			Object val = getValueByType(o, bp);
			where_sql.append(bp.getName() + " = " + formatVal(val));
		}
		
		if (where != null && where.trim().length() > 0) {
			// 追加WHERE句
		    where_sql.append(" AND " + where.trim());
		}

        
        // 1. SELECT FOR UPDATE 排他チェック
        if (exclusive_check_field != null && exclusive_check_field.length() > 0) {
            // 排他チェックフィールドが有効な場合だけロックする
            Object db_bean = selectOne(where_sql.toString() + " FOR UPDATE");
            if (equalFieldVal(db_bean, o, exclusive_check_field)) {
                // 値が一致したので更新する
            } else {
                // 既に更新されていたので排他エラーとする
                throw new POQLExclusiveException(getTableName() + "." + exclusive_check_field + " is not match in bean.");
            }
        }
        
        // 2. UPDATE
		StringBuffer sql =
			new StringBuffer("UPDATE " + getTableName() + " SET ");

		// 各Property名を利用して値を取得
		int c = 0;
	    DBDependQLStatement dep = DBDependQLStatementFactory.create(super_tr.getConnection(),
	            super_tr.mgr);
	    
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (bp.getColumnProp().isPrimaryKey()) {
				// Primary Key は更新しない
				continue;
			}
			// カラム値取得
			Object val = getValueByType(o, bp);
			// SQLに追加
			if (c > 0) {
				sql.append(",");
			}
			if (bp.columnProp.getName().equalsIgnoreCase(exclusive_check_field)) {
			    // 更新フィールドの場合現在日付を格納(DBサーバー日付)
			    sql.append(bp.getName() + "=" + dep.getSQL_NOW());
			    
			} else {
			    // 通常
				sql.append(bp.getName() + "=" + formatVal(val, bp.columnProp.getTypeAsDB(), dep));
//				sql.append(bp.getName() + "=" + formatVal(val));
			}

			c++;
		}

		if (c == 0) {
			throw new POQLException("don't get Column");
		}

		// Primary Key WHERE 句を追加
		sql.append(" WHERE " + where_sql.toString());

		return statement.executeUpdate(sql.toString());
    }

    /**
     * exclusive_check_field フィールドで排他チェックを行い、レコード取得時と同じ値なら
     * プライマリキーの値を元に物理削除する。<BR>
     * レコード取得時と異なる場合は排他エラーとし、POQLExclusiveException をスローする。
     * 
	 * @param o 追加するObject(テーブルに対応しているBean)
	 * @param where プライマリキー条件に追加する WHERE 条件
     * @param exclusive_check_field 排他チェックフィールド名。null or "" の場合は排他チェックを行わない
     * @return 削除件数
     * @throws POQLExclusiveException
     * @throws POQLException
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     * @throws SQLException
     */
    public int deleteExclusive(Object o, String where, String exclusive_check_field)
    	throws POQLExclusiveException, SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException { 
        validateBean(o);
        
        // 0. WHERE句作成
		int pc = 0;
		StringBuffer where_sql = new StringBuffer();

        BeanProperty[] pk_bps = getPrimaryKeyBeanProperty();

        for(int i = 0; i < pk_bps.length; i++) {
            // プライマリキーは複数存在するものとして扱う
            if (i > 0) {
                where_sql.append(" AND ");
            }
            where_sql.append(
                    pk_bps[i].getColumnProp().getName() + " = "
                    + POQLPreparedStatement.formatVal(
                            getValueByType(o, pk_bps[i])));
        }
		
		if (where != null && where.trim().length() > 0) {
			// 追加WHERE句
		    where_sql.append(" AND " + where.trim());
		}

        // 1. SELECT FOR UPDATE 排他チェック
        if (exclusive_check_field != null && exclusive_check_field.length() > 0) {
            // 排他チェックフィールドが有効な場合だけロックする
            Object db_bean = selectOne(where_sql.toString() + " FOR UPDATE");
            if (equalFieldVal(db_bean, o, exclusive_check_field)) {
                // 値が一致したので更新する
            } else {
                // 既に更新されていたので排他エラーとする
                throw new POQLExclusiveException(getTableName() + "." + exclusive_check_field + " is not match in bean.");
            }
        }
        
        // 2. DELETE
		StringBuffer sql =
			new StringBuffer("DELETE FROM " + getTableName() + " ");

		// Primary Key WHERE 句を追加
		sql.append(" WHERE " + where_sql.toString());

		return statement.executeUpdate(sql.toString());
    }

    /**
     * 全レコード数を取得
     * @return 一致レコード数
     * @throws SQLException
     */
    public int count() throws SQLException {
        return count(null);
    }
    
    /**
     * where 条件を指定してレコード数を取得
     * @param where 検索条件
     * @return 一致レコード数
     * @throws SQLException
     */
    public int count(String where) throws SQLException { 
    
	    StringBuffer sql = new StringBuffer("SELECT COUNT(*) AS NUM FROM " + getTableName());
	    if (where != null && where.trim().length() > 0) {
	        sql.append(" WHERE " + where);
	    }
	    this.lastSQL = sql.toString();
        
		ResultSet result = statement.executeQuery(sql.toString());
		int num = 0;
		if (result.next()) {
		    num = result.getInt("NUM");
		}
		result.close();
		return num;
    }

    /**
     * @return lastSQL を戻します。
     */
    public String getLastSQL() {
        return lastSQL;
    }
    /**
     * @return lastBindValues を戻します。
     */
    public Object[] getLastBindValues() {
        return lastBindValues;
    }

    /**
     * SQLを指定して検索
     * @param sql SQL
     * @return 検索結果
     * @throws SQLException
     * @throws POQLException
     */
    public Collection selectBySQL(String sql) throws SQLException, POQLException {
        // 最終実行SQLとして格納
        this.lastSQL = sql;

        // 実行
        ArrayList result_list = new ArrayList();
        ResultSet result = statement.executeQuery(sql);

        while (result.next()) {
            // 各検索結果をMappingに基づいてBeanに変換
            result_list.add(getBeanByResultSet(result));
        }

        result.close();
        return result_list;
    }

}
