package jp.co.powerbeans.powerql.dao;

import java.util.Collection;

import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;

/**
 * <p>タイトル: POQLRawSqlDAO</p>
 * <p>説明: SQLを直接指定して検索する汎用DAOメソッドを提供するインターフェース。<BR>
 * 
 * POQLRawSqlDAOを利用するビジネスロジック側では必要な場合だけ例外をcatchし、
 * 例外の種類に応じた処理を実装する。<BR>
 * <BR>
 * <pre>
 * <b>POQLRawSqlDAOを生成するまでに必要な前準備</b>
 * <ol>
 * <li>DBMSにテーブルを作成（以下の例ではTrnWork)
 * <li>作成したテーブルと同じ名称のJavaBeanクラスを作成。(パッケージはどこでもOK,大文字小文字は一致しなくても構いません）
 * <li>JDBCドライバをクラスパスに追加
 * <li>サーブレット、あるいは各ビジネスロジッククラスの継承元（基底）クラスの
 *     static {} 内などの初期化処理部分で POQLManager を生成
 * <li>ビジネスロジック内でPOQLManagerからDAOを生成しコミット
 * </ol>
 * <hr>
 * 階層1) WEBアプリケーション
 * ブラウザ<->WEB層<->コントローラ<->ビジネスロジック   <->   POQLDAO <-> DB
 *                                  ┗同一トランザクションで処理可能┛
 * 
 * 階層2) GUIアプリケーション
 * UI層<->イベントディスパッチャ<->ビジネスロジック  <->  POQLDAO <-> DB
 *                               ┗同一トランザクションで処理可能┛
 * <hr>
 * 
 * 例1）   // 排他チェックを行い、1メソッドでコミット、またはロールバックし、例外をスローする
 *         POQLManager bm = new POQLManager("org.postgresql.Driver",
 *               "jdbc:postgresql:bqltest", "trans",
 *               "t79rp90", "POQL_SEQUENCE");
 * 
 *         POQLRawSqlDAO dao = bm.createRawSqlDAO(); // DAO生成
 *         dao.setThrowException(true); // 例外をスローする
 * 
 *          Collection col = dao.find("select * from TrnWork", TrnWork.class);
 * 
 *
 * <b>デフォルト値</b>
 *   getExclusiveCheckField() = null
 *   getSingleCallMethod() = true
 *   getThrowException() = false
 * </pre>
 * 　
 * 
 * </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/16</p>
 * @author 門田明彦
 * @version $Revision$
 */

public interface POQLRawSqlDAO {

    /**
     * SQLを指定して複数レコードを取得する。<BR>
     * 検索結果はbean_classで取得する
     * @param sql SQL
     * @param bean_class 検索結果格納クラス
     * @return 検索結果
     */
    public Collection find(String sql, Class bean_class);
    
    /**
     * SQLを指定してレコードを1件取得する。<BR>
     * 検索結果はbean_classで取得する
     * @param sql SQL
     * @param bean_class 検索結果格納クラス
     * @return 検索結果
     */
    public Object find1(String sql, Class bean_class);
    
//    /**
//     * where 条件を指定してレコード数を取得
//     * @param sql 検索条件
//     * @return 一致レコード数
//     */
//    public int countBy(String sql);
//
    /**
     * 例外リスナークラスを返す
     * @return 例外リスナー
     */
    public POQLExceptionListener getExceptionListener();
    
    /**
     * 例外リスナークラスを設定する。<BR>
     * Managerのリスナーが設定されている場合は上書きされる。
     * @param exceptionListener 例外リスナー
     */
    public void setExceptionListener(POQLExceptionListener exceptionListener);
    
    /**
     * rtrimフラグを返す
     * @return rtrim を戻します。
     */
    public boolean isRtrim();

    /**
     * rtrimフラグを設定する
     * @param rtrim rtrimフラグ
     */
    public void setRtrim(boolean rtrim);
    
    /**
     * 最後に実行したSQLを出力する。
     * SQL内の?は変換しない
     * @return sql
     */
    public String getLastSQL();

    /**
     * 最後に実行したSQLのバインド値を出力する。
     * @return バインド値
     */
    public Object[] getLastBindValues();
    
    /**
     * JTA等の外部トランザクション管理モードフラグ, true JTA利用, false JTA非利用(default)
     * @return useJTATransaction を戻します。
     */
    public boolean isUseJTATransaction();
    
    /**
     * JTA等の外部トランザクション管理モードフラグを設定<BR>
     * true JTA利用, false JTA非利用(default)
     * @param useJTATransaction useJTATransaction を設定。
     */
    public void setUseJTATransaction(boolean useJTATransaction);
    
    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかをDAO個別に指定
     * @param throw_exception
     */
    public void setThrowException(boolean throw_exception);

    /**
     * エラー発生時にRuntimeExceptionをスローするかどうかのフラグを取得
     * @return throwException
     */
    public boolean isThrowException();
    
    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうかのフラグを取得する。<BR>
     * @return returnNull true nullで返す。false 空のBeanインスタンスで返す。
     */
    public boolean isReturnNull();

    /**
     * 戻り値が Object 型のfindメソッドで検索結果が0件の場合
     * null オブジェクトを返すかどうか設定する。<BR>
     * デフォルトはtrueでnullを返す。<BR>
     * false を設定するとBeanクラスを生成して値を何も設定せずに返す。
     * 
     * @param returnNull returnNull を設定します。true nullで返す。false 空のBeanインスタンスで返す。
     */
    public void setReturnNull(boolean returnNull);
}