package jp.co.powerbeans.powerql.dao;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import jp.co.powerbeans.powerql.BeanProperty;
import jp.co.powerbeans.powerql.CPCache;
import jp.co.powerbeans.powerql.POQLManager;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.POQLStatement;
import jp.co.powerbeans.powerql.POQLTransaction;
import jp.co.powerbeans.powerql.POQLUtil;
import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLExceptionListener;
import jp.co.powerbeans.powerql.exceptions.POQLPKNotFoundException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * POQLDAO の基本実装クラス。
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * @author A.Monden
 */
public class POQLStandardDAO extends POQLBaseDAO {

    /**
     * コンストラクタ
     * @param manager
     * @param bean_class Beanクラス
     */
    public POQLStandardDAO(POQLManager manager, Class bean_class) {
        super(manager,bean_class);
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findByPrimaryKey(int)
     */
    public Object findByPrimaryKey(int id) {
        return findByPrimaryKey(new Integer(id));
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findByPrimaryKey(java.lang.Object)
     */
    public Object findByPrimaryKey(Object obj)  {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        Object bean = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            BeanProperty[] bps = st.getPrimaryKeyBeanProperty();
            
            // プライマリキーフィールドの分だけ条件を追加
            if (bps.length == 1) {
                // PKが１つの場合。ほとんどがこの場合だと思われるため
                // == 1 の場合だけ別に記述
                Object val = POQLPreparedStatement.formatVal(obj);
                String col_name = bps[0].getColumnProp().getName();
                if (obj == null) {
                    bean = st.selectOne(col_name + " IS NULL");
                } else {
                	bean = st.selectOne(col_name + " = " + val);
                }
                
            } else if (bps.length > 0){
                // PKが複数の場合
	            if (!(obj instanceof Object[])) {
	                throw new IllegalArgumentException(getBeanClass() + " pk is " + bps.length +
	                        " items. bud parameter is 1 item.");
	            }
              int len = Array.getLength(obj);
//	            Object[] objs = (Object[])obj;
	            if (len != bps.length) {
	                throw new IllegalArgumentException(getBeanClass() + " pk is " + bps.length +
                    " items. bud parameter is " + len + " item.");
	            }
	            
                StringBuffer buf = new StringBuffer();
	            // PKフィールド数が一致したのでSQL作成
	            for(int i = 0; i < len; i++) {
	                if (i > 0) {
	                    buf.append(" AND ");
	                }
        					String col_name = bps[i].getColumnProp().getName();
        					if (Array.get(obj,i) == null) {
        						buf.append(col_name + " IS NULL");
        					} else {
        						buf.append(col_name + " = " + POQLPreparedStatement.formatVal(Array.get(obj,i)));
        					}
	            }
	            
                bean = st.selectOne(buf.toString());
            } else {
                // bps.length == 0. PKを取得できていない.
                throw new POQLPKNotFoundException();
            }

        } catch (Exception e) {
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        if (bean == null) {
            bean = createSafeBean();
        }
        
        return bean;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findByAll()
     */
    public Collection findByAll() {
        return findByAll("");
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findByAll()
     */
    public Collection findByAll(String order_by) {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        Collection col = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            col = st.selectAll(order_by);
            
        } catch (Exception e) {
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        if (col == null) {
            col = new ArrayList();
        }
        return col;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#find1By(java.lang.String)
     */
    public Object find1By(String where) {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        Object bean = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            bean = st.selectOne(where);

        } catch (Exception e) {
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        if (bean == null) {
            bean = createSafeBean();
        }
        return bean;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findBy(java.lang.String)
     */
    public Collection findBy(String where) {
        
        return findBy(where, null);
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#findBy(java.lang.String)
     */
    public Collection findBy(String where, String order_by) {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        Collection col = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            col = st.select(where, null, order_by);

        } catch (Exception e) {
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        if (col == null) {
            col = new ArrayList();
        }
        return col;
    }


    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#updateByPrimaryKey(java.lang.Object)
     */
    public int update(Object bean) {
        
        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += update(objs[i]);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += update(it.next());
            }
            return c;
        }        
        
        POQLTransaction bqlTrn = null;
        POQLPreparedStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createPreparedStatement_forUpdate(getBeanClass());
            
            // FOR UPDATE でエラーとならないように autoCommit を OFF にする
            bqlTrn.setAutoCommit(false);
            
            num = st.updateExclusive(bean, getExclusiveCheckField());
            
//            POQLStatement st = bqlTrn.createStatement(getBeanClass());
//            num = st.updateExclusive(bean, "", getExclusiveCheckField());
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
   }
//
//    /* (Javadoc なし)
//     * @see jp.co.powerbeans.powerql.dao.POQLDAO#updatesByPrimaryKey(java.util.Collection)
//     */
//    public int updatesByPrimaryKey(Collection beans) {
//        POQLTransaction bqlTrn = null;
//        int num = 0;
//        try {
//            bqlTrn = getPowerQLTransaction();
//            POQLStatement st = bqlTrn.createStatement(getBeanClass());
//            for(Iterator it = beans.iterator(); it.hasNext();) {
//                num += st.updateExclusive(it.next(), "", getExclusiveCheckField());
//            }
//            commit();
//            
//        } catch (Exception e) {
//            rollback();
//            onException(e);
//        } finally {
//            close();
//        }
//        
//        return num;
//    }


    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#create(java.lang.Object)
     */
    public int create(Object bean) {

        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += create(objs[i]);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += create(it.next());
            }
            return c;
        }        
        
        POQLTransaction bqlTrn = null;
        POQLPreparedStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
//            POQLStatement st = bqlTrn.createStatement(getBeanClass());
            st = bqlTrn.createPreparedStatement_forInsert(getBeanClass());
            num = st.insert(bean);
            
            // 追加したレコードのIDを取得しbeanに格納
            setAutoGenerateKeyValue(bean, bqlTrn, st);

            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

	/**
	 * @param bean
	 * @param bqlTrn
	 * @param st
	 * @throws POQLException
	 * @throws SQLException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	private void setAutoGenerateKeyValue(Object bean, POQLTransaction bqlTrn,
			POQLPreparedStatement st) throws POQLException, SQLException,
			IllegalAccessException, InvocationTargetException {
		Object key = getGeneratedKey(bqlTrn, st);
		BeanProperty[] pkbps = st.getPrimaryKeyBeanProperty();
		if (pkbps.length > 0) {
		    BeanProperty bp = pkbps[0]; // 現在は1カラムのみ対応
		    bp.getM_set().invoke(bean, new Object[]{key});
//	                if (bp.getType() == int.class ||
//	                    bp.getType() == Integer.class) {
//	                    // 数値型ならシーケンス値を格納
//	                    bp.getM_set().invoke(bean, new Object[]{new Integer(key.st.getNextSequence())});
//	                    
//	                } else if (bp.getType() == long.class ||
//	                        bp.getType() == Long.class) {
//	                    // 数値型ならシーケンス値を格納
//	                    bp.getM_set().invoke(bean, new Object[]{new Long(st.getNextSequence())});
//	                }
		}
	}

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#createByPowerQLSeq(java.lang.Object)
     */
    public int createByPowerQLSeq(Object bean) {

        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += createByPowerQLSeq(objs[i]);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += createByPowerQLSeq(it.next());
            }
            return c;
        }      
        
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            BeanProperty[] pkbps = st.getPrimaryKeyBeanProperty();
            if (pkbps.length > 0) {
                BeanProperty bp = pkbps[0];
                if (bp.getType() == int.class ||
                    bp.getType() == Integer.class) {
                    // 数値型ならシーケンス値を格納
                    bp.getM_set().invoke(bean, new Object[]{new Integer(st.getNextSequence())});
                    
                } else if (bp.getType() == long.class ||
                        bp.getType() == Long.class) {
                    // 数値型ならシーケンス値を格納
                    bp.getM_set().invoke(bean, new Object[]{new Long(st.getNextSequence())});
                }
            }
            
            // 取得したシーケンス値で格納
            num = st.insert(bean, false);
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#removeByAll()
     */
    public int removeByAll() {
        return removeBy("");
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#removeBy(java.lang.String)
     */
    public int removeBy(String where) {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            num = st.delete(where);
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#removeByPrimaryKey(int)
     */
    public int removeByPrimaryKey(int key) {
        return removeByPrimaryKey(new Integer(key));
    }
    
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#removeByPrimaryKey(java.lang.Object)
     */
    public int removeByPrimaryKey(Object key) {
        
        if (key == null) {
            return 0;
        }
        
        if (key.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])key;
            for(int i = 0; i < objs.length; i++) {
                c += removeByPrimaryKey(objs[i]);
            }
            return c;
        } else if (key instanceof Collection) {
            Collection col = (Collection)key;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += removeByPrimaryKey(it.next());
            }
            return c;
        }
        
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            BeanProperty[] pk_bps = st.getPrimaryKeyBeanProperty();
            String where = null;
            
            if (pk_bps.length > 0) {
                // プライマリキーは1個として扱う
                where = pk_bps[0].getColumnProp().getName() + " = "
                        + POQLPreparedStatement.formatVal(key);
            }
            num = st.delete(where);
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#remove(java.lang.Object)
     */
    public int remove(Object bean) {
        
        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += remove(objs[i]);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += remove(it.next());
            }
            return c;
        }
        
        POQLTransaction bqlTrn = null;
        int num = 0;
        POQLStatement st = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            BeanProperty[] pk_bps = st.getPrimaryKeyBeanProperty();
            
            num = st.deleteExclusive(bean, "", getExclusiveCheckField());
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#countByAll()
     */
    public int countByAll() {
        return countBy(null);
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#countBy(java.lang.String)
     */
    public int countBy(String where) {
        POQLTransaction bqlTrn = null;
        int count = 0;
        POQLStatement st = null;
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(getBeanClass());
            count = st.count(where);

        } catch (Exception e) {
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return count;
    }

    /* (非 Javadoc)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#update(java.lang.Object, java.lang.String[])
     */
    public int update(Object bean, String[] columns) {
        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += update(objs[i], columns);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += update(it.next(), columns);
            }
            return c;
        }        
        
        POQLTransaction bqlTrn = null;
        POQLPreparedStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            Set colset = createColSet(columns);
            st = bqlTrn.createPreparedStatement_forUpdateWithColumns(getBeanClass(), colset);
            
            // FOR UPDATE でエラーとならないように autoCommit を OFF にする
            bqlTrn.setAutoCommit(false);
            num = st.updateExclusive(bean, getExclusiveCheckField(), colset);
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    /* (非 Javadoc)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#create(java.lang.Object, java.lang.String[])
     */
    public int create(Object bean, String[] columns) {
        if (bean == null) {
            return 0;
        }
        
        if (bean.getClass().isArray()) {
            int c = 0;
            Object[] objs = (Object[])bean;
            for(int i = 0; i < objs.length; i++) {
                c += create(objs[i], columns);
            }
            return c;
        } else if (bean instanceof Collection) {
            Collection col = (Collection)bean;
            int c = 0;
            for(Iterator it = col.iterator(); it.hasNext();) {
                c += create(it.next(), columns);
            }
            return c;
        }        
        
        POQLTransaction bqlTrn = null;
        POQLPreparedStatement st = null;
        int num = 0;
        try {
            bqlTrn = getPowerQLTransaction();
            Set colset = createColSet(columns);
            st = bqlTrn.createPreparedStatement_forInsert(getBeanClass(), colset);
            num = st.insert(bean, colset);
            // 追加したレコードのIDを取得しbeanに格納
            setAutoGenerateKeyValue(bean, bqlTrn, st);
            commitSafe();
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
        
        return num;
    }

    public void dropTable(Class bean_class) {
        POQLTransaction bqlTrn = null;
        POQLStatement st = null;
        String table = POQLUtil.className2TableName(bean_class);
        try {
            bqlTrn = getPowerQLTransaction();
            st = bqlTrn.createStatement(bean_class);
            st.selectBySQL("DROP TABLE " + table);
            commitSafe();
            
            // キャッシュから削除
            CPCache.remove(table);
            st.removeBeamMap(bean_class);
            
        } catch (Exception e) {
            onRollback();
            onException(e);
        } finally {
            setLastSqlSafe(st);
            closeSafe();
        }
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.dao.CoreDAO#setExceptionListener(jp.co.powerbeans.powerql.exceptions.POQLExceptionListener)
	 */
	@Override
	public void setExceptionListener(POQLExceptionListener exceptionListener) {
		// S2 のための空実装
		super.setExceptionListener(exceptionListener);
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.dao.CoreDAO#setReturnNull(boolean)
	 */
	@Override
	public void setReturnNull(boolean returnNull) {
		// S2 のための空実装
		super.setReturnNull(returnNull);
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.dao.CoreDAO#setRtrim(boolean)
	 */
	@Override
	public void setRtrim(boolean rtrim) {
		// S2 のための空実装
		super.setRtrim(rtrim);
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.dao.CoreDAO#setThrowException(boolean)
	 */
	@Override
	public void setThrowException(boolean throw_exception) {
		// S2 のための空実装
		super.setThrowException(throw_exception);
	}
    
}
