package jp.co.powerbeans.powerql.vendor;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

import jp.co.powerbeans.powerql.ColumnProperty;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.exceptions.POQLException;

import org.apache.oro.text.perl.Perl5Util;


/**
 * <p>タイトル: DBDependQLStatementMySQL</p>
 * <p>説明: DBDependQLStatement の MySQL実装。</p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * 
 * <p>Created on 2003/10/14</p>
 * @author 門田明彦
 * @version $Revision: 1.5 $
 */
public class DBDependQLStatementMySQL extends DBDependQLStatementSupport {

	private static final String DB_TYPE = "MySQL";
	
	/** regexp lib */
	private static Perl5Util perl = new Perl5Util();

	/** type mapping */
	private static Hashtable db2javaTypeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] db2javaMap = { { "TINYINT", Short.class }, {
			"TINYINT\\([0-9]+\\)", Short.class }, {
			"SMALLINT", Integer.class }, {
			"SMALLINT\\([0-9]+\\)", Integer.class }, {
			"MEDIUMINT", Integer.class }, {
			"MEDIUMINT\\([0-9]+\\)", Integer.class }, {
			"INT", Integer.class }, {
			"INT\\([0-9]+\\)", Integer.class }, {
			"INTEGER", Integer.class }, {
			"INTEGER\\([0-9]+\\)", Integer.class }, {
			"BIGINT", Long.class }, {
			"BIGINT\\([0-9]+\\)", Long.class }, {

			"FLOAT", Float.class }, {
			"FLOAT\\([0-9]+,[0-9]+\\)", Float.class }, {
			"DOUBLE", Double.class }, {
			"DOUBLE \\([0-9]+,[0-9]+\\)", Double.class }, {
			    
			"NUMERIC\\([0-9]+,[0-9]+\\)", Double.class }, {
			"DECIMAL\\([0-9]+,[0-9]+\\)", Double.class }, {
			"REAL\\([0-9]+,[0-9]+\\)", Float.class }, {
			"DOUBLE PRECISION\\([0-9]+,[0-9]+\\)", Double.class }, {

			"DATE", Timestamp.class }, {
			"DATETIME", Timestamp.class }, {
			"TIMESTAMP", Timestamp.class }, {
			"TIMESTAMP\\([0-9]+\\)", Timestamp.class }, {
			"TIME", Timestamp.class }, {

			"TINYBLOB", String.class }, {
			"TINYTEXT", String.class }, {
			"BLOB", String.class }, {
			"TEXT", String.class }, {
			"MEDIUMBLOB", String.class }, {
			"MEDIUMTEXT", String.class }, {
			"LONGBLOB", String.class }, {
			"LONGTEXT", String.class }, {

			"CHAR", String.class }, {
			"VARCHAR", String.class }, {
			"CHAR\\([0-9]+\\)", String.class }, {
			"VARCHAR\\([0-9]+\\)", String.class
			
			},{ "YEAR", String.class // Integer も可能
			    
			},{ "ENUM", String.class
			},{ "SET", String.class
			}};

	/** type mapping data key */
	private static Object[][] java2dbMap = { { Byte.class, "TINYINT" }, {
			Short.class, "TINYINT" }, {
			Character.class, "TINYINT" }, {
			Integer.class, "INT" }, {
			Long.class, "BIGINT" }, {
			Double.class, "DOUBLE PRECISION" }, {

			String.class, "VARCHAR" }, {
			Timestamp.class, "DATE" }
	};
	/** type mapping */
	private static Hashtable java2dbTypeMap = new Hashtable();

	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < db2javaMap.length; i++) {
			db2javaTypeMap.put(db2javaMap[i][0], db2javaMap[i][1]);
		}

		// typeMap.put(Java type, MySQL type)
		for (int i = 0; i < java2dbMap.length; i++) {
			java2dbTypeMap.put(java2dbMap[i][0], java2dbMap[i][1]);
		}
	}
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getColumnPropertyList(java.lang.String)
	 */
	public ArrayList getColumnPropertyList(String table, String schema)
		throws SQLException, POQLException {
		ArrayList list = new ArrayList();
		String sql = "SHOW COLUMNS FROM " + table;

		ResultSet result = statement.executeQuery(sql.toString());

		while (result.next()) {
			// 各カラム情報を取得

			ColumnProperty cp = new ColumnProperty();
			cp.setName(result.getString("Field"));
			cp.setTypeAsDB(result.getString("Type"));
			cp.setNullable(result.getString("Null").equals("YES"));
			cp.setDefaultValue(result.getObject("Default"));
			cp.setPrimaryKey(result.getString("Key").equals("PRI"));
			cp.setForceAutoValue(result.getString("Extra").indexOf("auto_increment") != -1);
			cp.setType(getJavaTypeFromDBType(cp.getTypeAsDB()));
			list.add(cp);
		}

		result.close();

		return list;
	}

	/**
	 * getJavaTypeFromDBType<BR>
	 * DB(MySQL)のカラムの型からJAVAの型をClassで取得
	 * @param type_as_DB DB(MySQL)の型
	 * @return JAVAの型
	 */
	private Class getJavaTypeFromDBType(String type_as_DB)
		throws POQLException {

		String upper = type_as_DB.toUpperCase();
		
		if (db2javaTypeMap.containsKey(upper)) {
			// 完全一致
			return (Class) db2javaTypeMap.get(upper);

		} else {
			// 全パターンと正規表現比較
			for (int i = 0; i < db2javaMap.length; i++) {
				if (perl.match("/" + (String) db2javaMap[i][0] + "/", upper)) {
//					if (perl.match((String) db2javaMap[i][0], upper)) {
//					if (perl.match(upper, (String) db2javaMap[i][0])) { 
					return (Class) db2javaMap[i][1];
				}
			}

		}

		// 一致する型無し
		throw new POQLException(
			"not recoganize db TYPE(" + DB_TYPE + ") " + type_as_DB);
	}

	//	/* (non-Javadoc)
	//	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#createTable(java.lang.String, jp.co.powerbeans.powerql.BeanProperty[])
	//	 */
	//	public void createTable(String string, BeanProperty[] properties) throws SQLException {
	//		
	//		StringBuffer sql = new StringBuffer();
	//		
	////		create table a_e_users (
	////			a_e_users_id INTEGER not null,          -- <pk>
	////			address_ken_i_d INTEGER,                -- addressKenID
	////			mail_addr1 VARCHAR(255),                -- mailAddr1
	////			mail_addr2 VARCHAR(255),                -- mailAddr2
	////			mail_addr3 VARCHAR(255),                -- mailAddr3
	////			purpose_i_d INTEGER,                    -- purposeID
	////			seibetsu_i_d TINYINT,                   -- seibetsuID
	////			user_name VARCHAR(255),                 -- user_name
	////			user_pass VARCHAR(255),                 -- user_pass
	////			work_i_d INTEGER,                       -- workID
	////			jdo_version SMALLINT not null,          -- <opt-lock>
	////			constraint pk_a_e_users primary key (a_e_users_id),
	////			unique index idx_a_e_users_user_name (user_name),
	////			index idx_a_e_users_mail_addr1 (mail_addr1),
	////			index idx_a_e_users_mail_addr2 (mail_addr2),
	////			index idx_a_e_users_mail_addr3 (mail_addr3)
	////		) TYPE = InnoDB;
	//		
	//		int c = statement.executeUpdate(sql.toString());
	//
	//		
	//	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#isExistTable(java.lang.String)
	 */
	public boolean isExistTable(String table_name) throws SQLException {

		String sql = "SHOW TABLES";
		ResultSet result = statement.executeQuery(sql.toString());
		boolean is_exist = false;

		while (result.next()) {
			int c = 1;

			String table = result.getString(c++);

			if (table != null
				&& table_name != null
				&& table.toUpperCase().equals(table_name.toUpperCase())) {
				// 存在
				is_exist = true;
				break;
			}
		}

		result.close();
		return is_exist;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#createTable(java.lang.String, jp.co.powerbeans.powerql.ColumnProperty[])
	 */
	public void createTable(String table_name, ColumnProperty[] cp)
		throws POQLException, SQLException {

		if (cp == null || cp.length == 0) {
			throw new POQLException("ColumnProperty value is invalid:" + cp);
		}

		StringBuffer sql = new StringBuffer("CREATE TABLE " + table_name + "(");

		// column
		for (int i = 0; i < cp.length; i++) {
			sql.append(getJava2DBType(cp[i]) + ", ");
		}

		// pk
		sql.append(
			"constraint pk_"
				+ table_name
				+ " primary key ("
				+ cp[0].getName()
				+ ") ");
		sql.append(" ) TYPE = InnoDB");

		statement.executeUpdate(sql.toString());
	}

	/**
	 * getJava2DBType<BR>
	 * ColumnProperty から CREATE SQL 用カラム定義を作成
	 * @param property
	 * @return CREATE SQL 用カラム定義を作成
	 */
	private String getJava2DBType(ColumnProperty cp) throws POQLException {

		StringBuffer sql = new StringBuffer(cp.getName());
		if (!java2dbTypeMap.containsKey(cp.getType())) {
			// JAVAの型が設定されていない
			throw new POQLException(
				"cannot get type:"
					+ cp.getType()
					+ "(java 2 db) ["
					+ DB_TYPE
					+ "]");
		}

		sql.append(" " + (String) java2dbTypeMap.get(cp.getType()));

		if (cp.getSize() > 0) {
			sql.append("(" + cp.getSize() + ")");
		}

		if (!cp.isNullable()) {
			sql.append(" not null");
		}

		if (cp.getDefaultValue() != null
			&& cp.getDefaultValue().toString().length() > 0) {
			sql.append(" DEFAULT " + cp.getDefaultValue());
		}

		sql.append(" ");
		return sql.toString();
	}

  /**
   * getSelectSQL<BR>
   * パラメータの条件からSQLを作成。
   * MySQLはgroup by があっても SELECT * を記述できるので * を利用。
   * @param tableName テーブル名
   * @param bpList BeanPropertyList
   * @param where where句
   * @param groupby groupby句
   * @param orderby orderby句
   * @return select SQL
   */
  public String getSelectSQL(String tableName, ArrayList bpList, String where, String groupby, String orderby) {

    StringBuffer sql = new StringBuffer("SELECT * FROM " + tableName);

    if (where != null && where.trim().length() > 0) {
      sql.append(" WHERE " + where.trim());
    }
    
    if (groupby != null && groupby.trim().length() > 0) {
      sql.append(" GROUP BY " + groupby.trim());
    }
    
    if (orderby != null && orderby.trim().length() > 0) {
      sql.append(" ORDER BY " + orderby.trim());
    }
    return sql.toString();
  }

    /*
     * (Javadoc なし)
     * 
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW()
     */
    public String getSQL_NOW() {
        return " NOW() ";
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW(java.lang.String)
     */
    public String getSQL_NOW(String column_type) {
        return getSQL_NOW();
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#formatDateVal(java.util.Date, java.lang.String)
     */
    public String formatDateVal(Date date, String column_type) {
        return formatDateVal_std(date, column_type);
    }

//    /* (non-Javadoc)
//     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getBlobClass()
//     */
//    public Class getBlobClass() {
//        return super.getBlobClass("com.mysql.jdbc.PoqlBlob");
//    }

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableAllSql()
	 */
	public String getSysSelectTableAllSql() {
		throw new RuntimeException("not support getSysSelectTableAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectSchemaAllSql()
	 */
	public String getSysSelectSchemaAllSql() {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableBySchemaSql(java.lang.String)
	 */
	public String getSysSelectTableBySchemaSql(String schema) {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}

	// 予約語定義
	static Set RESERVED_WORD_SET;
	static {
		String[] words = {"ACCESSIBLE",
				"ADD",
				"ALL",
				"ALTER",
				"ANALYZE",
				"AND",
				"AS",
				"ASC",
				"ASENSITIVE",
				"BEFORE",
				"BETWEEN",
				"BIGINT",
				"BINARY",
				"BLOB",
				"BOTH",
				"BY",
				"CALL",
				"CASCADE",
				"CASE",
				"CHANGE",
				"CHAR",
				"CHARACTER",
				"CHECK",
				"COLLATE",
				"COLUMN",
				"CONDITION",
				"CONSTRAINT",
				"CONTINUE",
				"CONVERT",
				"CREATE",
				"CROSS",
				"CURRENT_DATE",
				"CURRENT_TIME",
				"CURRENT_TIMESTAMP",
				"CURRENT_USER",
				"CURSOR",
				"DATABASE",
				"DATABASES",
				"DAY_HOUR",
				"DAY_MICROSECOND",
				"DAY_MINUTE",
				"DAY_SECOND",
				"DEC",
				"DECIMAL",
				"DECLARE",
				"DEFAULT",
				"DELAYED",
				"DELETE",
				"DESC",
				"DESCRIBE",
				"DETERMINISTIC",
				"DISTINCT",
				"DISTINCTROW",
				"DIV",
				"DOUBLE",
				"DROP",
				"DUAL",
				"EACH",
				"ELSE",
				"ELSEIF",
				"ENCLOSED",
				"ESCAPED",
				"EXISTS",
				"EXIT",
				"EXPLAIN",
				"FALSE",
				"FETCH",
				"FLOAT",
				"FLOAT4",
				"FLOAT8",
				"FOR",
				"FORCE",
				"FOREIGN",
				"FROM",
				"FULLTEXT",
				"GRANT",
				"GROUP",
				"HAVING",
				"HIGH_PRIORITY",
				"HOUR_MICROSECOND",
				"HOUR_MINUTE",
				"HOUR_SECOND",
				"IF",
				"IGNORE",
				"IN",
				"INDEX",
				"INFILE",
				"INNER",
				"INOUT",
				"INSENSITIVE",
				"INSERT",
				"INT",
				"INT1",
				"INT2",
				"INT3",
				"INT4",
				"INT8",
				"INTEGER",
				"INTERVAL",
				"INTO",
				"IS",
				"ITERATE",
				"JOIN",
				"KEY",
				"KEYS",
				"KILL",
				"LEADING",
				"LEAVE",
				"LEFT",
				"LIKE",
				"LIMIT",
				"LINEAR",
				"LINES",
				"LOAD",
				"LOCALTIME",
				"LOCALTIMESTAMP",
				"LOCK",
				"LONG",
				"LONGBLOB",
				"LONGTEXT",
				"LOOP",
				"LOW_PRIORITY",
				"MASTER_SSL_VERIFY_SERVER_CERT",
				"MATCH",
				"MEDIUMBLOB",
				"MEDIUMINT",
				"MEDIUMTEXT",
				"MIDDLEINT",
				"MINUTE_MICROSECOND",
				"MINUTE_SECOND",
				"MOD",
				"MODIFIES",
				"NATURAL",
				"NOT",
				"NO_WRITE_TO_BINLOG",
				"NULL",
				"NUMERIC",
				"ON",
				"OPTIMIZE",
				"OPTION",
				"OPTIONALLY",
				"OR",
				"ORDER",
				"OUT",
				"OUTER",
				"OUTFILE",
				"PRECISION",
				"PRIMARY",
				"PROCEDURE",
				"PURGE",
				"RANGE",
				"READ",
				"READS",
				"READ_ONLY",
				"READ_WRITE",
				"REAL",
				"REFERENCES",
				"REGEXP",
				"RELEASE",
				"RENAME",
				"REPEAT",
				"REPLACE",
				"REQUIRE",
				"RESTRICT",
				"RETURN",
				"REVOKE",
				"RIGHT",
				"RLIKE",
				"SCHEMA",
				"SCHEMAS",
				"SECOND_MICROSECOND",
				"SELECT",
				"SENSITIVE",
				"SEPARATOR",
				"SET",
				"SHOW",
				"SMALLINT",
				"SPATIAL",
				"SPECIFIC",
				"SQL",
				"SQLEXCEPTION",
				"SQLSTATE",
				"SQLWARNING",
				"SQL_BIG_RESULT",
				"SQL_CALC_FOUND_ROWS",
				"SQL_SMALL_RESULT",
				"SSL",
				"STARTING",
				"STRAIGHT_JOIN",
				"TABLE",
				"TERMINATED",
				"THEN",
				"TINYBLOB",
				"TINYINT",
				"TINYTEXT",
				"TO",
				"TRAILING",
				"TRIGGER",
				"TRUE",
				"UNDO",
				"UNION",
				"UNIQUE",
				"UNLOCK",
				"UNSIGNED",
				"UPDATE",
				"USAGE",
				"USE",
				"USING",
				"UTC_DATE",
				"UTC_TIME",
				"UTC_TIMESTAMP",
				"VALUES",
				"VARBINARY",
				"VARCHAR",
				"VARCHARACTER",
				"VARYING",
				"WHEN",
				"WHERE",
				"WHILE",
				"WITH",
				"WRITE",
				"XOR",
				"YEAR_MONTH",
				"ZEROFILL",
				// MySQL5.1
				"ACCESSIBLE",
				"LINEAR",
				"MASTER_SSL_VERIFY_SERVER_CERT",
				"RANGE",
				"READ_ONLY",
				"READ_WRITE"};
		RESERVED_WORD_SET = new HashSet();
		for (int i = 0; i < words.length; i++) {
			RESERVED_WORD_SET.add(words[i]);
		}
	}
    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#escape(java.lang.String)
     */
    public String escape(String sql) {
    	return RESERVED_WORD_SET.contains(sql.toUpperCase()) ?
    		"`" + sql + "`" : sql;
    }

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getGeneratedKey(jp.co.powerbeans.powerql.POQLPreparedStatement)
	 */
	public Object getGeneratedKey(POQLPreparedStatement st) throws SQLException {
		ResultSet key_rset = st.getGeneratedKeys();
		// MySQL は AUTO INCREMENT 値を getInt のみで取得可能
		if (key_rset != null && key_rset.next()) {
			return key_rset.getInt(1);
		}
		return Integer.valueOf(0);
	}
}
