/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.stat.handler;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.IHandler;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.handlers.HandlerUtil;
import org.xml.sax.SAXException;

import benten.cat.glossary.ui.CatGlossaryUiPlugin;
import benten.cat.stat.CatStatPlugin;
import benten.cat.stat.handler.messages.AbstractStatHandlerMessages;
import benten.cat.stat.views.AbstractStatView;
import benten.core.io.ResourceFile;
import benten.core.io.XliffFileFilter;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;

/**
 * 翻訳統計取得ハンドラー基底クラス。
 *
 * <UL>
 * <LI>すべての翻訳統計取得ハンドラーの基底クラスです。
 * <LI>ジョブの実行制御、ビューとの連携機能を提供します。
 * </UL>
 *
 * @param <T> モデルの型
 * @author KASHIHARA Shinji
 */
public abstract class AbstractStatHandler<T> extends AbstractHandler implements IHandler {

	/**
	 * 翻訳統計取得ハンドラー抽象クラスのためのメッセージ。
	 */
	protected static final AbstractStatHandlerMessages fMsg = new AbstractStatHandlerMessages();

	/**
	 * 進捗モニター
	 */
	private IProgressMonitor monitor;

	/**
	 * {@inheritDoc}
	 */
	final public Object execute(final ExecutionEvent event) throws ExecutionException {
		final File[] selectedFiles = getSelectedFiles(event);
		if (selectedFiles.length == 0) {
			final UiStatus s = new UiStatus(IStatus.INFO, fMsg.getMsg001());
			UiPlugin.openDialog(fMsg.getMsg002(), s);
			return null;
		}
		try {
			// 用語集エンジンが取得できることを予め確認。
			// 用語集を必要としない統計情報においても、用語集の存在チェックは実施することとします。
			CatGlossaryUiPlugin.getDefault().getGlossaryEngine();
		} catch (final IllegalArgumentException e) {
			final UiStatus s = new UiStatus(IStatus.ERROR, e.getMessage());
			UiPlugin.openDialog(fMsg.getMsg007(), s);
			return null;
		}
		final Job job = new Job(getJobMessage()) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				AbstractStatHandler.this.monitor = monitor;
				monitor.beginTask(null, IProgressMonitor.UNKNOWN);
				final List<T> models = new LinkedList<T>();

				// 処理実行
				explore(selectedFiles, models);

				if (monitor.isCanceled()) {
					return Status.CANCEL_STATUS;
				}
				monitor.done();
				if (models.size() == 0) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							final UiStatus s = new UiStatus(IStatus.INFO, fMsg.getMsg003());
							UiPlugin.openDialog(fMsg.getMsg004(), s);
						}
					});
					return Status.CANCEL_STATUS;
				}

				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						setViewInput(event, models);
					}
				});
				return Status.OK_STATUS;
			}
		};
		job.schedule();
		return null;
	}

	/**
	 * 選択されたファイル配列の取得。
	 * @param event イベント
	 * @return 選択されたファイル配列。未選択の場合は長さ 0 の配列。
	 */
	private File[] getSelectedFiles(final ExecutionEvent event) {
		final List<File> files = new LinkedList<File>();
		final ISelection sel = HandlerUtil.getCurrentSelection(event);
		if (sel instanceof IStructuredSelection) {
			final IStructuredSelection selection = (IStructuredSelection) sel;
			for (final Object o : selection.toArray()) {
				if (o != null && o instanceof IResource) {
					final IResource resource = (IResource) o;
					files.add(new ResourceFile(resource));
				}
			}
		}
		return files.toArray(new File[files.size()]);
	}

	/**
	 * ジョブ・メッセージの取得。
	 * @return ジョブ・メッセージ
	 */
	protected String getJobMessage() {
		return fMsg.getMsg005();
	}

	/**
	 * 処理対象の探索。
	 * @param files ファイル配列
	 * @param models モデル・リスト
	 */
	protected void explore(final File[] files, final List<T> models) {
		for (final File file : files) {
			if (file.isDirectory()) {

				// このメソッドを再帰呼び出し
				explore(file.listFiles(), models);

			} else {

				if (XliffFileFilter.INSTANCE.accept(file, file.getName())) {
					try {
						if (monitor.isCanceled()) {
							return;
						}
						CatStatPlugin.getDefault().log(fMsg.getMsg006(file.getName()));
						monitor.subTask(file.getName());

						// ファイルを処理 (テンプレート・メソッド)
						final T model = processFile(file);

						models.add(model);
					} catch (final SAXException e) {
						CatStatPlugin.getDefault().log(e);
					} catch (final IOException e) {
						CatStatPlugin.getDefault().log(e);
					}
				}
			}
		}
	}

	/**
	 * ファイルの処理。
	 * @param file ファイル
	 * @return 処理結果
	 * @throws IOException 入出力例外が発生した場合
	 * @throws SAXException SAX 例外が発生した場合
	 */
	abstract protected T processFile(File file) throws IOException, SAXException;

	/**
	 * ビューの入力をセット。
	 * @param event イベント
	 * @param models モデル・リスト
	 */
	protected void setViewInput(final ExecutionEvent event, final List<T> models) {
		try {
			final IWorkbenchPart selectionPart = HandlerUtil.getActivePart(event);
			final IWorkbenchPage page = HandlerUtil.getActiveWorkbenchWindow(event).getActivePage();
			final AbstractStatView view = (AbstractStatView) page.showView(getViewId());

			view.setInput(models);

			// 連続実行したときに、選択が有効にならないバグを修正。
			// ビューを表示したときに、ISelection を持つビューの選択がはずれることが原因。
			// ビュー表示前にアクティブだったビューに setFocus する。
			// ただし、表示したビューに一度 setFocus する必要あり。
			view.setFocus();
			selectionPart.setFocus();

		} catch (final PartInitException e) {
			CatStatPlugin.getDefault().log(e);
		}
	}

	/**
	 * ビュー id の取得。
	 *
	 * <UL>
	 * <LI>ビュー id はハンドラー・クラス名を元に規定のルールで作成されます。
	 * </UL>
	 *
	 * @return ビュー id
	 */
	protected String getViewId() {
		final String handlerName = getClass().getName();
		return handlerName.replace(".handler.", ".views.").replace(".Show", ".").replace("Handler", "View"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
	}
}
