/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.core;

import java.util.ArrayList;
import java.util.List;

import benten.cat.tm.core.messages.BentenTmDriverManagerMessages;

/**
 * Benten TM (翻訳メモリー) ドライバーを管理するクラス。
 *
 * <UL>
 * <LI>Benten に登録済みの TM ドライバーを管理するためのクラスです。
 * <LI>Benten TM エンジンは、それ自身の実装において BentenTmDriverManager に自分自身を登録するような実装をおこないます。
 * </UL>
 *
 * @author IGA Tosiki
 */
public class BentenTmDriverManager {
	/**
	 * TM ドライバー・マネージャのためのメッセージ。
	 */
	protected static final BentenTmDriverManagerMessages fMsg = new BentenTmDriverManagerMessages();

	/**
	 * 登録されている TM (翻訳メモリー) ドライバーのリスト。
	 */
	protected static final List<BentenTmDriver> fDriverList = new ArrayList<BentenTmDriver>();

	/**
	 * TM ドライバーの登録。
	 *
	 * <UL>
	 * <LI>TM ドライバーを登録します。
	 * </UL>
	 *
	 * @param driver 登録したい TM ドライバー。
	 */
	public static void registerDriver(final BentenTmDriver driver) {
		if (driver == null) {
			throw new IllegalArgumentException(fMsg.getRegisterDriverIllegalArgument1());
		}
		if (driver.getName() == null) {
			throw new IllegalArgumentException(fMsg.getRegisterDriverIllegalArgument2());
		}

		for (final BentenTmDriver look : fDriverList) {
			if (look.getName().equals(driver.getName())) {
				// すでに登録済みのドライバーです。
				// 登録処理をせずに離脱します。
				return;
			}
		}

		fDriverList.add(driver);
	}

	/**
	 * TM ドライバーの登録解除。
	 *
	 * <UL>
	 * <LI>TM ドライバーを登録解除します。
	 * <LI>登録を解除したドライバーは、Class.forName による static ブロックでの登録はできなくなることに注意してください。
	 * </UL>
	 *
	 * @param driver 登録解除したい TM ドライバー。
	 */
	public static void deregisterDriver(final BentenTmDriver driver) {
		if (driver == null) {
			throw new IllegalArgumentException(fMsg.getDeregisterDriverIllegalArgument1());
		}
		if (driver.getName() == null) {
			throw new IllegalArgumentException(fMsg.getDeregisterDriverIllegalArgument2());
		}

		for (final BentenTmDriver look : fDriverList) {
			if (look.getName().equals(driver.getName())) {
				// 該当するドライバーが発見できました。
				// 除去のうえ離脱します。
				fDriverList.remove(look);
				return;
			}
		}
	}

	/**
	 * 指定された名前の TM ドライバーの取得。
	 *
	 * <UL>
	 * <LI>指定された名前の TM ドライバーを取得します。
	 * </UL>
	 *
	 * @param driverName ドライバー名
	 * @return TM ドライバー。
	 */
	public static BentenTmDriver getDriver(final String driverName) {
		if (driverName == null) {
			throw new IllegalArgumentException(fMsg.getGetDriverIllegalArgument1());
		}

		for (final BentenTmDriver driver : fDriverList) {
			if (driver.getName().equals(driverName)) {
				return driver;
			}
		}

		return null;
	}

	/**
	 * 登録されている TM ドライバーの一覧を取得。
	 *
	 * <UL>
	 * <LI>登録されている TM ドライバーの一覧を取得します。
	 * </LI>
	 *
	 * @return TM ドライバーの配列。
	 */
	public static BentenTmDriver[] getDrivers() {
		final BentenTmDriver[] drivers = new BentenTmDriver[fDriverList.size()];
		for (int index = 0; index < fDriverList.size(); index++) {
			drivers[index] = fDriverList.get(index);
		}

		return drivers;
	}
}
