/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.model;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * HTML ブロック・マッチャーです。
 *
 * @author KASHIHARA Shinji
 */
public class HtmlBlockMatcher {

	/**
	 * HTML4 の strict ブロック・タグの正規表現。
	 */
	private static final String HTML4_STRICT_BLOCK_TAGS = "p|div|table|dl|ul|ol|form|address|blockquote|h[1-6]|fieldset|hr|pre"; //$NON-NLS-1$

	/**
	 * 追加のブロック・タグの正規表現。
	 * <UL>
	 * <LI>{@link HTML4_STRICT_BLOCK_TAGS} に加え、構造上、ブロック・タグとして扱う必要があるタグのリストです。
	 * </UL>
	 */
	private static final String ADDITIONAL_BLOCK_TAGS = "body|head|meta|link|script|style|li|title|tr|th|td|thead|tbody|dt|dd"; //$NON-NLS-1$

	/**
	 * すべてのブロック・タグの正規表現。
	 */
	private static final String ALL_BLOCK_TAGS = "(" + ADDITIONAL_BLOCK_TAGS + "|" + HTML4_STRICT_BLOCK_TAGS + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

	/**
	 * ブロック・タグによる実際のタグ正規表現。
	 */
	private static final String BLOCK_TAG_PATTERN = "(?i)(?s)<(|/)" + ALL_BLOCK_TAGS + "(>|\\s+[^>]*>)"; //$NON-NLS-1$ //$NON-NLS-2$

	/**
	 * {@link BLOCK_TAG_PATTERN} の {@link Pattern} インスタンス。
	 */
	private static final Pattern blockTagPattern = Pattern.compile(BLOCK_TAG_PATTERN + "(.+?)" + BLOCK_TAG_PATTERN); //$NON-NLS-1$

	/**
	 * {@link blockTagPattern} のマッチャー。
	 */
	private final Matcher matcher;

	/**
	 * コンストラクター。
	 * @param input 入力文字列
	 */
	public HtmlBlockMatcher(final String input) {
		matcher = blockTagPattern.matcher(input);
	}

	/**
	 * ブロック・タグを検索。
	 * @param start 開始位置
	 * @return 一致した場合は true
	 */
	public boolean find(final int start) {
		return matcher.find(start);
	}

	/**
	 * 一致した開始位置の取得。
	 * @return 開始位置
	 */
	public int start() {
		return matcher.start();
	}

	/**
	 * 一致した部分文字列の取得。
	 * @return 部分文字列
	 */
	public String group() {
		return matcher.group();
	}

	/**
	 * タグの要素名を指定して先頭と末尾のタグを除去。
	 *
	 * <UL>
	 * <LI>tagPattern に含まれる正規表現を文字列として扱う場合は、正規表現エスケープする必要があります。
	 * </UL>
	 *
	 * @param s 対象となる文字列
	 * @param tagPattern タグ・パターン
	 * @return 処理後の文字列
	 */
	public static String trimTag(String s, final String tagPattern) {
		s = s.replaceFirst("(?i)(?s)^" + tagPattern + "\\s*", ""); // 先頭タグ除去 //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		s = s.replaceFirst("(?i)(?s)\\s*" + tagPattern + "$", ""); // 後方タグ除去 //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		return s;
	}

	/**
	 * 先頭と末尾のブロック・タグを除去。
	 * @param s 対象となる文字列
	 * @return 処理後の文字列
	 */
	public static String trimBlockTag(String s) {
		s = trimTag(s, BLOCK_TAG_PATTERN);
		s = trimTag(s, "<br\\s*(|/)\\s*>"); //$NON-NLS-1$
		s = trimTag(s, "(&nbsp;)+"); //$NON-NLS-1$
		s = s.trim();
		return s;
	}
}
