/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.cat.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.cat.core.valueobject.BentenApplyExactMatchProcessInput;

/**
 * Apache Antタスク [BentenApplyExactMatch]のクラス。
 *
 * 完全一致訳適用のための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.cat.core]にBentenApplyExactMatchBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenapplyexactmatch&quot; classname=&quot;benten.twa.cat.core.BentenApplyExactMatchTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenApplyExactMatchTask extends Task {
    /**
     * 完全一致訳適用のための Ant タスク。
     */
    protected BentenApplyExactMatchProcessInput fInput = new BentenApplyExactMatchProcessInput();

    /**
     * フィールド [tmxdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTmxdirProcessed = false;

    /**
     * フィールド [xliffdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldXliffdirProcessed = false;

    /**
     * フィールド [tmdriverclassname] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTmdriverclassnameProcessed = false;

    /**
     * フィールド [ignorewhitespacetmreference] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnorewhitespacetmreferenceProcessed = false;

    /**
     * フィールド [ignoremnemonickeytmreference] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnoremnemonickeytmreferenceProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[tmxdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 翻訳メモリーの TMX が格納されているディレクトリ。\n※tmxdir と tmdriverclassname とは両方同時に指定する必要があります。<br>
     *
     * @param arg セットしたい値
     */
    public void setTmxdir(final String arg) {
        fInput.setTmxdir(arg);
        fIsFieldTmxdirProcessed = true;
    }

    /**
     * Antタスクの[tmxdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 翻訳メモリーの TMX が格納されているディレクトリ。\n※tmxdir と tmdriverclassname とは両方同時に指定する必要があります。<br>
     *
     * @return このフィールドの値
     */
    public String getTmxdir() {
        return fInput.getTmxdir();
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 翻訳対象となる XLIFF が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setXliffdir(final String arg) {
        fInput.setXliffdir(arg);
        fIsFieldXliffdirProcessed = true;
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 翻訳対象となる XLIFF が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getXliffdir() {
        return fInput.getXliffdir();
    }

    /**
     * Antタスクの[tmdriverclassname]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * Ant タスクの場合には、TM ドライバーのクラス名を指定します。\n※tmxdir と tmdriverclassname とは両方同時に指定する必要があります。<br>
     *
     * @param arg セットしたい値
     */
    public void setTmdriverclassname(final String arg) {
        fInput.setTmdriverclassname(arg);
        fIsFieldTmdriverclassnameProcessed = true;
    }

    /**
     * Antタスクの[tmdriverclassname]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * Ant タスクの場合には、TM ドライバーのクラス名を指定します。\n※tmxdir と tmdriverclassname とは両方同時に指定する必要があります。<br>
     *
     * @return このフィールドの値
     */
    public String getTmdriverclassname() {
        return fInput.getTmdriverclassname();
    }

    /**
     * Antタスクの[ignorewhitespacetmreference]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * TMX を検索する際にホワイトスペースを無視するかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnorewhitespacetmreference(final boolean arg) {
        fInput.setIgnorewhitespacetmreference(arg);
        fIsFieldIgnorewhitespacetmreferenceProcessed = true;
    }

    /**
     * Antタスクの[ignorewhitespacetmreference]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * TMX を検索する際にホワイトスペースを無視するかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnorewhitespacetmreference() {
        return fInput.getIgnorewhitespacetmreference();
    }

    /**
     * Antタスクの[ignoremnemonickeytmreference]アトリビュートのセッターメソッド。
     *
     * 項目番号: 5<br>
     * TMX を検索する際にニーモニックを無視するかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnoremnemonickeytmreference(final boolean arg) {
        fInput.setIgnoremnemonickeytmreference(arg);
        fIsFieldIgnoremnemonickeytmreferenceProcessed = true;
    }

    /**
     * Antタスクの[ignoremnemonickeytmreference]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 5<br>
     * TMX を検索する際にニーモニックを無視するかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnoremnemonickeytmreference() {
        return fInput.getIgnoremnemonickeytmreference();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenApplyExactMatchTask begin.");

        // 項目番号[2]、アトリビュート[xliffdir]は必須入力です。入力チェックを行います。
        if (fIsFieldXliffdirProcessed == false) {
            throw new BuildException("必須アトリビュート[xliffdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- tmxdir:[" + getTmxdir() + "]");
            System.out.println("- xliffdir:[" + getXliffdir() + "]");
            System.out.println("- tmdriverclassname:[" + getTmdriverclassname() + "]");
            System.out.println("- ignorewhitespacetmreference:[" + getIgnorewhitespacetmreference() + "]");
            System.out.println("- ignoremnemonickeytmreference:[" + getIgnoremnemonickeytmreference() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenApplyExactMatchProcessインタフェースを実装して benten.twa.cat.coreパッケージに BentenApplyExactMatchProcessImplクラスを作成することにより解決できる場合があります。
            final BentenApplyExactMatchProcess proc = new BentenApplyExactMatchProcessImpl();
            if (proc.execute(fInput) != BentenApplyExactMatchBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
