/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.List;

import benten.core.BentenConstants;
import benten.core.io.IniFile;
import benten.twa.cat.core.valueobject.BentenMarkNoTransableProcessInput;
import benten.twa.cat.messages.BentenMarkNoTransableMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.BlancoXliffSerializer;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 翻訳対象外のマーク
 *
 * <pre>
 * XLIFF 内の翻訳の必要のない翻訳単位を、翻訳対象外としてマークします。
 *   1.  XLIFF の翻訳単位の内容から、翻訳の必要の有無を判断します。
 *   2.  翻訳の必要がないと判断されたものについて、翻訳対象外としてマークします。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式機械支援翻訳機能: 非翻訳対象マーク機能」に対応します。
 *
 * @author IGA Tosiki
 */
public class BentenMarkNoTransableProcessImpl extends AbstractTraverseDir implements BentenMarkNoTransableProcess {
	/**
	 * 翻訳対象外マーク機能のためのメッセージ。
	 */
	protected static final BentenMarkNoTransableMessages fMsg = new BentenMarkNoTransableMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenMarkNoTransableProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 翻訳対象外とマークする文字列のリスト。
	 */
	protected final List<String> markNoTransableList;

	/**
	 * コンストラクター。
	 */
	public BentenMarkNoTransableProcessImpl() {
		markNoTransableList = IniFile.asList(this, "conf/mark-no-transable.ini"); //$NON-NLS-1$
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenMarkNoTransableProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenMarkNoTransableProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirXliff = new File(fInput.getXliffdir());
		if (dirXliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getXliffdir()));
		}
		if (dirXliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getXliffdir()));
		}

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirXliff);

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException("BentenMarkNoTransableProcessImpl#processFile: 'fInput' is null."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		BlancoXliff xliff = null;
		try {
			xliff = new BlancoXliffParser().parse(file);
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE011(file.getName(), e.toString()));
		}

		int transUnitTotalCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			transUnitTotalCount += xliffFile.getBody().getTransUnitList().size();
		}
		beginTask(transUnitTotalCount);

		int transUnitCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			for (final BlancoXliffTransUnit transUnit : xliffFile.getBody().getTransUnitList()) {
				if (progress(fMsg.getCoreP101(file.getName(), BigDecimal.valueOf(++transUnitCount), BigDecimal
						.valueOf(transUnitTotalCount)))) {
					break;
				}

				processTransUnit(transUnit);
			}
		}

		// 書き出し。
		try {
			final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			new BlancoXliffSerializer().serialize(xliff, outStream);
			outStream.flush();

			if (BlancoFileUtil.bytes2FileIfNecessary(outStream.toByteArray(), file) == 0) {
				// 変更無し。
				getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
			} else {
				// 更新のあったファイル数をカウントします。
				getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			}
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE012(file.getName(), e.toString()));
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}
	}

	/**
	 * 翻訳単位を処理。
	 * @param transUnit 翻訳単位
	 */
	protected void processTransUnit(final BlancoXliffTransUnit transUnit) {
		if (transUnit.getTranslate() == false) {
			// すでに除外されているので、これ以上判断する必要はありません。
			return;
		}
		if (transUnit.getTarget() != null && transUnit.getTarget().getTarget() != null) {
			if (transUnit.getTarget().getTarget().length() > 0) {
				// trans-unit の target に何かしらの値が設定済みのもにについては、
				// 何か意図があって設定されているものとみなし、
				// 除外マーク判定の対象外とします。
				return;
			}
		}

		// 翻訳が不要な文字列かどうか判定します。
		if (checkNoTransableString(transUnit.getSource())) {
			fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);
			transUnit.setTranslate(false);
			return;
		}
	}

	/**
	 * 翻訳が不要な文字列かどうか判定します。
	 *
	 * @param argCheckTarget チェック対象となる文字列。
	 * @return 翻訳の必要がなければ true。
	 */
	protected boolean checkNoTransableString(final String argCheckTarget) {
		if (argCheckTarget == null) {
			return true;
		}

		final String trimmedCheckTarget = argCheckTarget.trim();

		// 翻訳不要マーク定義リストをもとに翻訳不要か判断します。
		for (final String pattern : markNoTransableList) {
			if (trimmedCheckTarget.matches(pattern)) {

				// 翻訳不要マーク定義パターンに一致した場合は翻訳不要です。
				return true;
			}
		}

		// 翻訳が必要です。
		return false;
	}
}
