/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.xliff.core;

import java.io.IOException;

import benten.twa.xliff.core.valueobject.BentenExtractXliffFragmentProcessInput;

/**
 * バッチ処理クラス [BentenExtractXliffFragmentBatchProcess]。
 *
 * <P>バッチ処理の呼び出し例。</P>
 * <code>
 * java -classpath (クラスパス) benten.twa.xliff.core.BentenExtractXliffFragmentBatchProcess -help
 * </code>
 */
public class BentenExtractXliffFragmentBatchProcess {
    /**
     * 正常終了。
     */
    public static final int END_SUCCESS = 0;

    /**
     * 入力異常終了。内部的にjava.lang.IllegalArgumentExceptionが発生した場合。
     */
    public static final int END_ILLEGAL_ARGUMENT_EXCEPTION = 7;

    /**
     * 入出力例外終了。内部的にjava.io.IOExceptionが発生した場合。
     */
    public static final int END_IO_EXCEPTION = 8;

    /**
     * 異常終了。バッチの処理開始に失敗した場合、および内部的にjava.lang.Errorまたはjava.lang.RuntimeExceptionが発生した場合。
     */
    public static final int END_ERROR = 9;

    /**
     * コマンドラインから実行された際のエントリポイントです。
     *
     * @param args コンソールから引き継がれた引数。
     */
    public static final void main(final String[] args) {
        final BentenExtractXliffFragmentBatchProcess batchProcess = new BentenExtractXliffFragmentBatchProcess();

        // バッチ処理の引数。
        final BentenExtractXliffFragmentProcessInput input = new BentenExtractXliffFragmentProcessInput();

        boolean isNeedUsage = false;
        boolean isFieldSourcexlifffileProcessed = false;
        boolean isFieldTargetxliffdirProcessed = false;

        // コマンドライン引数の解析をおこないます。
        for (int index = 0; index < args.length; index++) {
            String arg = args[index];
            if (arg.startsWith("-verbose=")) {
                input.setVerbose(Boolean.valueOf(arg.substring(9)).booleanValue());
            } else if (arg.startsWith("-sourcexlifffile=")) {
                input.setSourcexlifffile(arg.substring(17));
                isFieldSourcexlifffileProcessed = true;
            } else if (arg.startsWith("-targetxliffdir=")) {
                input.setTargetxliffdir(arg.substring(16));
                isFieldTargetxliffdirProcessed = true;
            } else if (arg.startsWith("-extractbystate=")) {
                input.setExtractbystate(arg.substring(16));
            } else if (arg.startsWith("-limitsize=")) {
                try {
                    input.setLimitsize(Integer.parseInt(arg.substring(11)));
                } catch (NumberFormatException e) {
                    System.out.println("BentenExtractXliffFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]のフィールド[limitsize]を数値(int)としてパースを試みましたが失敗しました。: " + e.toString());
                    System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
                }
            } else if (arg.startsWith("-includedate=")) {
                input.setIncludedate(Boolean.valueOf(arg.substring(13)).booleanValue());
            } else if (arg.equals("-?") || arg.equals("-help")) {
                usage();
                System.exit(END_SUCCESS);
            } else {
                System.out.println("BentenExtractXliffFragmentBatchProcess: 入力パラメータ[" + arg + "]は無視されました。");
                isNeedUsage = true;
            }
        }

        if (isNeedUsage) {
            usage();
        }

        if( isFieldSourcexlifffileProcessed == false) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcexlifffile]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldTargetxliffdirProcessed == false) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetxliffdir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }

        int retCode = batchProcess.execute(input);

        // 終了コードを戻します。
        // ※注意：System.exit()を呼び出している点に注意してください。
        System.exit(retCode);
    }

    /**
     * 具体的なバッチ処理内容を記述するためのメソッドです。
     *
     * このメソッドに実際の処理内容を記述します。
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IOException 入出力例外が発生した場合。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public int process(final BentenExtractXliffFragmentProcessInput input) throws IOException, IllegalArgumentException {
        // 入力パラメータをチェックします。
        validateInput(input);

        // この箇所でコンパイルエラーが発生する場合、BentenExtractXliffFragmentProcessインタフェースを実装して benten.twa.xliff.coreパッケージに BentenExtractXliffFragmentProcessImplクラスを作成することにより解決できる場合があります。
        final BentenExtractXliffFragmentProcess process = new BentenExtractXliffFragmentProcessImpl();

        // 処理の本体を実行します。
        final int retCode = process.execute(input);

        return retCode;
    }

    /**
     * クラスをインスタンス化してバッチを実行する際のエントリポイントです。
     *
     * このメソッドは下記の仕様を提供します。
     * <ul>
     * <li>メソッドの入力パラメータの内容チェック。
     * <li>IllegalArgumentException, RuntimeException, Errorなどの例外をcatchして戻り値へと変換。
     * </ul>
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public final int execute(final BentenExtractXliffFragmentProcessInput input) throws IllegalArgumentException {
        try {
            // バッチ処理の本体を実行します。
            int retCode = process(input);

            return retCode;
        } catch (IllegalArgumentException ex) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: 入力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_ILLEGAL_ARGUMENT_EXCEPTION;
        } catch (IOException ex) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: 入出力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_IO_EXCEPTION;
        } catch (RuntimeException ex) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: ランタイム例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            ex.printStackTrace();
            // 異常終了。
            return END_ERROR;
        } catch (Error er) {
            System.out.println("BentenExtractXliffFragmentBatchProcess: ランタイムエラーが発生しました。バッチ処理を中断します。:" + er.toString());
            er.printStackTrace();
            // 異常終了。
            return END_ERROR;
        }
    }

    /**
     * このバッチ処理クラスの使い方の説明を標準出力に示すためのメソッドです。
     */
    public static final void usage() {
        System.out.println("BentenExtractXliffFragmentBatchProcess: Usage:");
        System.out.println("  java benten.twa.xliff.core.BentenExtractXliffFragmentBatchProcess -verbose=値1 -sourcexlifffile=値2 -targetxliffdir=値3 -extractbystate=値4 -limitsize=値5 -includedate=値6");
        System.out.println("    -verbose");
        System.out.println("      説明[verboseモードで動作させるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -sourcexlifffile");
        System.out.println("      説明[抽出元となる XLIFF ファイル。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -targetxliffdir");
        System.out.println("      説明[抽出先となる XLIFF を格納するディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -extractbystate");
        System.out.println("      説明[状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。]");
        System.out.println("      型[文字列]");
        System.out.println("    -limitsize");
        System.out.println("      説明[件数で区切って抽出する場合の件数値。無指定なら -1 で件数では区切らない。]");
        System.out.println("      型[数値(int)]");
        System.out.println("      デフォルト値[-1]");
        System.out.println("    -includedate");
        System.out.println("      説明[ファイル名に日付を含めるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -? , -help");
        System.out.println("      説明[使い方を表示します。]");
    }

    /**
     * このバッチ処理クラスの入力パラメータの妥当性チェックを実施するためのメソッドです。
     *
     * @param input バッチ処理の入力パラメータ。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public void validateInput(final BentenExtractXliffFragmentProcessInput input) throws IllegalArgumentException {
        if (input == null) {
            throw new IllegalArgumentException("BlancoBatchProcessBatchProcess: 処理開始失敗。入力パラメータ[input]にnullが与えられました。");
        }
        if (input.getSourcexlifffile() == null) {
            throw new IllegalArgumentException("BentenExtractXliffFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcexlifffile]に値が設定されていません。");
        }
        if (input.getTargetxliffdir() == null) {
            throw new IllegalArgumentException("BentenExtractXliffFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetxliffdir]に値が設定されていません。");
        }
    }
}
