/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.xliff.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.xliff.core.valueobject.BentenExtractXliffFragmentProcessInput;

/**
 * Apache Antタスク [BentenExtractXliffFragment]のクラス。
 *
 * XLIFF フラグメント抽出のための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.xliff.core]にBentenExtractXliffFragmentBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenextractxlifffragment&quot; classname=&quot;benten.twa.xliff.core.BentenExtractXliffFragmentTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenExtractXliffFragmentTask extends Task {
    /**
     * XLIFF フラグメント抽出のための Ant タスク。
     */
    protected BentenExtractXliffFragmentProcessInput fInput = new BentenExtractXliffFragmentProcessInput();

    /**
     * フィールド [sourcexlifffile] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcexlifffileProcessed = false;

    /**
     * フィールド [targetxliffdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetxliffdirProcessed = false;

    /**
     * フィールド [extractbystate] に値がセットされたかどうか。
     */
    protected boolean fIsFieldExtractbystateProcessed = false;

    /**
     * フィールド [limitsize] に値がセットされたかどうか。
     */
    protected boolean fIsFieldLimitsizeProcessed = false;

    /**
     * フィールド [includedate] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIncludedateProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcexlifffile]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 抽出元となる XLIFF ファイル。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcexlifffile(final String arg) {
        fInput.setSourcexlifffile(arg);
        fIsFieldSourcexlifffileProcessed = true;
    }

    /**
     * Antタスクの[sourcexlifffile]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 抽出元となる XLIFF ファイル。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcexlifffile() {
        return fInput.getSourcexlifffile();
    }

    /**
     * Antタスクの[targetxliffdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 抽出先となる XLIFF を格納するディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetxliffdir(final String arg) {
        fInput.setTargetxliffdir(arg);
        fIsFieldTargetxliffdirProcessed = true;
    }

    /**
     * Antタスクの[targetxliffdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 抽出先となる XLIFF を格納するディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetxliffdir() {
        return fInput.getTargetxliffdir();
    }

    /**
     * Antタスクの[extractbystate]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * 状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。<br>
     *
     * @param arg セットしたい値
     */
    public void setExtractbystate(final String arg) {
        fInput.setExtractbystate(arg);
        fIsFieldExtractbystateProcessed = true;
    }

    /**
     * Antタスクの[extractbystate]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * 状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。<br>
     *
     * @return このフィールドの値
     */
    public String getExtractbystate() {
        return fInput.getExtractbystate();
    }

    /**
     * Antタスクの[limitsize]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * 件数で区切って抽出する場合の件数値。無指定なら -1 で件数では区切らない。<br>
     *
     * @param arg セットしたい値
     */
    public void setLimitsize(final String arg) {
        try {
            fInput.setLimitsize(Integer.parseInt(arg));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("Antタスクの[limitsize]アトリビュートに与えられた値の数値解析に失敗しました。" + e.toString());
        }
        fIsFieldLimitsizeProcessed = true;
    }

    /**
     * Antタスクの[limitsize]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * 件数で区切って抽出する場合の件数値。無指定なら -1 で件数では区切らない。<br>
     * デフォルト値[-1]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getLimitsize() {
        return String.valueOf(fInput.getLimitsize());
    }

    /**
     * Antタスクの[includedate]アトリビュートのセッターメソッド。
     *
     * 項目番号: 5<br>
     * ファイル名に日付を含めるかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIncludedate(final boolean arg) {
        fInput.setIncludedate(arg);
        fIsFieldIncludedateProcessed = true;
    }

    /**
     * Antタスクの[includedate]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 5<br>
     * ファイル名に日付を含めるかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIncludedate() {
        return fInput.getIncludedate();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenExtractXliffFragmentTask begin.");

        // 項目番号[1]、アトリビュート[sourcexlifffile]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcexlifffileProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcexlifffile]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[targetxliffdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetxliffdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetxliffdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcexlifffile:[" + getSourcexlifffile() + "]");
            System.out.println("- targetxliffdir:[" + getTargetxliffdir() + "]");
            System.out.println("- extractbystate:[" + getExtractbystate() + "]");
            System.out.println("- limitsize:[" + getLimitsize() + "]");
            System.out.println("- includedate:[" + getIncludedate() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenExtractXliffFragmentProcessインタフェースを実装して benten.twa.xliff.coreパッケージに BentenExtractXliffFragmentProcessImplクラスを作成することにより解決できる場合があります。
            final BentenExtractXliffFragmentProcess proc = new BentenExtractXliffFragmentProcessImpl();
            if (proc.execute(fInput) != BentenExtractXliffFragmentBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
