/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.views;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

/**
 * テーブル・カラム・ソート・アダプター。
 *
 * <UL>
 * <LI>{@link TableViewer} のカラムをソートするための基底アダプダー・クラスです。
 * </UL>
 *
 * @param <T> モデルの型
 * @author KASHIHARA Shinji
 */
public abstract class TableColumnSortAdapter<T> extends SelectionAdapter {

	/** テーブル・ビューアー */
	private final TableViewer viewer;

	/** カラム・インデックス */
	private final int columnIndex;

	/**
	 * コンストラクター。
	 * @param viewer テーブル・ビューアー
	 * @param columnIndex カラム・インデックス
	 */
	public TableColumnSortAdapter(final TableViewer viewer, final int columnIndex) {
		this.viewer = viewer;
		this.columnIndex = columnIndex;
	}

	@SuppressWarnings("unchecked")
	@Override
	public void widgetSelected(final SelectionEvent e) {
		final TableColumn column = (TableColumn) e.getSource();
		if (columnIndex == viewer.getTable().indexOf(column)) {
			final TableColumnSorter<T> sorter = createTableColumnSorter();
			final TableColumnSorter oldSorter = (TableColumnSorter<T>) viewer.getSorter();
			if (oldSorter != null) {
				sorter.ascendng = !oldSorter.ascendng;
			}
			viewer.setSorter(sorter);

			final Table table = viewer.getTable();
			table.setSortColumn(table.getColumn(columnIndex));
			table.setSortDirection(sorter.ascendng ? SWT.UP : SWT.DOWN);
		}
	}

	/**
	 * カラム・インデックスの取得。
	 * @return カラム・インデックス
	 */
	public int getColumnIndex() {
		return columnIndex;
	}

	/**
	 * テーブル・カラム・ソーターの作成。
	 * @return テーブル・カラム・ソーター
	 */
	abstract public TableColumnSorter<T> createTableColumnSorter();

	/**
	 * テーブル・カラム・ソーター・クラス。
	 * @param <T> モデルの型
	 */
	public abstract static class TableColumnSorter<T> extends ViewerSorter {

		/** ソートが昇順の場合は true */
		protected boolean ascendng;

		@SuppressWarnings("unchecked")
		@Override
		public int compare(final Viewer view, final Object obj1, final Object obj2) {
			final int direction = ascendng ? 1 : -1;
			return compare((T) obj1, (T) obj2) * direction;
		}

		/**
		 * モデルを比較するテンプレート・メソッド。
		 * @param model1 モデル 1
		 * @param model2 モデル 2
		 * @return 比較結果
		 */
		abstract public int compare(T model1, T model2);

		/**
		 * 文字列の数値比較。
		 * @param s1 文字列 1
		 * @param s2 文字列 2
		 * @return 比較結果
		 */
		protected int compareInteger(final String s1, final String s2) {
			try {
				return getInteger(s1).compareTo(getInteger(s2));
			} catch (final RuntimeException e) {
				return 0;
			}
		}

		/**
		 * 文字列を数値に変換。
		 * @param s 文字列
		 * @return 変換後の数値。変換できない場合は 0。
		 */
		protected Integer getInteger(final String s) {
			try {
				return Integer.parseInt(s.replaceAll("[^\\d]+", "")); //$NON-NLS-1$ //$NON-NLS-2$
			} catch (final RuntimeException e) {
				return 0;
			}
		}
	}
}
