/*
**  bif1_a.c
**  bif-c
**
**  Created by Joel Rees on 2009/08/16.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIF1/A, as mechanically as possible.
*/


#include "bif_m.h"

#include "bif7b_a.h"	/* To link into the BIF vocabulary. */
#include "bifb_a.h"	/* To link to the previous vocabulary. */
#include "bif1_a.h"


/*
00010 * Include file for BIF
00020 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00030 *
01000 	FCC 'MOVE'
01010 	FCB 4
01020 	FCB MFORE
01030 	FDB PREF-CFAOFF
01040 	FDB BIF+2
01050 	FDB MINUS-CFAOFF
01060 	FDB NFA-CFAOFF
*/
static character_t sMOVE[] = "\x4" "MOVE";
definition_header_s hMOVE =
{	{ (natural_t) sMOVE },
	{ 0 },
	{ (natural_t) &hPREF },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hMINUS },
	{ (natural_t) &hNFA },
	{ (natural_t) MOVE }
};
/*
01070 MOVE	LDD ,U++ count=0?
01080 	BEQ MOVEX
01090 	PSHS Y,U
01100 	LDY ,U
01110 	LDU 2,U
01120 MOVELP	PULU X
01130 	STX ,Y++
01140 	SUBD #1
01150 	BNE MOVELP
01160 	PULS Y,U
01170 MOVEX	LEAU 4,U
01180 	NEXT
01190 *
*/
void MOVE(void)	/* Ascending, overlapped works only when source is above destination, see below. */
{	cell_u * source = SP[ 2 ].cellp;
	cell_u * target = SP[ 1 ].cellp;
	natural_t count = SP[ 0 ].integer;
	SP += 3;
	for ( ; count > 0; --count )
	{	* target++ = * source++;
	}
}
/* About overlap, the full story is that overlopping works with ascending order 
// only when the destination start point is outside the source start point.
// That description more fully covers issues of address wrap-around.
// (Not usually a problem, since most computers don't have their address space entirely filled with RAM
//  and, since data at address 0 is preferably avoided when NULL (nil) is of the ( (void *) 0 ) variety.)
// So, the test would be somehting like this:
//	if ( ( target > source ) && ( target < source + count ) )
//	{	source += count;	// exceptional case, descending
//		target += count;
//		for ( ; count > 0; --count )
//		{	* --target = * --source;
//		}
//	}
//	else
//	{	for ( ; count > 0; --count )	// ascending
//		{	* target++ = * source++;
//		}
//	}
// This still doesn't properly cover the cases, however.
//
// Since it doesn't hurt to copy descending when there is no overlap,
// only the first test is really necessary when neither source nore destination wraps through address zero.
// It also works when the destination wraps through zero.
// It doesn't work when the source wraps through zero and the destination starts in the wrapped portion. 
//
// But I don't really want to test that now, and overlapping is going to be rare enough, 
// I don't want to tell myself I've solved the problem when I don't know that I have.
// Predecrement on the descending copy should be the way to handle the pointers pointing one beyond
// after the ends are calculated, but, as I say, it's not standard idiom, so I shouldn't trust my memory.
//
// If the source end is greater than the source start address (no wrap over zero),
// copy direction is almost safely determined by comparing the target start with the source start: 
//      below -> copy ascending, otherwise descending.
// If the source wraps (end is less than start), 
// copy direction is determined by comparing the target start with the source end:
//      below or equal to actual end -> copy descending, otherwise ascending.
//
// Almost.
//
// If the size of the object being copied exceeds half of memory space, 
// there is a possibility of overlap on both ends. 
// If there is overlap on both ends, we have multiple problems.
// If the object code we are running is in the same memory space (the usual case), 
// we have to move the code we are running.
// If not, we have to play strategy games to avoid overwriting things before they get copied.
//
// This exposes a hidden issue. 
// We are treating copies as if they should never fail, but they might.
// Part of the issue here is that it's impossible to test all the error input conditions in a timely manner.
// Another part of the issue is that the generalized copies are a feature of Unix systems, 
// where 0 is usually NULL. So strings should never wrap through address 0.
//
// Anyway, strings as big as half the memory space should never occur in ordinary programs, 
// and should be assumed to require their own special routines.
//
// But we don't have any way to fail gracefully on such input, 
// so we don't try to implement a universal move.
*/


/*
01200 	FCC 'CMOVE'
01210 	FCB 5
01220 	FCB MFORE
01225 	FDB MOVE-CFAOFF
01230 	FDB BIF+2
01240 	FDB CFA-CFAOFF
01250 	FDB COMP-CFAOFF
*/
static character_t sCMOVE[] = "\x5" "CMOVE";
definition_header_s hCMOVE =
{	{ (natural_t) sCMOVE },
	{ 0 },
	{ (natural_t) &hMOVE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hCFA },
	{ (natural_t) &hCOMP },
	{ (natural_t) CMOVE }
};
/*
01260 CMOVE	LDD #0
01270 	SUBD ,U++
01280 	PSHS A,Y
01290 	PULU X,Y
01300 	BEQ CMOVEX
01310 CMOVEL	LDA ,Y+
01320 	STA ,X+
01330 	INCB
01340 	BNE CMOVEL
01350 	INC ,S
01360 	BNE CMOVEL
01370 CMOVEX	PULS A,Y
01380 	NEXT
01390 *
*/
void CMOVE(void)	/* Ascending, see MOVE for comments on overlap. */
{	byte_t * source = SP[ 2 ].bytep;
	byte_t * target = SP[ 1 ].bytep;
	natural_t count = SP[ 0 ].integer;
	SP += 3;
	for ( ; count > 0; --count )
	{	* target++ = * source++;
	}
}


/*
01400 	FCC 'U*'
01410 	FCB 2
01420 	FCB MFORE
01430 	FDB CMOVE-CFAOFF
01440 	FDB BIF+2
01450 	FDB TOG-CFAOFF
01460 	FDB UPDATE-CFAOFF
*/
static character_t sUSTAR[] = "\x2" "U*";
definition_header_s hUSTAR =
{	{ (natural_t) sUSTAR },
	{ 0 },
	{ (natural_t) &hCMOVE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hTOG },
	{ (natural_t) &hUPDATE },
	{ (natural_t) USTAR }
};
/*
01470 USTAR	LEAS -4,S
01480 	LDA 1,U LEAST
01490 	LDB 3,U
01500 	MUL
01510 	STD 2,S
01520 	LDA ,U most
01530 	LDB 2,U
01540 	MUL
01550 	STD ,S
01560 	LDD 1,U inner
01570 	MUL
01580 	ADDD 1,S
01590 	BCC *+4
01600 	INC ,S
01605 	STD 1,S
01610 	LDA ,U
01620 	LDB 3,U
01630 	MUL
01640 	ADDD 1,S
01650 	BCC *+4
01660 	INC ,S
01670 	STD 1,S
01680 	PULS D,X
01690 	STD ,U
01700 	STX 2,U
01710 	NEXT
01720 *
*/
void USTAR(void)
{
#if !defined MANUFACTURED_DOUBLE
#	if !defined LOW_C_CELL_FIRST	/* FORTH CPU byte order for doubles. */
	dblnatural_t * stack;
	dblnatural_t right = ( * SP++ ).integer;
	dblnatural_t left = ( * SP++ ).integer;
	stack = (dblnatural_t *) ( (char *) SP );
	/* dblnatural_t result = left * right;
	// dblnatural_t * resultp = (dblnatural_t *) SP;
	// * resultp = result; 
	*/
	* --stack = left * right;	/* Native CPU byte order is FORTH order. */
	SP = (cell_u *) ( (char*) stack );
#	else /* defined LOW_C_CELL_FIRST */
	dblnatural_t right = SP[ 0 ].integer;
	dblnatural_t left = SP[ 1 ].integer;
	left *= right;
	SP[ 1 ].integer = (natural_t) left;	/* Native CPU byte order is _not_ FORTH order. */
	SP[ 0 ].integer = (natural_t) ( left >> BITSPERCELL );
#	endif /* !defined LOW_C_CELL_FIRST */
#else /* defined MANUFACTURED_DOUBLE */
/*dbg // printf( "%lx %lx U*\n", SP[ 1 ].integer, SP[ 0 ].integer ); */
	natural_t leftHi = HIHALFDOWN( SP[ 1 ].integer );
	natural_t leftLo = LOHALFCELL( SP[ 1 ].integer );
	natural_t rightHi = HIHALFDOWN( SP[ 0 ].integer );
	natural_t rightLo = LOHALFCELL( SP[ 0 ].integer );
	natural_t rLo = leftLo * rightLo;
	natural_t rMid1 = leftLo * rightHi;
	natural_t rMid2 = leftHi * rightLo;
	natural_t accm = HIHALFDOWN( rLo ) + LOHALFCELL( rMid1 ) + LOHALFCELL( rMid2 );
	SP[ 1 ].integer = LOHALFCELL( rLo ) | LOHALFUP( accm );	/* Will spill bits on odd size cells. */
	accm >>= LOHALFCT;
	SP[ 0 ].integer = accm + leftHi * rightHi + HIHALFDOWN( rMid1 ) + HIHALFDOWN( rMid2 );
/*dbg // printf( " U* result %lx:%lx\n", SP[ 0 ].integer, SP[ 1 ].integer ); */
#endif /* !defined MANUFACTURED_DOUBLE */
}


/*
01730 	FCC 'U/'
01740 	FCB 2
01750 	FCB MFORE
01760 	FDB USTAR-CFAOFF
01770 	FDB BIF+2
01780 	FDB 0
01790 	FDB 0
*/
static character_t sUSLASH[] = "\x2" "U/";
definition_header_s hUSLASH =
{	{ (natural_t) sUSLASH },
	{ 0 },
	{ (natural_t) &hUSTAR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) USLASH }
};
/*
01800 USLASH	LDA #17 bit ct
01810 	PSHS A
01820 	LDD 2,U dividend
01830 USLDIV	CMPD ,U divisor
01840 	BHS USLSUB
01850 	ANDCC #.NOT.1
01860 	BRA USLBIT
01870 USLSUB	SUBD ,U
01880 	ORCC #1 quotient,
01890 USLBIT	ROL 5,U save it
01900 	ROL 4,U
01910 	DEC ,S more bits?
01920 	BEQ USLR
01930 	ROLB remainder
01940 	ROLA
01950 	BCC USLDIV
01960 	BRA USLSUB
01970 USLR	LEAS 1,S
01980 	LEAU 2,U
01990 	LDX 2,U
02000 	STD 2,U
02010 	STX ,U
02020 	NEXT
02030 *
*/
void USLASH(void)
{
	natural_t divisor = ( * SP++ ).integer;	/* Leave it pointing to the unsigned double dividend. */
#if !defined MANUFACTURED_DOUBLE
	/* If we were to re-use M/MOD, it would look like this: 
	// dblnatural_t * stack;
	// natural_t temp;
	// MSMOD();
	// stack = (dblnatural_t *) ( (char *) SP );
	// temp = * stack++;
	// SP = (cell_u *) ( (char *) stack );
	// * --SP = temp;
	// So we might as well do it again here.
	*/
#	if !defined LOW_C_CELL_FIRST	/* FORTH CPU byte order for doubles. */
	dblnatural_t * stack = (dblnatural_t *) ( (char *) SP );
	dblnatural_t dividend = * stack++;	/* Native CPU byte order for doubles. */
	SP = (cell_u *) ( (char *) stack );
	SP -= 2;	/* Leave the stack ready for the results. */
#	else /* defined LOW_C_CELL_FIRST */
	dblnatural_t dividend =  ( (dblnatural_t) SP[ 0 ].integer << BITSPERCELL ) | SP[ 1 ].integer;
#	endif /* !defined LOW_C_CELL_FIRST */
	SP[ 1 ].integer = (natural_t) ( dividend % divisor );
	SP[ 0 ].integer = (natural_t) ( dividend / divisor );	/* Drops potentially significant high bits. */
#else /* defined MANUFACTURED_DOUBLE */
	natural_t dividendHi = SP[ 0 ].integer;
	natural_t dividendLo = SP[ 1 ].integer;
/* dbg // printf( "start dividendHi: %d, dividendLo: %d, divisor: %d\n", dividendHi, dividendLo, divisor ); */
	if ( dividendHi == 0 ) 	/* Short circuit */
	{	SP[ 1 ].integer = dividendLo % divisor;
		SP[ 0 ].integer = dividendLo / divisor;
/* dbg // printf( "native qLo: %d, rem: %d\n", SP[ 0 ].integer, SP[ 1 ].integer ); */
	}
	else if ( HIHALFCELL( divisor ) == 0 )	/* Fix bugs here in M/MOD, too. */
	{ 	/* This covers the full range of dividends vs. half-cell-only divisors. */
		/* natural_t quotientHi = dividendHi / divisor; // throwaway, except when debugging */
		natural_t carry = dividendHi % divisor; /* 0 .. halfcell */
		natural_t dividendMidHi = LOHALFUP( carry ) | HIHALFDOWN ( dividendLo );
		natural_t quotientMidLo = dividendMidHi / divisor;
		natural_t quotientLo;
/* dbg // printf( "qHi: %d, dMHi: %d, qMLo: %d, carry: %d\n", quotientHi, dividendMidHi, quotientMidLo, carry ); */
		carry = dividendMidHi % divisor;
		dividendLo &= HALFLOMASK;
		dividendLo |= LOHALFUP( carry );	/* will spill on odd cell sizes */
		quotientLo = dividendLo / divisor;
/* dbg printf( "qLo: %d, dLo: %d, carry: %d\n", quotientLo, dividendLo, carry ); */
		SP[ 1 ].integer = dividendLo % divisor;
		SP[ 0 ].integer = LOHALFUP( quotientMidLo ) | quotientLo;	/* will spill on odd cell sizes */
/* dbg // printf( "half qLo: %d, rem: %d\n", SP[ 0 ].integer, SP[ 1 ].integer ); */
	}
	else	/* Fix bugs here in M/MOD, too. */
	{ 	/* Divisor > halfcell */
		unsigned count = BITSPERCELL * 2 + 1;	/* Should be able to reduce this to BITSPERCELL. */
		natural_t quotientHi = 0;
		natural_t quotientLo = 0;
		natural_t carry = 0;
		natural_t dividendLead = 0;
		for ( ;; )
		{
			if ( carry || ( dividendLead >= divisor ) ) /* 2's complement only */
			{
				quotientLo |= 1;
				dividendLead -= divisor;
			}
			if ( --count <= 0 )
			{ 	break;
			}
			quotientHi <<= 1;
			if ( ( quotientLo & CELL_HIGH_BIT ) != 0 )
			{ 	quotientHi |= 1;
			}
			quotientLo <<= 1;
			carry = dividendLead & CELL_HIGH_BIT;
			dividendLead <<= 1;
			if ( ( dividendHi & CELL_HIGH_BIT ) != 0 )
			{ 	dividendLead |= 1;
			}
			dividendHi <<= 1;
			if ( ( dividendLo & CELL_HIGH_BIT ) != 0 )
			{	dividendHi |= 1;
			}
			dividendLo <<= 1;
		}
		SP[ 0 ].integer = quotientLo;
		SP[ 1 ].integer = dividendLead; /* At this point, the remainder. */
/* dbg // printf( "nibble qLo: %d, rem: %d\n", SP[ 0 ].integer, SP[ 1 ].integer ); */
	}
#endif /* !defined MANUFACTURED_DOUBLE */
}


/*
02200 	FCC 'AND'
02210 	FCB 3
02220 	FCB MFORE
02230 	FDB USLASH-CFAOFF
02240 	FDB BIF+2
02250 	FDB ABS-CFAOFF
02260 	FDB BACK-CFAOFF
*/
static character_t sAND[] = "\x3" "AND";
definition_header_s hAND =
{	{ (natural_t) sAND },
	{ 0 },
	{ (natural_t) &hUSLASH },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hABS },
	{ (natural_t) &hBACK },
	{ (natural_t) AND }
};
/*
02270 AND	PULU D
02280 	ANDB 1,U
02290 	ANDA ,U
02300 	STD ,U
02310 	NEXT
02320 *
*/
void AND(void)
{	natural_t right = ( * SP++ ).integer;
	SP[ 0 ].integer = SP[ 0 ].integer & right;
}


/*
02330 	FCC 'OR'
02340 	FCB 2
02350 	FCB MFORE
02360 	FDB AND-CFAOFF
02370 	FDB BIF+2
02380 	FDB OCT-CFAOFF
02390 	FDB OUT-CFAOFF
*/
static character_t sOR[] = "\x2" "OR";
definition_header_s hOR =
{	{ (natural_t) sOR },
	{ 0 },
	{ (natural_t) &hAND },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hOCT },
	{ (natural_t) &hOUT },
	{ (natural_t) OR }
};
/*
02400 OR	PULU D
02410 	ORB 1,U
02420 	ORA ,U
02430 	STD ,U
02440 	NEXT
02450 *
*/
void OR(void)
{	natural_t right = ( * SP++ ).integer;
	SP[ 0 ].integer = SP[ 0 ].integer | right;
}


/*
02460 	FCC 'XOR'
02470 	FCB 3
02480 	FCB MFORE
02490 	FDB OR-CFAOFF
02500 	FDB BIF+2
02510 	FDB 0
02520 	FDB 0
*/
static character_t sXOR[] = "\x3" "XOR";
definition_header_s hXOR =
{	{ (natural_t) sXOR },
	{ 0 },
	{ (natural_t) &hOR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hWORD },
	{ (natural_t) &hBCOMP },
	{ (natural_t) XOR }
};
/*
02530 XOR	PULU D
02540 	EORB 1,U
02550 	EORA ,U
02560 	STD ,U
02570 	NEXT
02580 *
*/
void XOR(void)
{	natural_t right = ( * SP++ ).integer;
	SP[ 0 ].integer = SP[ 0 ].integer ^ right;
}


/*
02590 	FCC 'SP@'
02600 	FCB 3
02610 	FCB MFORE
02620 	FDB XOR-CFAOFF
02630 	FDB BIF+2
02640 	FDB SPSTO-CFAOFF
02660 	FDB 0
*/
static character_t sSPFEH[] = "\x3" "SP@";
definition_header_s hSPFEH =
{	{ (natural_t) sSPFEH },
	{ 0 },
	{ (natural_t) &hXOR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hSPSTO },
	{ 0 },
	{ (natural_t) SPFEH }
};
/*
02670 SPFEH	TFR U,X
02680 	PSHU X
02690 	NEXT
02700 *
*/
void SPFEH(void)
{	cell_u *  top = SP;
	( * --SP ).cellp = top;
}


/*
02710 	FCC 'SP!'
02720 	FCB 3
02730 	FCB MFORE
02740 	FDB SPFEH-CFAOFF
02750 	FDB BIF+2
02760 	FDB 0
02770 	FDB 0
*/
static character_t sSPSTO[] = "\x3" "SP!";
definition_header_s hSPSTO =
{	{ (natural_t) sSPSTO },
	{ 0 },
	{ (natural_t) &hSPFEH },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) SPSTO }
};
/*
02780 SPSTO LDX <UP
02790 	LDU US0,X
02793 	CLR ,U hole
02796 	CLR 1,U
02800 	NEXT
02810 *
*/
void SPSTO(void)
{	SP = UP.task->dataStackBase.cellp;
	SP[ 0 ].integer = SP[ 1 ].integer = 0;	/* Clear the crash buffer hole. */
}


/*
02820 	FCC 'RP!'
02830 	FCB 3
02840 	FCB MFORE
02850 	FDB SPSTO-CFAOFF
02860 	FDB BIF+2
02870 	FDB 0
02880 	FDB 0
*/
static character_t sRPSTO[] = "\x3" "RP!";
definition_header_s hRPSTO =
{	{ (natural_t) sRPSTO },
	{ 0 },
	{ (natural_t) &hSPSTO },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) RPSTO }
};
/*
02890 RPSTO LDX <UP
02900 	LDS UR0,X
02903 	CLR ,S hole
02906 	CLR 1,S
02910 	NEXT
02920 *
*/
void RPSTO(void)
{	RP = UP.task->returnStackBase.cellp;
	RP[ 0 ].integer = RP[ 1 ].integer = 0;	/* Clear the crash buffer hole. */
}


/*
02930 	FCC ';S'
02940 	FCB MIMM.OR.2
02950 	FCB MFORE
02960 	FDB RPSTO-CFAOFF
02970 	FDB BIF+2
02980 	FDB 0
02990 	FDB 0
*/
static character_t sSEMIS[] = "\x2" ";S";
definition_header_s hSEMIS =
{	{ (natural_t) sSEMIS },
	{ MIMM },
	{ (natural_t) &hRPSTO },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) SEMIS }
};
/*
03000 SEMIS	PULS Y un-nest
03010 	NEXT
03020 *
*/
void SEMIS(void)
{	/* IP = ( * RP++ ).cellp;	// I kind of want to do this after the XNEXT loop in XCOL and XDOES, but ... */
	if ( sysSIG.integer == ICODE_LIST_CONTINUE )
	{	sysSIG.integer = ICODE_LIST_END;
	}
}


/*
03030 	FCC 'LEAVE'
03040 	FCB MCOMP.OR.5
03050 	FCB MFORE
03060 	FDB SEMIS-CFAOFF
03070 	FDB BIF+2
03080 	FDB LATEST-CFAOFF
03090 	FDB LFA-CFAOFF
*/
static character_t sLEAVE[] = "\x5" "LEAVE";
definition_header_s hLEAVE =
{	{ (natural_t) sLEAVE },
	{ MCOMP },
	{ (natural_t) &hSEMIS },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hLATEST },
	{ (natural_t) &hLFA },
	{ (natural_t) LEAVE }
};
/*
03100 LEAVE	LDD ,S index
03110 	STD 2,S to limit
03120 	NEXT
03130 *
*/
void LEAVE(void)
{	RP[ 1 ] = RP[ 0 ];	/* Set index to limit. This also might have sign issues. */
}


/*
03140 	FCC '>R'
03150 	FCB MCOMP.OR.2
03160 	FCB MFORE
03170 	FDB LEAVE-CFAOFF
03180 	FDB BIF+2
03190 	FDB GT-CFAOFF
03200 	FDB QDOT-CFAOFF
*/
static character_t sTOR[] = "\x2" ">R";
definition_header_s hTOR =
{	{ (natural_t) sTOR },
	{ MCOMP },
	{ (natural_t) &hLEAVE },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hGT },
	{ (natural_t) &hQDOT },
	{ (natural_t) TOR }
};
/*
03210 TOR	PULU D
03220 	PSHS D
03230 	NEXT
03240 *
*/
void TOR(void)
{	( * --RP ) = ( * SP++ );
}


/*
03250 	FCC 'R>'
03260 	FCB MCOMP.OR.2
03270 	FCB MFORE
03280 	FDB TOR-CFAOFF
03290 	FDB BIF+2
03300 	FDB R-CFAOFF
03310 	FDB REPEAT-CFAOFF
*/
static character_t sRFROM[] = "\x2" "R>";
definition_header_s hRFROM =
{	{ (natural_t) sRFROM },
	{ MCOMP },
	{ (natural_t) &hTOR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hR },
	{ (natural_t) &hREPEAT },
	{ (natural_t) RFROM }
};
/*
03320 RFROM	JMP <XVAR not JSR!
03350 *
*/
void RFROM(void)
{	( * --SP ) = ( * RP++ );
}


/*
03360 	FCC 'R'
03370 	FCB 1
03380 	FCB MFORE
03390 	FDB RFROM-CFAOFF
03400 	FDB BIF+2
03410 	FDB QUIT-CFAOFF
03420 	FDB RNUM-CFAOFF
*/
static character_t sR[] = "\x1" "R";
definition_header_s hR =
{	{ (natural_t) sR },
	{ 0 },
	{ (natural_t) &hRFROM },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hQUIT },
	{ (natural_t) &hRNUM },
	{ (natural_t) I }
};
/*
03430 R	JMP I
03431 *
*/
/* 
void R(void)
{	I();
}
*/


/*
03432 	FCC '='
03433 	FCB 1
03434 	FCB MFORE
03435 	FDB R-CFAOFF
03436 	FDB BIF+2
03437 	FDB LT-CFAOFF
03438 	FDB QCOMP-CFAOFF
*/
static character_t sEQ[] = "\x1" "=";
definition_header_s hEQ =
{	{ (natural_t) sEQ },
	{ 0 },
	{ (natural_t) &hR },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hLT },
	{ (natural_t) &hQCOMP },
	{ (natural_t) EQ }
};
/*
03439 EQ	PULU D
03440 	CMPD ,U
03441 	BEQ TRUE
03442 	BRA FALSE
*/
void EQ(void)
{	natural_t right = ( * SP++ ).integer;
	SP[ 0 ].integer = ( SP[ 0 ].integer == right ) ? TRUE : FALSE;
}


/*
03451 *
03452 	FCC '<'
03453 	FCB 1
03454 	FCB MFORE
03455 	FDB EQ-CFAOFF
03456 	FDB BIF+2
03457 	FDB SCODE-CFAOFF
03458 	FDB BEGHSH-CFAOFF
*/
static character_t sLT[] = "\x1" "<";
definition_header_s hLT =
{	{ (natural_t) sLT },
	{ 0 },
	{ (natural_t) &hEQ },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hSCODE },
	{ (natural_t) &hBEGHSH },
	{ (natural_t) LT }
};
/*
03459 LT	LDD 2,U
03460 	CMPD ,U++
03461 	BLT TRUE
03462 	BRA FALSE
03463 *
*/
void LT(void)
{	snatural_t right = ( * SP++ ).sinteger;
	SP[ 0 ].integer = ( SP[ 0 ].sinteger < right ) ? TRUE : FALSE;
}


/*
03490 	FCC '0='
03491 	FCB 2
03492 	FCB MFORE
03493 	FDB LT-CFAOFF
03494 	FDB BIF+2
03500 	FDB 0
03510 	FDB ONE-CFAOFF
*/
static character_t sZEQ[] = "\x2" "0=";
definition_header_s hZEQ =
{	{ (natural_t) sZEQ },
	{ 0 },
	{ (natural_t) &hLT },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ (natural_t) &hONE },
	{ (natural_t) ZEQ }
};
/*
03520 ZEQ	LDD ,U
03530 	BNE FALSE
03540 TRUE	LDD #-1
03550 	STD ,U
03560 	NEXT
03570 *
*/
void ZEQ(void)
{	SP[ 0 ].integer = ( SP[ 0 ].integer == 0 ) ? TRUE : FALSE;
}


/*
03580 	FCC '0<'
03590 	FCB 2
03600 	FCB MFORE
03610 	FDB ZEQ-CFAOFF
03620 	FDB BIF+2
03630 	FDB ZERO-CFAOFF
03640 	FDB ZEQ-CFAOFF
*/
static character_t sZLESS[] = "\x2" "0<";
definition_header_s hZLESS =
{	{ (natural_t) sZLESS },
	{ 0 },
	{ (natural_t) &hZEQ },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hZERO },
	{ (natural_t) &hZEQ },
	{ (natural_t) ZLESS }
};
/*
03650 ZLESS	LDA ,U
03660 	BMI TRUE
03665 FALSE	LDD #0
03670 	STD ,U
03680 	NEXT
03681 *
*/
void ZLESS(void)
{	SP[ 0 ].integer = ( SP[ 0 ].sinteger < 0 ) ? TRUE : FALSE;
}


/*
03682 	FCC '>'
03683 	FCB 1
03684 	FCB MFORE
03685 	FDB ZLESS-CFAOFF
03686 	FDB BIF+2
03687 	FDB 0
03688 	FDB TOPRT-CFAOFF
*/
static character_t sGT[] = "\x1" ">";
definition_header_s hGT =
{	{ (natural_t) sGT },
	{ 0 },
	{ (natural_t) &hZLESS },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ (natural_t) &hTOPRT },
	{ (natural_t) GT }
};
/*
03689 GT	LDD 2,U
03690 	CMPD ,U++
03691 	BGT TRUE
03692 	BRA FALSE
03693 *
*/
void GT(void)
{	snatural_t right = ( * SP++ ).sinteger;
	SP[ 0 ].integer = ( SP[ 0 ].sinteger > right ) ? TRUE : FALSE;
}


/*
03700 	FCC '+'
03710 	FCB 1
03720 	FCB MFORE
03730 	FDB GT-CFAOFF
03740 	FDB BIF+2
03750 	FDB PAREN-CFAOFF
03760 	FDB SLASH-CFAOFF
*/
static character_t sADD[] = "\x1" "+";
definition_header_s hADD =
{	{ (natural_t) sADD },
	{ 0 },
	{ (natural_t) &hGT },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hPAREN },
	{ (natural_t) &hSLASH },
	{ (natural_t) ADD }
};
/*
03770 ADD	PULU D
03780 	ADDTOP
03810 *
*/
void ADD(void)
{	natural_t right = ( * SP++ ).integer;
/*	ADDTOP( right ); Hides things we may not want hidden. */
	( * SP ).integer += right;
}


/*
03820 	FCC '-'
03830 	FCB 1
03840 	FCB MFORE
03850 	FDB ADD-CFAOFF
03860 	FDB BIF+2
03870 	FDB 0
03880 	FDB 0
*/
static character_t sSUB[] = "\x1" "-";
definition_header_s hSUB =
{	{ (natural_t) sSUB },
	{ 0 },
	{ (natural_t) &hADD },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) SUB }
};
/*
03890 SUB	LDD 2,U
03900 	SUBD ,U++
03910 	STD ,U
03920 	NEXT
03930 *
*/
void SUB(void)
{	natural_t right = ( * SP++ ).integer;
/*	SUBTOP( right ); Hides things we may not want hidden. */
	( * SP ).integer -= right;
}


/*
03940 	FCC 'D+'
03950 	FCB 2
03960 	FCB MFORE
03970 	FDB SUB-CFAOFF
03980 	FDB BIF+2
03990 	FDB CONST-CFAOFF
04000 	FDB DABS-CFAOFF
*/
static character_t sDADD[] = "\x2" "D+";
definition_header_s hDADD =
{	{ (natural_t) sDADD },
	{ 0 },
	{ (natural_t) &hSUB },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hCONST },
	{ (natural_t) &hDABS },
	{ (natural_t) DADD }
};
/*
04010 DADD	LDD 6,U
04020 	ADDD 2,U
04030 	STD 6,U
04040 	LDD 4,U
04050 	ADCB 1,U
04060 	ADCA ,U
04070 	LEAU 4,U
04080 	STD ,U
04090 	NEXT
04100 *
*/
void DADD(void)
{
#if !defined MANUFACTURED_DOUBLE
#	if !defined LOW_C_CELL_FIRST	/* FORTH CPU byte order for doubles. */
	dblnatural_t * stack = (dblnatural_t *) (char *) SP;
	dblnatural_t right = * stack++;
	* stack += right;
	SP = (cell_u *) (char *) stack;
#	else /* defined LOW_C_CELL_FIRST */
	dblnatural_t left = ( (dblnatural_t) SP[ 0 ].integer << BITSPERCELL ) | SP[ 1 ].integer;
	dblnatural_t right = ( (dblnatural_t) SP[ 2 ].integer << BITSPERCELL ) | SP[ 3 ].integer;
	SP += 2; 
	left += right;
	SP[ 1 ].integer = (natural_t) left;
	SP[ 0 ].integer = (natural_t) ( left >> BITSPERCELL );
#	endif /* !defined LOW_C_CELL_FIRST */
#else /* defined MANUFACTURED_DOUBLE */
	natural_t accmLo = LOHALFCELL( SP[ 3 ].integer ) + LOHALFCELL( SP[ 1 ].integer );
	natural_t accmHi = HIHALFDOWN( SP[ 3 ].integer ) + HIHALFDOWN( SP[ 1 ].integer ) + HIHALFDOWN( accmLo );
	SP[ 3 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
	accmLo = LOHALFCELL( SP[ 2 ].integer ) + LOHALFCELL( SP[ 0 ].integer ) + HIHALFDOWN( accmHi );
	accmHi = HIHALFDOWN( SP[ 2 ].integer ) + HIHALFDOWN( SP[ 0 ].integer ) + HIHALFDOWN( accmLo );
	SP[ 2 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
	SP += 2;
#endif /* !defined MANUFACTURED_DOUBLE */

}


/*
04120 	FCC 'D-'
04130 	FCB 2
04140 	FCB MFORE
04150 	FDB DADD-CFAOFF
04160 	FDB BIF+2
04170 	FDB DCHS-CFAOFF
04180 	FDB DDOT-CFAOFF
*/
static character_t sDSUB[] = "\x2" "D-";
definition_header_s hDSUB =
{	{ (natural_t) sDSUB },
	{ 0 },
	{ (natural_t) &hDADD },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hDCHS },
	{ (natural_t) &hDDOT },
	{ (natural_t) DSUB }
};
/*
04190 DSUB	LDD 6,U
04200 	SUBD 2,U
04210 	STD 6,U
04220 	LDD 4,U
04230 	SBCB 1,U
04240 	SBCA ,U
04250 	LEAU 4,U
04260 	STD ,U
04270 	NEXT
04280 *
*/
void DSUB(void)
{
#if !defined MANUFACTURED_DOUBLE
#	if !defined LOW_C_CELL_FIRST	/* FORTH CPU byte order for doubles. */
	dblnatural_t * stack = (dblnatural_t *) (char *) SP;
	dblnatural_t right = * stack++;
	* stack -= right;	/* Native CPU byte order for doubles. Except I'm not sure about the pointer conversion. */
	SP = (cell_u *) (char *) stack;
#	else /* defined LOW_C_CELL_FIRST */
	dblnatural_t left = ( (dblnatural_t) SP[ 0 ].integer << BITSPERCELL ) | SP[ 1 ].integer;
	dblnatural_t right = ( (dblnatural_t) SP[ 2 ].integer << BITSPERCELL ) | SP[ 3 ].integer;
	SP += 2;
	left -= right;
	SP[ 1 ].integer = (natural_t) left;
	SP[ 0 ].integer = (natural_t) ( left >> BITSPERCELL );
#	endif /* !defined LOW_C_CELL_FIRST */
#else /* defined MANUFACTURED_DOUBLE */
	natural_t accmLo = LOHALFCELL( SP[ 3 ].integer ) - LOHALFCELL( SP[ 1 ].integer );
	natural_t accmHi = HIHALFDOWN( SP[ 3 ].integer ) - HIHALFDOWN( SP[ 1 ].integer ) - LOHALFCARRYBIT( accmLo );
	SP[ 3 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
	accmLo = LOHALFCELL( SP[ 2 ].integer ) - LOHALFCELL( SP[ 0 ].integer ) - LOHALFCARRYBIT( accmHi );
	accmHi = HIHALFDOWN( SP[ 2 ].integer ) - HIHALFDOWN( SP[ 0 ].integer ) - LOHALFCARRYBIT( accmLo );
	SP[ 2 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
	SP += 2;
#endif /* !defined MANUFACTURED_DOUBLE */
}


/*
04290 	FCC 'MINUS'
04300 	FCB 5
04310 	FCB MFORE
04320 	FDB DSUB-CFAOFF
04330 	FDB BIF+2
04340 	FDB 0
04350 	FDB MOD-CFAOFF
*/
static character_t sMINUS[] = "\x5" "MINUS";
definition_header_s hMINUS =
{	{ (natural_t) sMINUS },
	{ 0 },
	{ (natural_t) &hDSUB },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ (natural_t) &hMOD },
	{ (natural_t) MINUS }
};
/*
04360 MINUS	LDD #0
04370 	SUBD ,U
04380 	STD ,U
04390 	NEXT
04400 *
*/
void MINUS(void)
{
	SP[ 0 ].sinteger = 0 - SP[ 0 ].sinteger;	/* Semantically, same as -SP[ 0 ].sinteger. */
}


/*
04410 	FCC 'DMINUS'
04420 	FCB 6
04430 	FCB MFORE
04440 	FDB MINUS-CFAOFF
04450 	FDB BIF+2
04460 	FDB 0
04470 	FDB 0
*/
static character_t sDMINUS[] = "\x6" "DMINUS";
definition_header_s hDMINUS =
{	{ (natural_t) sDMINUS },
	{ 0 },
	{ (natural_t) &hMINUS },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) DMINUS }
};
/*
04480 DMINUS	LDD #0
04490 	SUBD 2,U
04500 	STD 2,U
04510 	LDD #0
04520 	SBCB 1,U
04530 	SBCA ,U
04570 	STD ,U
04580 	NEXT
04590 *
*/
void DMINUS(void)
{
#if !defined MANUFACTURED_DOUBLE
#	if !defined LOW_C_CELL_FIRST	/* FORTH CPU CELL order for doubles. */
	sdblnatural_t * stack = (sdblnatural_t *) ( (char *) SP );
	* stack = 0LL - * stack;	/* Native CPU byte order for doubles. */
	/* Semantically, same as - * stack, I think. */
#	else /* defined LOW_C_CELL_FIRST */
	sdblnatural_t result = ( (sdblnatural_t) SP[ 0 ].sinteger << BITSPERCELL ) | SP[ 1 ].integer;
	result = 0LL - result;
	SP[ 1 ].integer = (natural_t) result;
	SP[ 0 ].sinteger = (snatural_t) ( result >> BITSPERCELL );
#	endif /* !defined LOW_C_CELL_FIRST */
#else /* defined MANUFACTURED_DOUBLE */
	natural_t accmLo = 0L - LOHALFCELL( SP[ 1 ].integer );
	natural_t accmHi = 0L - HIHALFDOWN( SP[ 1 ].integer ) - LOHALFCARRYBIT( accmLo );
	SP[ 1 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
	accmLo = 0L - LOHALFCELL( SP[ 0 ].integer ) - LOHALFCARRYBIT( accmHi );
	accmHi = 0L - HIHALFDOWN( SP[ 0 ].integer ) - LOHALFCARRYBIT( accmLo );
	SP[ 0 ].integer = LOHALFUP( accmHi ) | LOHALFCELL( accmLo );	/* odd bits beyond carries */
#endif /* !defined MANUFACTURED_DOUBLE */
}


/*
04600 	FCC 'OVER'
04610 	FCB 4
04620 	FCB MFORE
04630 	FDB DMINUS-CFAOFF
04640 	FDB BIF+2
04650 	FDB OR-CFAOFF
04660 	FDB PFA-CFAOFF
*/
static character_t sOVER[] = "\x4" "OVER";
definition_header_s hOVER =
{	{ (natural_t) sOVER },
	{ 0 },
	{ (natural_t) &hDMINUS },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hOR },
	{ (natural_t) &hPFA },
	{ (natural_t) OVER }
};
/*
04670 OVER	LDD 2,U
04680 	PSHU D
04690 	NEXT
04700 *
*/
void OVER(void)
{	--SP;	/* Don't really want to mess with the time point issues. */
	SP[ 0 ] = SP[ 2 ];
}


/*
04710 	FCC 'DROP'
04720 	FCB 4
04730 	FCB MFORE
04740 	FDB OVER-CFAOFF
04750 	FDB BIF+2
04760 	FDB DPL-CFAOFF
04770 	FDB EMIT-CFAOFF
*/
static character_t sDROP[] = "\x4" "DROP";
definition_header_s hDROP =
{	{ (natural_t) sDROP },
	{ 0 },
	{ (natural_t) &hOVER },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hDPL },
	{ (natural_t) &hEMIT },
	{ (natural_t) DROP }
};
/*
04780 DROP	LEAU 2,U
04790 	NEXT
04800 *
*/
void DROP(void)
{	++SP;
}


/*
04810 	FCC 'SWAP'
04820 	FCB 4
04830 	FCB MFORE
04840 	FDB DROP-CFAOFF
04850 	FDB BIF+2
04860 	FDB ROT-CFAOFF
04870 	FDB VAR-CFAOFF
*/
static character_t sSWAP[] = "\x4" "SWAP";
definition_header_s hSWAP =
{	{ (natural_t) sSWAP },
	{ 0 },
	{ (natural_t) &hDROP },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) &hROT },
	{ (natural_t) &hVAR },
	{ (natural_t) SWAP }
};
/*
04880 SWAP	PULU D,X
04890 	PSHU D
04900 	PSHU X
04910 	NEXT
04920 *
*/
void SWAP(void)
{	cell_u	cell = SP[ 0 ];
	SP[ 0 ] = SP[ 1 ];
	SP[ 1 ] = cell;
}


/*
04930 	FCC 'DUP'
04940 	FCB 3
04940 	FCB MFORE
04950 	FDB SWAP-CFAOFF
04960 	FDB BIF+2
04970 	FDB 0 * DUMP-CFAOFF
04980 	FDB ELSE-CFAOFF
*/
static character_t sDUP[] = "\x3" "DUP";
definition_header_s hDUP =	
{	{ (natural_t) sDUP },
	{ 0 },
	{ (natural_t) &hSWAP },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ (natural_t) &hELSE },
	{ (natural_t) DUP }
};
/*
04990 DUP	LDD ,U
05000 	PSHU D
05010 	NEXT
05020 *
*/
void DUP(void)
{	--SP;	/* Don't really want to mess with the time point issues. */
	SP[ 0 ] = SP[ 1 ];
}


/*
05030 	FCC '+!'
05040 	FCB 2
05050 	FCB MFORE
05060 	FDB DUP-CFAOFF
05070 	FDB BIF+2
05080 	FDB 0
05090 	FDB 0
*/
static character_t sADDSTO[] = "\x2" "+!";
definition_header_s hADDSTO =	
{	{ (natural_t) sADDSTO },
	{ 0 },
	{ (natural_t) &hDUP },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) ADDSTO }
};
/*
05100 ADDSTO	PULU X
05110 	LDD ,X
05120 	ADDD ,U++
05130 	STD ,X
05140 	NEXT
05150 *
*/
void ADDSTO(void)
{	cell_u * target = SP[ 0 ].cellp;
	( * target ).integer += SP[ 1 ].integer;
	SP += 2;
}
