/*
 * Copyright (C) 2006-2007 BLI Project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,
 * with or without modification, are permitted provided
 * that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain
 * the above copyright notice, this list of conditions
 * and the following disclaimer as the first lines of this file unmodified.
 * 
 * 2. Redistributions  in  binary  form must reproduce 
 * the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.sf.bli.framework.batch;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.bli.framework.common.BusinessLogicException;
import jp.sf.bli.framework.common.Globals;
import jp.sf.bli.framework.common.util.FrameworkUtil;
import jp.sf.bli.framework.logging.Logging;

/**
 * バッチ処理用 ビジネスロジックの実行をコントロールするクラスのabstractクラス
 * 
 * @author Author: bose
 * @version 1
 */
abstract public class BusinessLogicCtrlBT {

    protected String                    loginId        = "Batch-User";

    /** BLValidate */
    protected BusinessLogicValidateBT   blValidate     = null;

    /** BlDo List */
    protected List< BusinessLogicDoBT > blList         = null;

    /** BL処理前データ格納Map */
    protected Map< String, Object >     inputValueMap  = null;

    /** BL処理データ連携用Map */
    protected Map< String, Object >     outputValueMap = null;


    /**
     * @return this.loginId
     */
    public String getLoginId() {

        return this.loginId;
    }


    /**
     * @param loginId
     *            this.loginId = loginId
     */
    public void setLoginId( String loginId ) {

        this.loginId = loginId;
    }


    /**
     * @return this.blList
     */
    public List< BusinessLogicDoBT > getBlList() {

        return this.blList;
    }


    /**
     * @return this.blValidate
     */
    public BusinessLogicValidateBT getBlValidate() {

        return this.blValidate;
    }


    /**
     * @return this.inputValueMap
     */
    public Map< String, Object > getInputValueMap() {

        return this.inputValueMap;
    }


    /**
     * @param inputValueMap
     *            this.inputValueMap = inputValueMap
     */
    public void setInputValueMap( Map< String, Object > inputValueMap ) {

        this.inputValueMap = inputValueMap;
    }


    /**
     * @return this.outputValueMap
     */
    public Map< String, Object > getOutputValueMap() {

        return this.outputValueMap;
    }


    /**
     * @param outputValueMap
     *            this.outputValueMap = outputValueMap
     */
    public void setOutputValueMap( Map< String, Object > outputValueMap ) {

        this.outputValueMap = outputValueMap;
    }


    /**
     * @param blValidate
     *            設定する BusinessLogicValidate
     */
    public void setBlValidate( BusinessLogicValidateBT blValidate ) {

        this.blValidate = blValidate;

    }


    /**
     * @param blList
     *            設定する BusinessLogicDo
     */
    public void setBlList( List< BusinessLogicDoBT > blList ) {

        this.blList = blList;

    }


    /**
     * ビジネスロジックの実行を制御する
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @return 遷移先を意味するint
     * @throws BusinessLogicException
     *             実行時に各種Exceptionが起きた場合に<br />
     *             BusinessLogicExceptionを発生させてthrowする
     */
    public int doBizCtrl( String[] args ) throws BusinessLogicException {

        int resultFlagInt = -1000;

        try {
            /*
             * BLCtrl Initialize
             * ------------------------------------------------
             */

            init( args );

            initProcess( args );

            /*
             * FormBean To inputValueMap
             * ----------------------------------------
             */

            parameter2InputValueMap( args );

            /*
             * BLValidate、BLDoの実行
             * -------------------------------------------------------------
             */

            blDoLoop( args, doVaildate( args ) );

            /*
             * BLCtrl End Processing
             * --------------------------------------------
             */

            endProcess( args );

            end( args );

        } catch ( Exception e ) {

            Logging.fatal( this.getClass(), "BLI000020", getLoginId(),
                    "doBizCtrl:BLCtrl catch Exception. " + e.getMessage() );

            e.printStackTrace();
            throw new BusinessLogicException( "LogCount：BLI000017" );

        }

        resultFlagInt = ( ( Integer ) outputValueMap.get( Globals.RESULT_FLAG ) )
                .intValue();

        // ActionにresultFlagを返して処理を終える
        return resultFlagInt;
    }


    /**
     * @param session
     * @param resultFlagInt
     * @throws Exception
     */
    protected void blDoLoop( String[] args, int resultFlagInt )
            throws Exception {

        Integer resultFlag;

        if ( resultFlagInt != -999 && resultFlagInt != 0 ) {

            Iterator< BusinessLogicDoBT > itr = this.blList.iterator();

            // blDoの数だけBLDoを実行する
            while ( itr.hasNext() ) {

                Logging.debug( this.getClass(), "BLI000023", getLoginId(),
                        "blDoLoop:BusineesLogicDo Strat" );

                BusinessLogicDoBT blDo = itr.next();

                if ( doBL( args, blDo ) == false ) {

                    Logging.debug( this.getClass(), "BLI000024", getLoginId(),
                            "blDoLoop:BusineesLogicDo No Normal End" );

                    resultFlag = ( Integer ) this.outputValueMap
                            .get( Globals.RESULT_FLAG );

                    resultFlagInt = resultFlag.intValue();

                    if ( resultFlagInt == -999 || resultFlagInt == 0 ) {

                        Logging.debug( this.getClass(), "BLI000025",
                                getLoginId(), "blDoLoop:break; resultFlag:"
                                        + resultFlagInt );

                        break;

                    }

                }

                Logging.debug( this.getClass(), "BLI000026", getLoginId(),
                        "blDoLoop:BusineesLogicDo Normal End" );
            }
        }
    }


    /**
     * BlCtrlの初期処理前のシステムの初期処理を行う
     * 
     * @param session
     * @throws Exception
     */
    public void init( String[] args ) throws Exception {

        this.inputValueMap = new HashMap< String, Object >();

        this.outputValueMap = new HashMap< String, Object >();

        Logging.debug( this.getClass(), "BLI000021", getLoginId(),
                "doBizCtrl:BLCtrl started." );
    }


    /**
     * BlCtrlの実行を行う前に初期処理を行う
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @throws Exception
     */
    abstract protected void initProcess( String[] args ) throws Exception;


    /**
     * 画面からの入力をinputValueMapへ詰め替える
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @param ActionFormのname
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void parameter2InputValueMap( String[] args )
            throws Exception;


    /**
     * BlValidateの実行を制御する
     * 
     * @param HttpSession
     * @param inputValueMap
     * @param outputValueMap
     * @return validateチェック合格:true/validateチェック不合格:false
     * @throws Exception
     */
    protected int doVaildate( String[] args ) throws Exception {

        int resultFlagInt = 0;

        Integer resultFlag = null;

        if ( this.blValidate != null ) {

            // BLValidate実行
            this.blValidate.doValidate( this.loginId, this.inputValueMap,
                    this.outputValueMap );

            // BLValidateの処理結果をoutputValueMapから取得
            resultFlag = ( Integer ) outputValueMap.get( Globals.RESULT_FLAG );

            // BLValidateでエラーになった場合 エラーの場合は0を返す
            if ( resultFlag == null || resultFlag.intValue() == 0 ) {

                // HttpSessionにViewに表示するエラーメッセージを格納する
                String errorMes = ( String ) this.outputValueMap
                        .get( Globals.MAP_KEY_ERROR_MES );

                FrameworkUtil.setMapErrorMes( this.outputValueMap, errorMes, 0 );

                if ( resultFlag == null ) {

                    // 想定外のエラーをセット
                    resultFlagInt = -999;
                    resultFlag = Integer.valueOf( -999 );
                    outputValueMap.put( Globals.RESULT_FLAG, Integer
                            .valueOf( resultFlagInt ) );

                }

                Logging.error( this.getClass(), "BLI000022", getLoginId(),
                        "doBizCtrl:Validate ended because of the error. rtnFlag:["
                                + resultFlag + "] errerMes:[" + errorMes + "]" );
            }

        } else {

            resultFlag = Integer.valueOf( -1000 );

        }

        resultFlagInt = resultFlag.intValue();

        return resultFlagInt;
    }


    /**
     * BlDoの実行を制御する
     * 
     * @param HttpSession
     * @param BusinessLogicDo
     * @return BlDo正常終了:true/BlDo論理エラー終了:false
     * @throws Exception
     */
    protected boolean doBL( String[] args, BusinessLogicDoBT blDo )
            throws Exception {

        boolean rtnFlag = true; // 成功

        blDo.doBizLogic( this.loginId, this.inputValueMap, this.outputValueMap );

        // URLの引数でSampleDBを検索する

        // BLDoの処理結果をoutputValueMapから取得
        Integer resultFlag = ( Integer ) outputValueMap
                .get( Globals.RESULT_FLAG );

        // BLDoでエラーになった場合
        if ( resultFlag == null || resultFlag.intValue() == 0 ) {

            // BL処理結果をfailureのfalseにセットする
            rtnFlag = false;

            // ログに出力するエラーメッセージを格納する
            String errorMes = ( String ) this.outputValueMap
                    .get( Globals.MAP_KEY_ERROR_MES );

            FrameworkUtil.setMapErrorMes( this.outputValueMap, errorMes, 1 );

            if ( resultFlag == null ) {

                // 想定外のエラーをセット
                resultFlag = Integer.valueOf( -999 );
                outputValueMap.put( Globals.RESULT_FLAG, resultFlag );

            }

            Logging.error( this.getClass(), "BLI000027", getLoginId(),
                    "doBizCtrl:Validate ended because of the error. rtnFlag:["
                            + resultFlag + "] errerMes:[" + errorMes + "]" );

        }

        return rtnFlag;

    }


    /**
     * BlCtrlの実行後に終了処理を行う
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @throws Exception
     */
    abstract protected void endProcess( String[] args ) throws Exception;


    /**
     * BlCtrlの初期処理前のシステムの初期処理を行う
     * 
     * @param session
     * @throws Exception
     */
    protected void end( String[] args ) throws Exception {

        Logging.debug( this.getClass(), "BLI000028", loginId,
                "doBizCtrl:BLCtrl ended normally. rtnFlag:[0]" );

    }

}
