/*
 * Copyright (C) 2006-2007 BLI Project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,
 * with or without modification, are permitted provided
 * that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain
 * the above copyright notice, this list of conditions
 * and the following disclaimer as the first lines of this file unmodified.
 * 
 * 2. Redistributions  in  binary  form must reproduce 
 * the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.sf.bli.framework.ss;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import jp.sf.bli.framework.common.Globals;
import jp.sf.bli.framework.common.util.FrameworkUtil;
import jp.sf.bli.framework.logging.Logging;

import org.apache.struts.action.Action;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;

/**
 * BLI Frameworkの基礎Struts Action<br />
 * このクラスを継承してロジックを記述していく
 * 
 * @author bose
 */
abstract public class BliStrutsAction extends Action {

    /** 画面表示エラーメッセージ */
    private static final String LOG_MESSAGE       = "Fatal error occurred.";

    /** 処理開始ログメッセージ */
    private static final String START_LOG_MESSAGE = "BliStrutsAction started:";

    /** パラメータログメッセージ */
    private static final String PARAM_LOG_MESSAGE = "Parameter:";

    /** 処理終了ログメッセージ */
    private static final String END_LOG_MESSAGE   = "BliStrutsAction ended:";

    /** FINDNAMEログメッセージ */
    private static final String END_LOG_FOWARD    = " foward:";

    /** 実行時間ログメッセージ */
    private static final String TIME_LOG_MESSAGE  = " Processing time:";

    /** BusinessLogicCtrl */
    private BusinessLogicCtrlSS blCtrl            = null;

    /** loginId */
    private String              loginId           = "";

    /** loginRole */
    private List< String >      loginRole         = null;

    /** ログインチェックフラグ DIにてセット */
    private String              loginChkFlag      = "1";                       // 0がチェックなし

    /** セッションメッセージ初期化フラグ DIにてセット */
    private String              mesInitFlag       = "0";


    /**
     * @return this.mesInitFlag
     */
    public String getMesInitFlag() {

        return this.mesInitFlag;
    }


    /**
     * @param mesInitFlag
     *            this.mesInitFlag = mesInitFlag
     */
    public void setMesInitFlag( String mesInitFlag ) {

        this.mesInitFlag = mesInitFlag;
    }


    /**
     * @param blCtrl
     *            this.blCtrl = blCtrl
     */
    public void setBlCtrl( BusinessLogicCtrlSS blCtrl ) {

        this.blCtrl = blCtrl;

    }


    /**
     * @return blCtrl
     */
    public BusinessLogicCtrlSS getBlCtrl() {

        return blCtrl;
    }


    /**
     * @return this.loginChkFlag
     */
    public String getLoginChkFlag() {

        return this.loginChkFlag;
    }


    /**
     * @param loginChkFlag
     *            this.loginChkFlag = loginChkFlag
     */
    public void setLoginChkFlag( String loginChkFlag ) {

        this.loginChkFlag = loginChkFlag;
    }


    /**
     * @return this.loginId
     */
    public String getLoginId() {

        return this.loginId;
    }


    /**
     * @param loginId
     *            this.loginId = loginId
     */
    public void setLoginId( String loginId ) {

        this.loginId = loginId;
    }


    /**
     * @return this.loginRole
     */
    public List< String > getLoginRole() {

        return this.loginRole;
    }


    /**
     * @param loginRole
     *            this.loginRole = loginRole
     */
    public void setLoginRole( List< String > loginRole ) {

        if ( loginRole == null ) {
            loginRole = new ArrayList< String >();
        }

        this.loginRole = loginRole;
    }


    /**
     * Strutsから最初に実行されるメソッド
     * 
     * @param mapping
     *            リクエストとアクションクラスのインスタンスマッピング
     * @param form
     *            ActionMappings に関連づけることのできるJavaBean
     * @param request
     *            HTTPリクエスト
     * @param response
     *            HTTPレスポンス
     * @return Actionクラスの処理の結果
     */
    public ActionForward execute( ActionMapping mapping, ActionForm form,
            HttpServletRequest req, HttpServletResponse res ) {

        /** execute実行開始時間 */
        long start = System.currentTimeMillis();

        /** アプリケーションコンテキスト */
        ServletContext sc = null;

        /** HTTPセッション */
        HttpSession session = null;

        String fowardName = null;

        boolean fatalFlag = false;

        String fatalLogCount = "";

        ActionForward actionForward = null;

        try {
            
            sc = super.getServlet().getServletContext();
            session = req.getSession();

            // ログイン情報をSessionからセット
            this.loginId = getLoginId( session );
            this.loginRole = getLoginRole( session );

            fowardName = loginChk( req );

            loggingStart( req );

            mesInit( session );

            if ( fowardName == null ) {

                // 機能別Action実行前処理実行
                fowardName = doProcessBefore( sc, session, req, res );

            }

            // fowardNameがnullの場合、機能別Actionを実行
            if ( fowardName == null ) {

                // 機能別Actionを実行
                fowardName = doProcess( mapping, form, sc, session, req, res );

                // 機能別Action実行後処理実行
                doProcessAfter( mapping, form, sc, session, req, res );

            }

        } catch ( Exception e ) {

            // エラーログ出力
            Logging.fatal( this.getClass(), "BLI000013", this.loginId,
                    LOG_MESSAGE + " " + e.getMessage() );

            e.printStackTrace();

            fowardName = Globals.STATUS_FATAL;

            fatalFlag = true;
            fatalLogCount = "BLI000013";

        }

        loggingEnd( start, fowardName );

        actionForward = forwardSet( mapping, session, fowardName, fatalFlag,
                fatalLogCount );

        return actionForward;

    }


    /**
     * Springの設定をチェックしてログインチェックを行う
     */
    protected String loginChk( HttpServletRequest req ) {

        String rtnStr = null;

        String clientAdd = req.getRemoteAddr();
        String reqURL = null;
        
        //RequestURLが取得できない場合はreqURLにnullをセットする
        try {

                reqURL = req.getRequestURL().toString();
                
        } catch ( NullPointerException e ) {
            
            reqURL = null;
        
        }
        
        // ログインチェックが有効ならばログインチェックを行う
        if ( this.loginChkFlag.equals( "0" ) ) {

            // ログインId確認できるかチェックする
            if ( this.loginId == null || this.loginId.equals( "" )
                    || this.loginId.equals( Globals.NO_LOGIN_ID ) ) {

                rtnStr = Globals.STATUS_NO_LOGIN;

                // エラーログ出力
                Logging.error( this.getClass(), "BLI000003", this.loginId,
                        "Login Error : NoLoginUser Access IP: " + clientAdd
                                + " URL: " + reqURL );
            }

        } else {

            Logging.debug( this.getClass(), "BLI000004", this.loginId,
                    "No Login Check IP: " + clientAdd + " URL: " + reqURL );

        }

        StringBuffer sbuf = new StringBuffer();
        sbuf.append( "Audit Log. Login Role : [ " );

        Iterator< String > itr = this.loginRole.iterator();

        while ( itr.hasNext() ) {

            sbuf = sbuf.append( itr.next() );
            sbuf = sbuf.append( " " );

        }

        sbuf.append( "]" );

        // 監査のためにログ出力
        Logging.info( this.getClass(), "BLI000005", this.loginId, sbuf
                .toString()
                + " IP: " + clientAdd + " URL: " + reqURL );

        return rtnStr;

    }


    /**
     * fowardNameからActionForwardをセットする
     * 
     * @param mapping
     * @param session
     * @param fowardName
     * @param fatalFlag
     * @param fatalLogCount
     * @return
     */
    protected ActionForward forwardSet( ActionMapping mapping,
            HttpSession session, String fowardName, boolean fatalFlag,
            String fatalLogCount ) {

        ActionForward actionForward = null;

        if ( fowardName == null ) {

            actionForward = null;

        } else {

            if ( fatalFlag ) {

                // HttpSessionにViewに表示するFATAL発生時の
                // エラーメッセージを格納する
                String errorMes = "System Fatal Error. "
                        + "Please, contact system administrator. "
                        + "loginId:[" + this.loginId + "] logCount:["
                        + fatalLogCount + "]";

                FrameworkUtil.setSessionErrorMes( session, errorMes, 0 );

            }

            actionForward = mapping.findForward( fowardName );

        }

        return actionForward;

    }


    /**
     * 終了のロギングを行う
     * 
     * @param start
     * @param message
     * @param fowardName
     */
    protected void loggingEnd( long start, String fowardName ) {

        StringBuffer message = new StringBuffer();

        // 処理終了表示メッセージを生成
        message.append( END_LOG_MESSAGE );
        message.append( END_LOG_FOWARD );
        message.append( fowardName );
        message.append( TIME_LOG_MESSAGE );
        message.append( System.currentTimeMillis() - start );

        // 終了ログ出力
        Logging.info( this.getClass(), "BLI000002", this.loginId, message
                .toString() );
    }


    /**
     * 処理開始のロギングを行う
     * 
     * @param req
     * @param message
     * @param parameterMap
     */
    protected void loggingStart( HttpServletRequest req ) {

        Map parameterMap = req.getParameterMap();

        // 処理開始表示メッセージを生成
        StringBuffer message = new StringBuffer();

        message.append( START_LOG_MESSAGE );
        message.append( PARAM_LOG_MESSAGE );

        if ( parameterMap != null ) {

            if ( parameterMap.size() != 0 ) {

                for ( Iterator itr = parameterMap.keySet().iterator(); itr
                        .hasNext(); ) {

                    String key = ( String ) itr.next();

                    String[] keys = req.getParameterValues( key );

                    for ( int i = 0; keys.length > i; i++ ) {
                        message.append( " [" );
                        message.append( key );
                        message.append( "=" );
                        message.append( keys[i] );
                        message.append( "]" );
                    }
                }

            } else {
                message.append( " [ No Parameter ]" );
            }
        }

        Logging.info( this.getClass(), "BLI000001", this.loginId, message
                .toString() );
    }


    /**
     * 機能別Action実行前処理
     * 
     * @param application
     *            HTTPコンテキスト
     * @param session
     *            HTTPセッション
     * @param request
     *            HTTPリクエスト
     * @param response
     *            HTTPレスポンス
     * @return 論理名
     * @throws Exception
     *             処理例外
     */
    abstract public String doProcessBefore( ServletContext sc,
            HttpSession session, HttpServletRequest req, HttpServletResponse res )
            throws Exception;


    /**
     * 機能別Action実行処理
     * 
     * @param mapping
     * @param form
     * @param application
     *            HTTPコンテキスト
     * @param session
     *            HTTPセッション
     * @param request
     *            HTTPリクエスト
     * @param response
     *            HTTPレスポンス
     * @return 論理名
     * @throws Exception
     *             処理例外
     */
    abstract public String doProcess( ActionMapping mapping, ActionForm form,
            ServletContext sc, HttpSession session, HttpServletRequest req,
            HttpServletResponse res ) throws Exception;


    /**
     * 機能別Action実行後処理 遷移先を変更はしない
     * 
     * @param mapping
     * @param form
     * @param application
     *            HTTPコンテキスト
     * @param session
     *            HTTPセッション
     * @param request
     *            HTTPリクエスト
     * @param response
     *            HTTPレスポンス
     * @throws Exception
     *             処理例外
     */
    abstract public void doProcessAfter( ActionMapping mapping,
            ActionForm form, ServletContext sc, HttpSession session,
            HttpServletRequest req, HttpServletResponse res ) throws Exception;


    /**
     * ユーザID取得<br>
     * <br>
     * HttpSessionに格納されているユーザIDの取得を行う。
     * 
     * @param session
     * @return
     */
    public String getLoginId( HttpSession session ) {

        String loginId = ( String ) session
                .getAttribute( Globals.SESSION_KEY_LOGIN_ID );

        if ( loginId == null ) {
            loginId = Globals.NO_LOGIN_ID;
        }

        return loginId;
    }


    /**
     * ユーザロール取得<br>
     * <br>
     * HttpSessionに格納されているユーザロールの取得を行う。
     * 
     * @param session
     * @return
     */
    @SuppressWarnings ( "unchecked" )
    public List< String > getLoginRole( HttpSession session ) {

        List< String > loginRole = ( List< String > ) session
                .getAttribute( Globals.SESSION_KEY_LOGIN_ROLE );

        if ( loginRole == null ) {
            loginRole = new ArrayList< String >();
        }

        return loginRole;
    }


    /**
     * Springの設定によってSession中のメッセージを初期化する
     * 
     * @param session
     */
    public void mesInit( HttpSession session ) {

        // mesInitFlagが0ならセッションの
        // エラーメッセージ、表示メッセージを初期化
        if ( this.mesInitFlag.equals( "0" ) ) {

            // HttpSessionに格納されたエラーメッセージを削除し、初期化
            session.removeAttribute( Globals.SESSION_KEY_ERROR_MES );

            // HttpSessionに格納された表示メッセージを削除し、初期化
            session.removeAttribute( Globals.SESSION_KEY_MES );
        }

    }

}
