/*
 * Copyright (C) 2006-2007 BLI Project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,
 * with or without modification, are permitted provided
 * that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain
 * the above copyright notice, this list of conditions
 * and the following disclaimer as the first lines of this file unmodified.
 * 
 * 2. Redistributions  in  binary  form must reproduce 
 * the above copyright notice, this list of conditions 
 * and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.sf.bli.framework.ss;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import jp.sf.bli.framework.common.BusinessLogicException;
import jp.sf.bli.framework.common.Globals;
import jp.sf.bli.framework.common.util.FrameworkUtil;
import jp.sf.bli.framework.logging.Logging;

import org.apache.struts.action.ActionForm;

/**
 * ビジネスロジックの実行をコントロールするクラスのabstractクラス
 * 
 * @author takeda
 */
abstract public class BusinessLogicCtrlSS {

    /** ログインID */
    protected String                    loginId        = "";

    /** BLValidate */
    protected BusinessLogicValidateSS   blValidate     = null;

    /** BlDo List */
    protected List< BusinessLogicDoSS > blList         = null;

    /** BL処理前データ格納Map */
    protected Map< String, Object >     inputValueMap  = null;

    /** BL処理データ連携用Map */
    protected Map< String, Object >     outputValueMap = null;


    /**
     * @return this.blList
     */
    public List< BusinessLogicDoSS > getBlList() {

        return this.blList;
    }


    /**
     * @return this.blValidate
     */
    public BusinessLogicValidateSS getBlValidate() {

        return this.blValidate;
    }


    /**
     * @return this.inputValueMap
     */
    public Map< String, Object > getInputValueMap() {

        return this.inputValueMap;
    }


    /**
     * @param inputValueMap
     *            this.inputValueMap = inputValueMap
     */
    public void setInputValueMap( Map< String, Object > inputValueMap ) {

        this.inputValueMap = inputValueMap;
    }


    /**
     * @return this.outputValueMap
     */
    public Map< String, Object > getOutputValueMap() {

        return this.outputValueMap;
    }


    /**
     * @param outputValueMap
     *            this.outputValueMap = outputValueMap
     */
    public void setOutputValueMap( Map< String, Object > outputValueMap ) {

        this.outputValueMap = outputValueMap;
    }


    /**
     * @return this.loginId
     */
    public String getLoginId() {

        return this.loginId;
    }


    /**
     * @param loginId
     *            this.loginId = loginId
     */
    public void setLoginId( String loginId ) {

        this.loginId = loginId;
    }


    /**
     * @param blValidate
     *            設定する BusinessLogicValidate
     */
    public void setBlValidate( BusinessLogicValidateSS blValidate ) {

        this.blValidate = blValidate;

    }


    /**
     * @param blList
     *            設定する BusinessLogicDo
     */
    public void setBlList( List< BusinessLogicDoSS > blList ) {

        this.blList = blList;

    }


    /**
     * ビジネスロジックの実行を制御する
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @return 遷移先を意味するint
     * @throws BusinessLogicException
     *             実行時に各種Exceptionが起きた場合に<br />
     *             BusinessLogicExceptionを発生させてthrowする
     */
    public int doBizCtrl( ServletContext sc, HttpSession session,
            HttpServletResponse res, HttpServletRequest req, ActionForm form )
            throws BusinessLogicException {

        int resultFlagInt = -1000;

        try {
            /*
             * BLCtrl Initialize
             * ------------------------------------------------
             */

            init( session );

            initProcess( sc, session, res, req, form );

            /*
             * FormBean To inputValueMap
             * ----------------------------------------
             */

            parameter2InputValueMap( sc, session, res, req, form );

            /*
             * BLValidate、BLDoの実行
             * -------------------------------------------------------------
             */

            blDoLoop( session, doVaildate( session ) );

            /*
             * View Disp
             * Set-----------------------------------------------------
             */

            viewDispSet( sc, session, res, req, form );

            /*
             * BLCtrl End Processing
             * --------------------------------------------
             */

            endProcess( sc, session, res, req, form );

            end( sc, session, res, req );

        } catch ( Exception e ) {

            Logging.fatal( this.getClass(), "BLI000006", getLoginId(),
                    "doBizCtrl:BLCtrl catch Exception. " + e.getMessage() );

            e.printStackTrace();
            throw new BusinessLogicException( "LogCount：BLI000006" );

        }

        resultFlagInt = ( ( Integer ) outputValueMap.get( Globals.RESULT_FLAG ) )
                .intValue();

        // ActionにresultFlagを返して処理を終える
        return resultFlagInt;
    }

    /**
     * BlCtrlの初期処理前のシステムの初期処理を行う
     * 
     * @param session
     * @throws Exception
     */
    public void init( HttpSession session ) throws Exception {

        /** ログイン名 */
        setLoginId( FrameworkUtil.getLoginId( session ) );

        this.inputValueMap = new HashMap< String, Object >();

        this.outputValueMap = new HashMap< String, Object >();

        Logging.debug( this.getClass(), "BLI000007", getLoginId(),
                "doBizCtrl:BLCtrl started." );
    }


    /**
     * BlCtrlの実行を行う前に初期処理を行う
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @throws Exception
     */
    abstract protected void initProcess( ServletContext sc,
            HttpSession session, HttpServletResponse res,
            HttpServletRequest req, ActionForm form ) throws Exception;


    /**
     * 画面からの入力をinputValueMapへ詰め替える
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @param ActionFormのname
     * @param inputValueMap
     * @param outputValueMap
     * @throws Exception
     */
    abstract protected void parameter2InputValueMap( ServletContext sc,
            HttpSession session, HttpServletResponse res,
            HttpServletRequest req, ActionForm form ) throws Exception;


    /**
     * BlValidateの実行を制御する
     * 
     * @param HttpSession
     * @param inputValueMap
     * @param outputValueMap
     * @return validateチェック合格:true/validateチェック不合格:false
     * @throws Exception
     */
    protected int doVaildate( HttpSession session ) throws Exception {

        int resultFlagInt = 0;

        Integer resultFlag = null;

        if ( this.blValidate != null ) {

            // BLValidate実行
            this.blValidate.doValidate( this.loginId, this.inputValueMap,
                    this.outputValueMap );

            // BLValidateの処理結果をoutputValueMapから取得
            resultFlag = ( Integer ) outputValueMap.get( Globals.RESULT_FLAG );

            // BLValidateでエラーになった場合 エラーの場合は0を返す
            if ( resultFlag == null || resultFlag.intValue() == 0 ) {

                // HttpSessionにViewに表示するエラーメッセージを格納する
                String errorMes = ( String ) this.outputValueMap
                        .get( Globals.MAP_KEY_ERROR_MES );

                FrameworkUtil.setSessionErrorMes( session, errorMes, 1 );

                if ( resultFlag == null ) {

                    // 想定外のエラーをセット
                    resultFlagInt = -999;
                    resultFlag = Integer.valueOf( -999 );
                    outputValueMap.put( Globals.RESULT_FLAG, Integer
                            .valueOf( resultFlagInt ) );

                }

                Logging.error( this.getClass(), "BLI000008", getLoginId(),
                        "doBizCtrl:Validate ended because of the error. rtnFlag:["
                                + resultFlag + "] errerMes:[" + errorMes + "]" );
            }

        } else {

            resultFlag = Integer.valueOf( -1000 );

        }

        resultFlagInt = resultFlag.intValue();

        return resultFlagInt;
    }

    /**
     * Springから渡されたビジネスロジックのListを順番に実行する
     * 
     * @param session
     * @param resultFlagInt
     * @throws Exception
     */
    protected void blDoLoop( HttpSession session, int resultFlagInt )
            throws Exception {

        Integer resultFlag;

        if ( resultFlagInt != -999 && resultFlagInt != 0 ) {

            Iterator< BusinessLogicDoSS > itr = this.blList.iterator();

            // blDoの数だけBLDoを実行する
            while ( itr.hasNext() ) {

                Logging.debug( this.getClass(), "BLI000009", getLoginId(),
                        "blDoLoop:BusineesLogicDo Strat" );

                BusinessLogicDoSS blDo = itr.next();

                if ( doBL( session, blDo ) == false ) {

                    Logging.debug( this.getClass(), "BLI000010", getLoginId(),
                            "blDoLoop:BusineesLogicDo No Normal End" );

                    resultFlag = ( Integer ) this.outputValueMap
                            .get( Globals.RESULT_FLAG );

                    resultFlagInt = resultFlag.intValue();

                    if ( resultFlagInt == -999 || resultFlagInt == 0 ) {

                        Logging.debug( this.getClass(), "BLI000011",
                                getLoginId(), "blDoLoop:break; resultFlag:"
                                        + resultFlagInt );

                        break;

                    }

                }

                Logging.debug( this.getClass(), "BLI000012", getLoginId(),
                        "blDoLoop:BusineesLogicDo Normal End" );
            }
        }
    }

    /**
     * BlDoの実行を制御する
     * 
     * @param HttpSession
     * @param BusinessLogicDo
     * @return BlDo正常終了:true/BlDo論理エラー終了:false
     * @throws Exception
     */
    protected boolean doBL( HttpSession session, BusinessLogicDoSS blDo )
            throws Exception {

        boolean rtnFlag = true; // 成功

        blDo.doBizLogic( this.loginId, this.inputValueMap, this.outputValueMap );

        // URLの引数でSampleDBを検索する

        // BLDoの処理結果をoutputValueMapから取得
        Integer resultFlag = ( Integer ) outputValueMap
                .get( Globals.RESULT_FLAG );

        // BLDoでエラーになった場合
        if ( resultFlag == null || resultFlag.intValue() == 0 ) {

            // BL処理結果をfailureのfalseにセットする
            rtnFlag = false;

            // HttpSessionにViewに表示するエラーメッセージを格納する
            String errorMes = ( String ) this.outputValueMap
                    .get( Globals.MAP_KEY_ERROR_MES );
            FrameworkUtil.setSessionErrorMes( session, errorMes, 0 );

            if ( resultFlag == null ) {

                // 想定外のエラーをセット
                resultFlag = Integer.valueOf( -999 );
                outputValueMap.put( Globals.RESULT_FLAG, resultFlag );

            }

            Logging.error( this.getClass(), "BLI000012", getLoginId(),
                    "doBizCtrl:Validate ended because of the error. rtnFlag:["
                            + resultFlag + "] errerMes:[" + errorMes + "]" );

        }

        return rtnFlag;

    }


    /**
     * 画面表示に使用するデータをServletContext, HttpSession, <br />
     * HttpServletRequestのいずれかにセットをする。<br />
     * BLValidate、BLDoの処理で論理エラー(outputValueMapのresultFlagが0)に<br />
     * なってもこの処理に到達する。
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @param ActionForm
     * @param ActionFormのname
     * @throws Exception
     */
    abstract protected void viewDispSet( ServletContext sc,
            HttpSession session, HttpServletResponse res,
            HttpServletRequest req, ActionForm form ) throws Exception;


    /**
     * BlCtrlの実行後に終了処理を行う
     * 
     * @param ServletContext
     * @param HttpSession
     * @param HttpServletResponse
     * @param HttpServletRequest
     * @throws Exception
     */
    abstract protected void endProcess( ServletContext sc, HttpSession session,
            HttpServletResponse res, HttpServletRequest req, ActionForm form )
            throws Exception;


    /**
     * BlCtrlの初期処理前のシステムの初期処理を行う
     * 
     * @param session
     * @throws Exception
     */
    protected void end( ServletContext sc, HttpSession session,
            HttpServletResponse res, HttpServletRequest req ) throws Exception {

        Logging.debug( this.getClass(), "BLI000013", loginId,
                "doBizCtrl:BLCtrl ended normally. rtnFlag:[0]" );

    }

}
