/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jp.bitmeister.asn1.type.ConstructiveType;

/**
 * Indicates that a field is an element of a structured type.
 * 
 * <p>
 * If an {@code @ASN1Element} annotation is present on a field, the field is
 * treated as a component of a SEQUENCE type or a SET type. A field annotated as
 * {@code @ASN1Element} must be declared as {@code public} non-{@code final}
 * non-{@code static}, and the type shall be a sub-class of {@code ASN1Type}.
 * The {@code value} parameter represents an order of appearance of a field in
 * an ASN.1 type definition. The {@code value} parameter must be set even if a
 * container type is a {@code SET} type. It will be used by the automatic
 * tagging process.
 * </p>
 * <p>
 * For example, following ASN.1 definition :
 * 
 * <pre>
 * Sample ::= SET {
 * 	number	INTEGER,
 * 	bool	BOOLEAN OPTIONAL }
 * </pre>
 * 
 * can be translated as :
 * 
 * <code><pre>
 * public class Sample extends SET {
 * 
 * 	&#064;ASN1ElementType(0)
 * 	public INTEGER number;
 * 
 * 	&#064;ASN1ElementType(value = 1, optional = true)
 * 	public BOOLEAN bool;
 * 
 * }
 * </pre></code>
 * 
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ConstructiveType
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ASN1Element {

	/**
	 * Indicates defined order of this element.
	 * 
	 * @return Order of this element.
	 */
	public int value();

	/**
	 * Indicates that the element is optional or not.
	 * 
	 * @return {@code true} if the element is optional.
	 */
	public boolean optional() default false;

	/**
	 * Indicates that the element has default value or not.
	 * 
	 * @return {@code true} if the element has default value.
	 */
	public boolean hasDefault() default false;

}
