/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Indicates identifier used for ASN.1 definition.
 * 
 * <p>
 * If an {@code @ASN1Identifier} annotation is present on a type or a field,
 * this value is used as ASN.1 identifier instead of their class name that taken
 * by calling {@code Class.getSimpleName()} or field name that taken by calling
 * {@code Field.getName()}. If an ASN.1 identifier can't be used for Java
 * identifier because of syntax restriction, this annotation shall be used.
 * </p>
 * <p>
 * For example, following ASN.1 definition :
 * 
 * <pre>
 * Signed-Number ::= SEQUENCE {	-- '-' can't be used for identifier.
 * 	int INTEGER		-- 'int' is a reserved word.
 * }
 * </pre>
 * 
 * can be translated as :
 * 
 * <code><pre>
 * &#064;ASN1Identifier(&quot;Signed-Number&quot;)
 * public class Signed_Number extends SEQUENCE {
 * 
 * 	&#064;ASN1Identifier(&quot;int&quot;)
 * 	&#064;ASN1Element(0)
 * 	public INTEGER _int;
 * 
 * }
 * </pre></code>
 * 
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@Target({ ElementType.TYPE, ElementType.FIELD })
@Retention(RetentionPolicy.RUNTIME)
public @interface ASN1Identifier {

	/**
	 * Indicates the identifier used in ASN.1 definitions.
	 * 
	 * @return The ASN.1 identifier.
	 */
	public String value();

}
